package com.atlassian.maven.plugins.amps;

import com.atlassian.maven.plugins.amps.product.ProductHandler;
import com.atlassian.maven.plugins.amps.product.ProductHandlerFactory;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Parameter;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;

import static java.util.Collections.singletonList;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.lang3.StringUtils.isBlank;

public abstract class AbstractTestGroupsHandlerMojo extends AbstractProductHandlerMojo {
    /**
     * The test group to run. If provided, determines the products to run.
     */
    @Parameter(property = "testGroup")
    protected String testGroup;

    /**
     * The configured test groups.
     */
    @Parameter
    private List<TestGroup> testGroups = new ArrayList<>();

    protected final List<TestGroup> getTestGroups() {
        return testGroups;
    }

    @Nonnull
    protected final List<ProductExecution> getTestGroupProductExecutions(String testGroupId)
            throws MojoExecutionException {
        final List<ProductExecution> products = new ArrayList<>();
        int dupCounter = 0;
        final Set<String> uniqueProductIds = new HashSet<>();
        final Map<String, Product> productContexts = getProductContexts();
        for (final String instanceId : getTestGroupInstanceIds(testGroupId)) {
            final Product ctx = productContexts.get(instanceId);
            if (ctx == null) {
                throw new MojoExecutionException("The test group '" + testGroupId + "' refers to a product '" +
                        instanceId + "' that doesn't have an associated <product> configuration.");
            }
            final ProductHandler productHandler = createProductHandler(ctx.getId());

            // Give unique ids to duplicate product instances
            if (uniqueProductIds.contains(instanceId)) {
                ctx.setInstanceId(instanceId + "-" + dupCounter++);
            } else {
                uniqueProductIds.add(instanceId);
            }
            products.add(new ProductExecution(ctx, productHandler));
        }

        if (products.size() > 1) {
            validatePortConfiguration(products);
        }

        return products;
    }

    /**
     * Returns the products in the test group:
     * <ul>
     * <li>If a {@literal <testGroup>} is defined, all the products of this test group</li>
     * <li>If testGroupId is __no_test_group__, adds it</li>
     * <li>If testGroupId is a product instanceId, adds it</li>
     * </ul>
     */
    private List<String> getTestGroupInstanceIds(String testGroupId) throws MojoExecutionException {
        List<String> instanceIds = new ArrayList<>();
        if (NO_TEST_GROUP.equals(testGroupId)) {
            instanceIds.add(getProductId());
        }

        for (TestGroup group : testGroups) {
            if (StringUtils.equals(group.getId(), testGroupId)) {
                instanceIds.addAll(group.getInstanceIds());
            }
        }
        if (ProductHandlerFactory.getIds().contains(testGroupId) && !instanceIds.contains(testGroupId)) {
            instanceIds.add(testGroupId);
        }

        if (instanceIds.isEmpty()) {
            getLog().warn("Unknown test group ID: " + testGroupId + " Detected IDs: " + getTestGroupIds());
        }

        return instanceIds;
    }

    protected final Set<String> getTestGroupIds() {
        return testGroups.stream()
                .map(TestGroup::getId)
                .filter(Objects::nonNull)
                .collect(toSet());
    }

    @Nonnull
    protected final List<ProductExecution> getProductExecutions() throws MojoExecutionException {
        final List<ProductExecution> productExecutions;
        if (!isBlank(testGroup)) {
            productExecutions = getTestGroupProductExecutions(testGroup);
        } else if (!isBlank(instanceId)) {
            Product ctx = getProductContexts().get(instanceId);
            if (ctx == null) {
                throw new MojoExecutionException("No product with instance ID '" + instanceId + "'");
            }
            ProductHandler product = createProductHandler(ctx.getId());
            productExecutions = singletonList(new ProductExecution(ctx, product));
        } else {
            Product ctx = getProductContexts().get(getProductId());
            ProductHandler product = createProductHandler(ctx.getId());
            productExecutions = singletonList(new ProductExecution(ctx, product));
        }
        return productExecutions;
    }

    /**
     * Ensures that there are no port conflicts between products and raises an exception if there
     * are conflicts
     *
     * @param executions two or more product executions, for which the configured ports should be validated
     * @throws MojoExecutionException if any of the configured ports collide between products
     * @since 8.0
     */
    void validatePortConfiguration(List<ProductExecution> executions) throws MojoExecutionException {
        Map<Integer, ConfiguredPort> portsById = new HashMap<>();

        MutableInt collisions = new MutableInt();
        executions.stream()
                .map(ProductExecution::getProduct)
                .flatMap(AbstractTestGroupsHandlerMojo::streamConfiguredPorts)
                .filter(ConfiguredPort::isStatic) // Only verify statically-configured ports
                .forEach(configured -> {
                    ConfiguredPort conflict = portsById.get(configured.port);
                    if (conflict == null) {
                        portsById.put(configured.port, configured);
                    } else {
                        getLog().error(configured.instanceId + ": The configured " + configured.type +
                                " port, " + configured.port + ", is in use by the " + conflict.type +
                                " port for " + conflict.instanceId);
                        collisions.increment();
                    }
                });

        int collisionCount = collisions.intValue();
        if (collisionCount != 0) {
            throw new MojoExecutionException(collisionCount + " port conflict" +
                    ((collisionCount == 1) ? " was" : "s were") + " detected between the " +
                    executions.size() + " products in the '" + testGroup + "' test group");
        }
    }

    /**
     * Returns the {@link ConfiguredPort configured ports} for the given {@link Product product}.
     *
     * @param product the product for which to get the ports
     * @return the configured ports
     * @since 8.0
     */
    private static Stream<ConfiguredPort> streamConfiguredPorts(final Product product) {
        final String instanceId = product.getInstanceId();
        return Stream.of(
                new ConfiguredPort(instanceId, product.getWebPort(), product.getProtocol().toUpperCase()),
                new ConfiguredPort(instanceId, product.getAjpPort(), "AJP"),
                new ConfiguredPort(instanceId, product.getRmiPort(), "RMI"));
    }

    /**
     * Describes a configured port for a {@link Product}, detailing the instance ID as well as the port and its type.
     * <p>
     * This data class simplifies maintaining an instance+port+type association, which facilitates using descriptive
     * error messages when the configured ports for two instances collide.
     *
     * @since 8.0
     */
    private static class ConfiguredPort {
        private final String instanceId;
        private final int port;
        private final String type;

        ConfiguredPort(String instanceId, int port, String type) {
            this.instanceId = requireNonNull(instanceId, "instanceId");
            this.port = port;
            this.type = requireNonNull(type, "type");
        }

        boolean isStatic() {
            return port != 0;
        }
    }
}
