package com.atlassian.maven.plugins.amps.product;

import com.atlassian.maven.plugins.amps.DataSource;
import com.atlassian.maven.plugins.amps.MavenContext;
import com.atlassian.maven.plugins.amps.MavenGoals;
import com.atlassian.maven.plugins.amps.Product;
import com.atlassian.maven.plugins.amps.ProductArtifact;
import com.atlassian.maven.plugins.amps.XmlOverride;
import com.atlassian.maven.plugins.amps.product.common.ValidationException;
import com.atlassian.maven.plugins.amps.product.common.XMLDocumentHandler;
import com.atlassian.maven.plugins.amps.product.common.XMLDocumentProcessor;
import com.atlassian.maven.plugins.amps.product.jira.JiraDatabaseType;
import com.atlassian.maven.plugins.amps.product.jira.config.DatabaseTypeUpdaterTransformer;
import com.atlassian.maven.plugins.amps.product.jira.config.DbConfigValidator;
import com.atlassian.maven.plugins.amps.product.jira.config.SchemeUpdaterTransformer;
import com.atlassian.maven.plugins.amps.util.ConfigFileUtils.Replacement;
import com.atlassian.maven.plugins.amps.util.JvmArgsFix;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.versioning.ComparableVersion;
import org.apache.maven.plugin.MojoExecutionException;

import javax.annotation.Nonnull;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import static com.atlassian.maven.plugins.amps.product.jira.JiraDatabaseType.getDatabaseType;
import static com.atlassian.maven.plugins.amps.util.ConfigFileUtils.RegexReplacement;
import static com.atlassian.maven.plugins.amps.util.FileUtils.fixWindowsSlashes;
import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.isBlank;

public class JiraProductHandler extends AbstractWebappProductHandler {
    @VisibleForTesting
    static final String INSTALLED_PLUGINS_DIR = "installed-plugins";

    @VisibleForTesting
    static final String PLUGINS_DIR = "plugins";

    @VisibleForTesting
    static final String BUNDLED_PLUGINS_UNZIPPED = "WEB-INF/atlassian-bundled-plugins";

    @VisibleForTesting
    static final String BUNDLED_PLUGINS_FROM_4_1 = "WEB-INF/classes/atlassian-bundled-plugins.zip";

    @VisibleForTesting
    static final String BUNDLED_PLUGINS_UPTO_4_0 = "WEB-INF/classes/com/atlassian/jira/plugin/atlassian-bundled-plugins.zip";

    @VisibleForTesting
    static final String FILENAME_DBCONFIG = "dbconfig.xml";

    private static final String JIRADS_PROPERTIES_FILE = "JiraDS.properties";

    private static final String JIRA_HOME_PLACEHOLDER = "${jirahome}";

    private static void checkNotFile(final File sharedHomeDir) {
        if (sharedHomeDir.isFile()) {
            final String error =
                    String.format("The specified shared home '%s' is a file, not a directory", sharedHomeDir);
            throw new IllegalArgumentException(error);
        }
    }

    private static void createIfNotExists(final File sharedHome) {
        sharedHome.mkdirs();
        if (!sharedHome.isDirectory()) {
            final String error = String.format("The specified shared home '%s' cannot be created", sharedHome);
            throw new IllegalStateException(error);
        }
    }

    public JiraProductHandler(final MavenContext context, final MavenGoals goals, ArtifactFactory artifactFactory) {
        super(context, goals, new JiraPluginProvider(), artifactFactory);
    }

    public String getId() {
        return "jira";
    }

    @Override
    protected void customiseInstance(Product ctx, File homeDir, File explodedWarDir) {
        // Jira 7.12.x has new tomcat version which requires additional characters to be whitelisted
        if (new ComparableVersion(ctx.getVersion()).compareTo(new ComparableVersion("7.12.0")) >= 0) {
            ctx.setCargoXmlOverrides(serverXmlJiraOverride());
        }
    }

    @Override
    protected void fixJvmArgs(Product ctx) {
        ComparableVersion productVersion = new ComparableVersion(ctx.getVersion());
        // Jira 8 raises memory requirements, to account for increased memory usage by Lucene
        if (productVersion.compareTo(new ComparableVersion("8.0.0-ALPHA")) >= 0) {
            ctx.setJvmArgs(JvmArgsFix.empty()
                    .with("-Xmx", "2g")
                    .with("-Xms", "1g")
                    .apply(ctx.getJvmArgs()));
        }
        // In Jira 7.7+ we have a HealthCheck that requires min / max memory to be set to a certain minimums or it can block startup.
        else if (productVersion.compareTo(new ComparableVersion("7.7.0-ALPHA")) >= 0) {
            ctx.setJvmArgs(JvmArgsFix.empty()
                    .with("-Xmx", "768m")
                    .with("-Xms", "384m")
                    .apply(ctx.getJvmArgs()));
        } else {
            super.fixJvmArgs(ctx);
        }
    }

    @Override
    public ProductArtifact getArtifact() {
        return new ProductArtifact("com.atlassian.jira", "atlassian-jira-webapp", "RELEASE");
    }

    @Override
    public ProductArtifact getTestResourcesArtifact() {
        return new ProductArtifact("com.atlassian.jira.plugins", "jira-plugin-test-resources");
    }

    @Override
    public int getDefaultHttpPort() {
        return 2990;
    }

    @Override
    public int getDefaultHttpsPort() {
        return 8442;
    }

    // only neeeded for older versions of JIRA; 7.0 onwards will have JiraDS.properties
    protected static File getHsqlDatabaseFile(final File homeDirectory) {
        return new File(homeDirectory, "database");
    }

    @Override
    public String getDefaultContainerId() {
        return "tomcat7x";
    }

    @Nonnull
    @Override
    protected Collection<String> getExtraJarsToSkipWhenScanningForTldsAndWebFragments() {
        // Fixes AMPS-1429 by skipping these JARs
        return ImmutableList.of("jotm*.jar", "xapool*.jar");
    }

    @Override
    public Map<String, String> getSystemProperties(final Product ctx) {
        final ImmutableMap.Builder<String, String> properties = ImmutableMap.builder();
        properties.putAll(super.getSystemProperties(ctx));
        properties.put("jira.home", fixWindowsSlashes(getHomeDirectory(ctx).getPath()));
        properties.put("cargo.servlet.uriencoding", "UTF-8");
        if (ctx.isAwaitFullInitialization()) {
            properties.put("com.atlassian.jira.startup.LauncherContextListener.SYNCHRONOUS", "true");
        }
        return properties.build();
    }

    @Override
    protected DataSource getDefaultDataSource(final Product product) {
        return getDataSourceFromJiraDSFile(product).orElse(getHsqlDataSource(product));
    }

    private String getJiraHome(final Product product) {
        return fixWindowsSlashes(getHomeDirectory(product).getAbsolutePath());
    }

    private Optional<DataSource> getDataSourceFromJiraDSFile(final Product jira) {
        final File dsPropsFile = new File(getHomeDirectory(jira), JIRADS_PROPERTIES_FILE);
        if (dsPropsFile.isFile()) {
            final DataSource dataSource = new DataSource();
            try (final FileInputStream inputStream = new FileInputStream(dsPropsFile)) {
                final Properties dsProps = new Properties();
                dsProps.load(inputStream);
                dataSource.setJndi(dsProps.getProperty("jndi"));
                dataSource.setUrl(dsProps.getProperty("url").replace(JIRA_HOME_PLACEHOLDER, getJiraHome(jira)));
                dataSource.setDriver(dsProps.getProperty("driver-class"));
                dataSource.setUsername(dsProps.getProperty("username"));
                dataSource.setPassword(dsProps.getProperty("password"));
                return Optional.of(dataSource);
            } catch (IOException e) {
                log.warn("failed to read " + dsPropsFile.getAbsolutePath(), e);
            }
        }
        return Optional.empty();
    }

    private DataSource getHsqlDataSource(final Product jira) {
        final DataSource dataSource = new DataSource();
        dataSource.setJndi("jdbc/JiraDS");
        dataSource.setUrl(format("jdbc:hsqldb:%s/database", getJiraHome(jira)));
        dataSource.setDriver("org.hsqldb.jdbcDriver");
        dataSource.setUsername("sa");
        dataSource.setPassword("");
        return dataSource;
    }

    @Override
    public File getUserInstalledPluginsDirectory(final Product product, final File webappDir, final File homeDir) {
        final File pluginHomeDirectory = getPluginHomeDirectory(product.getSharedHome(), homeDir);
        return new File(new File(pluginHomeDirectory, PLUGINS_DIR), INSTALLED_PLUGINS_DIR);
    }

    private File getPluginHomeDirectory(final String sharedHomePath, final File homeDir) {
        if (isBlank(sharedHomePath)) {
            return homeDir;
        }

        // A shared home was specified
        final File sharedHomeDir = new File(sharedHomePath);
        checkNotFile(sharedHomeDir);
        createIfNotExists(sharedHomeDir);
        return sharedHomeDir;
    }

    @Override
    public List<ProductArtifact> getExtraContainerDependencies() {
        return Arrays.asList(
                new ProductArtifact("hsqldb", "hsqldb", "1.8.0.5"),
                new ProductArtifact("javax.transaction", "jta", "1.0.1B"),
                new ProductArtifact("ots-jts", "ots-jts", "1.0"),

                // for data source and transaction manager providers
                new ProductArtifact("jotm", "jotm", "1.4.3"),
                new ProductArtifact("jotm", "jotm-jrmp_stubs", "1.4.3"),
                new ProductArtifact("jotm", "jotm-iiop_stubs", "1.4.3"),
                new ProductArtifact("jotm", "jonas_timer", "1.4.3"),
                new ProductArtifact("jotm", "objectweb-datasource", "1.4.3"),
                new ProductArtifact("carol", "carol", "1.5.2"),
                new ProductArtifact("carol", "carol-properties", "1.0"),
                new ProductArtifact("xapool", "xapool", "1.3.1"),
                new ProductArtifact("commons-logging", "commons-logging", "1.1.1")
        );
    }

    @Override
    public File getBundledPluginPath(Product ctx, File appDir) {
        // the zip became a directory in 6.3, so if the directory exists and is a directory, use it,
        // otherwise fallback to the old behaviour.
        final File bundleDir = new File(appDir, BUNDLED_PLUGINS_UNZIPPED);

        if (bundleDir.exists() && bundleDir.isDirectory()) {
            return bundleDir;
        } else {
            // this location used from 4.1 onwards (inclusive), until replaced by unzipped dir.
            String bundledPluginPluginsPath = BUNDLED_PLUGINS_FROM_4_1;
            String[] version = ctx.getVersion().split("-", 2)[0].split("\\.");
            try {
                long major = Long.parseLong(version[0]);
                long minor = (version.length > 1) ? Long.parseLong(version[1]) : 0;

                if (major < 4 || major == 4 && minor == 0) {
                    bundledPluginPluginsPath = BUNDLED_PLUGINS_UPTO_4_0;
                }
            } catch (NumberFormatException e) {
                log.debug(String.format("Unable to parse JIRA version '%s', assuming JIRA 4.1 or newer.", ctx.getVersion()), e);
            }
            return new File(appDir, bundledPluginPluginsPath);
        }
    }

    @Override
    public void processHomeDirectory(final Product ctx, final File homeDir) throws MojoExecutionException {
        super.processHomeDirectory(ctx, homeDir);
        createDbConfigXmlIfNecessary(homeDir);
        if (ctx.getDataSources().size() == 1) {
            final DataSource ds = ctx.getDataSources().get(0);
            final JiraDatabaseType dbType = getDatabaseType(ds).orElseThrow(
                    () -> new MojoExecutionException("Could not find database type for " + ds));
            updateDbConfigXml(homeDir, dbType, ds.getSchema());
        } else if (ctx.getDataSources().size() > 1) {
            throw new MojoExecutionException("JIRA does not support multiple data sources");
        }
    }

    /**
     * Update JIRA dbconfig.xml in case user provide their own database connection configuration in pom
     * Jira database type was detected by uri/url prefix and database driver
     * Jira database type defines database-type and schema or schema-less for specific Jira database
     * Please refer documentation url: http://www.atlassian.com/software/jira/docs/latest/databases/index.html
     * example:
     * <pre>
     * {@code
     * <dataSource>
     *   <jndi>${dataSource.jndi}</jndi>
     *   <url>${dataSource.url}</url>
     *   <driver>${dataSource.driver}</driver>
     *   <username>${dataSource.user}</username>
     *   <password>${dataSource.password}</password>
     *   <schema>${dataSource.schema}</schema>
     * </dataSource>
     * }
     * </pre>
     *
     * @param homeDir the application's home directory
     * @param dbType  the database type in use
     * @param schema  the schema to use
     * @throws MojoExecutionException if {@code dbconfig.xml} can't be updated
     */
    @VisibleForTesting
    public void updateDbConfigXml(final File homeDir, final JiraDatabaseType dbType, final String schema)
            throws MojoExecutionException {
        final File dbConfigXml = new File(homeDir, FILENAME_DBCONFIG);
        if (!dbConfigXml.exists() || dbType == null) {
            return;
        }

        try {
            new XMLDocumentProcessor(new XMLDocumentHandler(dbConfigXml))
                    .load()
                    .validate(new DbConfigValidator())
                    .transform(new DatabaseTypeUpdaterTransformer(dbType))
                    .transform(new SchemeUpdaterTransformer(dbType, schema))
                    .saveIfModified();
        } catch (ValidationException e) {
            throw new MojoExecutionException("Validation of dbconfig.xml file failed", e);
        }
    }

    @Override
    public List<Replacement> getReplacements(final Product ctx) {
        String contextPath = ctx.getContextPath();
        if (!contextPath.startsWith("/")) {
            contextPath = "/" + contextPath;
        }

        final String baseUrl = ctx.getProtocol() + "://" + ctx.getServer() + ":" + ctx.getWebPort() + contextPath;

        List<Replacement> replacements = super.getReplacements(ctx);
        File homeDir = getHomeDirectory(ctx);
        // We don't rewrap snapshots with these values:
        replacements.add(0, new Replacement("http://localhost:8080", baseUrl, false));
        replacements.add(new Replacement("@project-dir@", homeDir.getParent(), false));
        replacements.add(new Replacement("/jira-home/", "/home/", false));
        replacements.add(new Replacement("@base-url@", baseUrl, false));
        replacements.add(new RegexReplacement("'[A-B]{1}[A-Z0-9]{3}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}'", "''")); // blank out the server ID
        return replacements;
    }

    @Override
    public List<File> getConfigFiles(Product product, File homeDir) {
        List<File> configFiles = super.getConfigFiles(product, homeDir);
        configFiles.add(new File(homeDir, "database.log"));
        configFiles.add(new File(homeDir, "database.script"));
        configFiles.add(new File(homeDir, FILENAME_DBCONFIG));
        return configFiles;
    }

    // only neeeded for older versions of JIRA; 7.0 onwards will have JiraDS.properties
    static void createDbConfigXmlIfNecessary(File homeDir) throws MojoExecutionException {
        File dbConfigXml = new File(homeDir, FILENAME_DBCONFIG);
        if (dbConfigXml.exists()) {
            return;
        }

        InputStream templateIn = JiraProductHandler.class.getResourceAsStream("jira-dbconfig-template.xml");
        if (templateIn == null) {
            throw new MojoExecutionException("Missing internal resource: jira-dbconfig-template.xml");
        }

        try {
            String template = IOUtils.toString(templateIn, UTF_8);
            File dbFile = getHsqlDatabaseFile(homeDir);
            String jdbcUrl = "jdbc:hsqldb:file:" + dbFile.toURI().getPath();
            String result = template.replace("@jdbc-url@", jdbcUrl);
            FileUtils.writeStringToFile(dbConfigXml, result, UTF_8);
        } catch (IOException ioe) {
            throw new MojoExecutionException("Unable to create config file: " + FILENAME_DBCONFIG, ioe);
        }
    }

    @Override
    public List<ProductArtifact> getDefaultLibPlugins() {
        return Collections.emptyList();
    }

    @Override
    public List<ProductArtifact> getDefaultBundledPlugins() {
        return Collections.emptyList();
    }

    private static class JiraPluginProvider extends AbstractPluginProvider {

        @Override
        protected Collection<ProductArtifact> getSalArtifacts(String salVersion) {
            return Arrays.asList(
                    new ProductArtifact("com.atlassian.sal", "sal-api", salVersion),
                    new ProductArtifact("com.atlassian.sal", "sal-jira-plugin", salVersion));
        }

        @Override
        protected Collection<ProductArtifact> getPdkInstallArtifacts(String pdkInstallVersion) {
            List<ProductArtifact> plugins = new ArrayList<>(super.getPdkInstallArtifacts(pdkInstallVersion));
            plugins.add(new ProductArtifact("commons-fileupload", "commons-fileupload", "1.2.1"));
            return plugins;
        }
    }

    @Override
    public void cleanupProductHomeForZip(Product product, File snapshotDir) throws MojoExecutionException, IOException {
        super.cleanupProductHomeForZip(product, snapshotDir);

        FileUtils.deleteQuietly(new File(snapshotDir, "log/atlassian-jira.log"));
    }

    private Collection<XmlOverride> serverXmlJiraOverride() {
        return Collections.unmodifiableList(Arrays.asList(
                new XmlOverride("conf/server.xml", "//Connector", "relaxedPathChars", "[]|"),
                new XmlOverride("conf/server.xml", "//Connector", "relaxedQueryChars", "[]|{}^\\`\"<>")
        ));
    }
}
