package com.atlassian.maven.plugins.amps.wadl;

import com.atlassian.maven.plugins.amps.MavenContext;
import com.atlassian.maven.plugins.amps.util.PluginXmlUtils;
import com.atlassian.maven.plugins.amps.util.PluginXmlUtils.RESTModuleInfo;
import org.apache.maven.artifact.DependencyResolutionRequiredException;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.twdata.maven.mojoexecutor.MojoExecutor.Element;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.atlassian.maven.plugins.amps.MavenGoals.getArtifactVersion;
import static com.atlassian.maven.plugins.amps.util.FileUtils.fixWindowsSlashes;
import static com.atlassian.maven.plugins.amps.util.MojoUtils.executeWithMergedConfig;
import static java.lang.String.format;
import static java.lang.Thread.currentThread;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.emptyMap;
import static java.util.Objects.requireNonNull;
import static java.util.regex.Matcher.quoteReplacement;
import static java.util.stream.Collectors.joining;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.commons.io.IOUtils.resourceToString;
import static org.apache.commons.lang3.StringUtils.replace;
import static org.twdata.maven.mojoexecutor.MojoExecutor.artifactId;
import static org.twdata.maven.mojoexecutor.MojoExecutor.configuration;
import static org.twdata.maven.mojoexecutor.MojoExecutor.element;
import static org.twdata.maven.mojoexecutor.MojoExecutor.goal;
import static org.twdata.maven.mojoexecutor.MojoExecutor.groupId;
import static org.twdata.maven.mojoexecutor.MojoExecutor.name;
import static org.twdata.maven.mojoexecutor.MojoExecutor.plugin;
import static org.twdata.maven.mojoexecutor.MojoExecutor.version;

/**
 * Generates WADL documentation for P2 plugin REST modules.
 *
 * @since 8.1.2 was previously in the {@code MavenGoals} class
 */
@ParametersAreNonnullByDefault
public class RestDocsGenerator {

    private final MavenContext mavenContext;

    /**
     * Constructor.
     *
     * @param mavenContext the Maven context
     */
    public RestDocsGenerator(final MavenContext mavenContext) {
        this.mavenContext = requireNonNull(mavenContext);
    }

    private Log log() {
        return mavenContext.getLog();
    }

    private MavenProject mavenProject() {
        return mavenContext.getProject();
    }

    /**
     * Generates <a href="https://en.wikipedia.org/wiki/Web_Application_Description_Language">WADL</a> documentation for
     * the REST modules in the current plugin project.
     *
     * Works by invoking the Maven JavaDoc plugin, using the custom doclet
     * {@code com.sun.jersey.wadl.resourcedoc.ResourceDocletJSON}, the source for which can be found in
     * <a href="https://bitbucket.org/atlassian/atlassian-jersey-restdoc/>atlassian/atlassian-jersey-restdoc</a>
     *
     * @param jacksonModules any custom Jackson serializer modules to be used; a colon-separated list of their fully
     *                      qualified class names, each implementing {@code org.codehaus.jackson.map.Module}; see REST-267
     * @throws MojoExecutionException if the operation fails
     */
    public void generateRestDocs(@Nullable final String jacksonModules) throws MojoExecutionException {
        final List<RESTModuleInfo> restModules = PluginXmlUtils.getRestModules(mavenContext);
        if (restModules.isEmpty()) {
            log().info("No REST modules found, skipping WADL doc generation");
            return;
        }

        withoutGlobalJavadocPlugin(() -> invokeJavadocPluginWithCustomDoclet(restModules, jacksonModules));

        generateApplicationXmlFiles();
    }

    private void invokeJavadocPluginWithCustomDoclet(final List<RESTModuleInfo> restModules, @Nullable final String jacksonModules)
            throws MojoExecutionException {
        executeWithMergedConfig(
                plugin(
                        groupId("org.apache.maven.plugins"),
                        artifactId("maven-javadoc-plugin"),
                        version(getArtifactVersion("maven-javadoc-plugin", mavenContext))
                ),
                goal("javadoc"),
                configuration(
                        element(name("maxmemory"), "1024m"),
                        element(name("sourcepath"), getPackagesPath(restModules)),
                        element(name("doclet"), "com.sun.jersey.wadl.resourcedoc.ResourceDocletJSON"),
                        element(name("docletPath"), getDocletPath()),
                        element(name("docletArtifacts"),
                                element(name("docletArtifact"),
                                        element(name("groupId"), "com.atlassian.plugins.rest"),
                                        element(name("artifactId"), "atlassian-rest-doclet"),
                                        element(name("version"), "2.9.2")
                                ),
                                element(name("docletArtifact"),
                                        element(name("groupId"), "xerces"),
                                        element(name("artifactId"), "xercesImpl"),
                                        element(name("version"), "2.9.1")
                                ),
                                element(name("docletArtifact"),
                                        element(name("groupId"), "commons-lang"),
                                        element(name("artifactId"), "commons-lang"),
                                        element(name("version"), "2.6")
                                )
                        ),
                        element(name("outputDirectory")),
                        element(name("additionalOptions"), getAdditionalOptions(jacksonModules)),
                        element(name("useStandardDocletOptions"), "false")
                ),
                mavenContext.getExecutionEnvironment()
        );
    }

    private Element[] getAdditionalOptions(@Nullable final String jacksonModules) {
        final String resourcedocPath = fixWindowsSlashes(
                mavenProject().getBuild().getOutputDirectory() + File.separator + "resourcedoc.xml");
        final List<Element> additionalOptions = new ArrayList<>();
        additionalOptions.add(element(name("additionalOption"), "-output \"" + resourcedocPath + "\""));
        if (jacksonModules != null) {
            additionalOptions.add(element(name("additionalOption"), " -modules \"" + jacksonModules + "\""));
        }
        return additionalOptions.toArray(new Element[0]);
    }

    private String getDocletPath() throws MojoExecutionException {
        final StringBuilder docletPath =
                new StringBuilder(File.pathSeparator + mavenProject().getBuild().getOutputDirectory());
        final Set<String> docletPaths = new HashSet<>();

        try {
            docletPaths.addAll(mavenProject().getCompileClasspathElements());
            docletPaths.addAll(mavenProject().getRuntimeClasspathElements());
            //noinspection deprecation -- no suggested replacement
            docletPaths.addAll(mavenProject().getSystemClasspathElements());

            // AMPS-663: add plugin execution classes to doclet path
            final URL[] pluginUrls = ((URLClassLoader) currentThread().getContextClassLoader()).getURLs();
            for (URL pluginUrl : pluginUrls) {
                docletPaths.add(new File(pluginUrl.getFile()).getPath());
            }

            for (String path : docletPaths) {
                docletPath.append(File.pathSeparator);
                docletPath.append(path);
            }
            return docletPath.toString();
        } catch (final DependencyResolutionRequiredException e) {
            throw new MojoExecutionException("Dependencies must be resolved", e);
        }
    }

    private String getPackagesPath(final Collection<RESTModuleInfo> restModules) {
        if (anyRestModuleHasNoExplicitPackages(restModules)) {
            final String sourceDirectory = mavenProject().getBuild().getSourceDirectory();
            log().info(format("Scanning all of %s for REST resources", sourceDirectory));  // fixes AMPS-1152
            return sourceDirectory;
        }
        return restModules.stream()
                .flatMap(module -> module.getPackagesToScan().stream())
                .map(this::getAbsoluteDirectory)
                .distinct()
                .collect(joining(File.pathSeparator));
    }

    private boolean anyRestModuleHasNoExplicitPackages(final Collection<RESTModuleInfo> restModules) {
        return restModules.stream()
                .map(RESTModuleInfo::getPackagesToScan)
                .anyMatch(List::isEmpty);
    }

    private String getAbsoluteDirectory(final String packageName) {
        final String relativePackageDirectory = packageName.replaceAll("\\.", quoteReplacement(File.separator));
        return mavenProject().getBuild().getSourceDirectory() + File.separator + relativePackageDirectory;
    }

    private void generateApplicationXmlFiles() throws MojoExecutionException {
        try {
            // application-doc.xml
            final PluginXmlUtils.PluginInfo pluginInfo = PluginXmlUtils.getPluginInfo(mavenContext);
            final Map<String, String> applicationXmlReplacements = new HashMap<>();
            applicationXmlReplacements.put("${rest.doc.title}", pluginInfo.getName());
            applicationXmlReplacements.put("${rest.doc.description}", pluginInfo.getDescription());
            generateApplicationXmlFile("application-doc.xml", applicationXmlReplacements);

            // application-grammars.xml
            generateApplicationXmlFile("application-grammars.xml", emptyMap());
        } catch (Exception e) {
            throw new MojoExecutionException("Error writing REST application XML files", e);
        }
    }

    private void generateApplicationXmlFile(final String filename, final Map<String, String> replacements)
            throws IOException {
        final File outputFile = new File(mavenProject().getBuild().getOutputDirectory(), filename);
        if (!outputFile.exists()) {
            String outputText = resourceToString(filename, UTF_8, getClass().getClassLoader());
            for (final Map.Entry<String, String> replacement : replacements.entrySet()) {
                outputText = replace(outputText, replacement.getKey(), replacement.getValue());
            }
            writeStringToFile(outputFile, outputText, UTF_8);
            log().info("Wrote " + outputFile.getAbsolutePath());
        }
    }

    private interface CustomJavadocAction {

        void invoke() throws MojoExecutionException;
    }

    /**
     * AMPSDEV-127: 'generate-rest-docs' fails with JDK8 - invalid flag: -Xdoclint:all
     * Root cause: ResourceDocletJSON doclet does not support option doclint
     * Solution: Temporarily remove global javadoc configuration (remove doclint)
     */
    private void withoutGlobalJavadocPlugin(final CustomJavadocAction javadocAction) throws MojoExecutionException {
        final Plugin globalJavadoc = mavenProject().getPlugin("org.apache.maven.plugins:maven-javadoc-plugin");
        if (globalJavadoc != null) {
            mavenProject().getBuild().removePlugin(globalJavadoc);
        }
        javadocAction.invoke();
        if (globalJavadoc != null) {
            mavenProject().getBuild().addPlugin(globalJavadoc);
        }
    }
}
