package com.atlassian.maven.plugins.amps.util;

import com.atlassian.maven.plugins.amps.ProductArtifact;
import com.google.common.annotations.VisibleForTesting;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.repository.metadata.ArtifactRepositoryMetadata;
import org.apache.maven.artifact.repository.metadata.Metadata;
import org.apache.maven.artifact.repository.metadata.RepositoryMetadata;
import org.apache.maven.artifact.repository.metadata.RepositoryMetadataManager;
import org.apache.maven.artifact.repository.metadata.RepositoryMetadataResolutionException;
import org.apache.maven.artifact.repository.metadata.SnapshotArtifactRepositoryMetadata;
import org.apache.maven.artifact.resolver.ArtifactNotFoundException;
import org.apache.maven.artifact.resolver.ArtifactResolutionException;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.artifact.versioning.ArtifactVersion;
import org.apache.maven.artifact.versioning.DefaultArtifactVersion;
import org.apache.maven.plugin.MojoExecutionException;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;

import static com.atlassian.maven.plugins.amps.product.FeCruProductHandler.isFecruRelease;
import static org.apache.commons.lang3.StringUtils.isBlank;

public class ArtifactRetriever
{
    private final ArtifactResolver artifactResolver;
    private final ArtifactFactory artifactFactory;
    private final ArtifactRepository localRepository;
    private final List<ArtifactRepository> remoteRepositories;
    private final RepositoryMetadataManager repositoryMetadataManager;

    public ArtifactRetriever(ArtifactResolver artifactResolver, ArtifactFactory artifactFactory, ArtifactRepository localRepository, List<ArtifactRepository> remoteRepositories, RepositoryMetadataManager repositoryMetadataManager)
    {
        this.artifactResolver = artifactResolver;
        this.artifactFactory = artifactFactory;
        this.localRepository = localRepository;
        this.remoteRepositories = remoteRepositories;
        this.repositoryMetadataManager = repositoryMetadataManager;
    }

    public String resolve(ProductArtifact dependency) throws MojoExecutionException
    {
        Artifact artifact = this.artifactFactory.createArtifact(dependency.getGroupId(),
        dependency.getArtifactId(), dependency.getVersion(), "compile", "jar");
        try
        {
            this.artifactResolver.resolve(artifact, remoteRepositories, localRepository);
        }
        catch (ArtifactResolutionException e)
        {
            throw new MojoExecutionException("Cannot resolve artifact", e);
        }
        catch (ArtifactNotFoundException e)
        {
            throw new MojoExecutionException("Cannot find artifact", e);
        }
        return artifact.getFile().getPath();
    }
    
    public String getLatestStableVersion(final Artifact artifact) throws MojoExecutionException
    {
        RepositoryMetadata metadata;
        
        if(!artifact.isSnapshot() || Artifact.LATEST_VERSION.equals(artifact.getBaseVersion()) || Artifact.RELEASE_VERSION.equals(artifact.getBaseVersion()))
        {
            metadata = new ArtifactRepositoryMetadata(artifact);
        }
        else
        {
            metadata = new SnapshotArtifactRepositoryMetadata(artifact);
        }

        try
        {
            repositoryMetadataManager.resolve( metadata, remoteRepositories, localRepository );
            artifact.addMetadata( metadata );

            Metadata repoMetadata = metadata.getMetadata();
            String version = null;

            if (repoMetadata != null && repoMetadata.getVersioning() != null)
            {
                version = getLatestOfficialRelease(repoMetadata.getVersioning().getVersions(), artifact);
            }

            if (version == null)
            {
                // use the local copy, or if it doesn't exist - go to the remote repo for it
                version = artifact.getBaseVersion();
            }
            
            return version;
        }
        catch (RepositoryMetadataResolutionException e)
        {
            throw new MojoExecutionException("Error resolving stable version", e);
        }
    }

    /**
     * Returns the latest of the given versions, excluding unofficial releases.
     *
     * @param versions the versions from which to get the latest
     * @param artifact the artifact to which these versions relate
     * @return null if no official release versions were given
     */
    @Nullable
    @VisibleForTesting
    static String getLatestOfficialRelease(final Collection<String> versions, final Artifact artifact)
    {
        return versions.stream()
                .map(DefaultArtifactVersion::new)
                .filter(version -> isOfficialRelease(version, artifact))
                .max(ArtifactVersion::compareTo)
                .map(ArtifactVersion::toString)
                .orElse(null);
    }

    private static boolean isOfficialRelease(final ArtifactVersion version, final Artifact artifact) {
        // Most products' final releases are like 1.2.3 (no qualifier), but Fecru has a timestamp suffix
        return isBlank(version.getQualifier()) || isFecruRelease(version, artifact);
    }
}
