package com.atlassian.mail.server.impl;

import java.io.UnsupportedEncodingException;
import java.time.Duration;
import java.time.format.DateTimeParseException;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import javax.annotation.Nonnull;
import javax.mail.Address;
import javax.mail.Authenticator;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.NoSuchProviderException;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.URLName;
import javax.mail.internet.MimeMessage;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;

import com.atlassian.mail.Email;
import com.atlassian.mail.MailConstants;
import com.atlassian.mail.MailException;
import com.atlassian.mail.MailProtocol;
import com.atlassian.mail.server.AbstractMailServer;
import com.atlassian.mail.server.MailServerManager;
import com.atlassian.mail.server.SMTPMailServer;
import com.atlassian.mail.server.auth.AuthenticationContext;
import com.atlassian.mail.server.impl.util.MessageCreator;

import static java.time.temporal.ChronoUnit.MILLIS;
import static java.time.temporal.ChronoUnit.MINUTES;
import static java.util.Optional.ofNullable;

import static com.atlassian.mail.MailConstants.DEFAULT_SMTP_PROTOCOL;
import static com.atlassian.mail.MailConstants.DEFAULT_TIMEOUT;

public class SMTPMailServerImpl extends AbstractMailServer implements SMTPMailServer {

    private static final long serialVersionUID = -1443984302060710065L;

    // MAIL-113
    private static final String JNDI_JAVA_SCHEME = "java:";
    // four minutes of client side timeout are plenty, it has to be shorter than a potential server timeout
    public static final Duration DEFAULT_TRANSPORT_CACHE_TTL = Duration.of(4L, MINUTES);
    public static final String TRANSPORT_CACHE_TTL_PROPERTY = "mail.smtp.transport.cache.ttl";

    private boolean isSessionServer;
    private String defaultFrom;
    private String defaultReplyTo;
    private String prefix;
    private String jndiLocation;
    private boolean removePrecedence;
    private boolean tlsHostnameCheckRequired;

    private transient Session session;
    private transient LoadingCache<Session, Transport> transportCache;

    public SMTPMailServerImpl(Builder<?> builder) {
        super(builder);
        setSessionServer(builder.sessionServer);
        setDefaultFrom(builder.defaultFrom);
        setDefaultReplyTo(builder.defaultReplyTo);
        setPrefix(builder.prefix);
        setRemovePrecedence(builder.removePrecedence);
        setTlsHostnameCheckRequired(builder.tlsHostnameCheckRequired);
        if (builder.sessionServer) {
            setJndiLocation(builder.location);
            setHostname(null);
        } else {
            setHostname(builder.location);
        }
        initTransportCache();
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl() {
        super(DEFAULT_TIMEOUT);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            String location,
            String username,
            String password) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                DEFAULT_SMTP_PROTOCOL,
                location,
                DEFAULT_SMTP_PORT,
                false,
                username,
                password);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            String username,
            String password) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                false,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                username,
                password,
                DEFAULT_TIMEOUT);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            String username,
            String password,
            long timeout) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                false,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                username,
                password,
                timeout);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            boolean removePrecedence,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            String username,
            String password,
            long timeout) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                removePrecedence,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                username,
                password,
                timeout,
                null,
                null);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            String username,
            String password,
            long timeout,
            String socksHost,
            String socksPort) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                false,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                username,
                password,
                timeout,
                socksHost,
                socksPort);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            boolean removePrecedence,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            String username,
            String password,
            long timeout,
            String socksHost,
            String socksPort) {
        super(id, name, description, protocol, location, smtpPort, username, password, timeout, socksHost, socksPort);
        setDefaultFrom(from);
        setPrefix(prefix);
        setSessionServer(isSession);
        setRemovePrecedence(removePrecedence);
        setTlsRequired(tlsRequired);

        if (isSession) {
            setJndiLocation(location);
            setHostname(null);
        }
        initTransportCache();
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            String location,
            AuthenticationContext authenticationContext) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                DEFAULT_SMTP_PROTOCOL,
                location,
                DEFAULT_SMTP_PORT,
                false,
                authenticationContext);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            AuthenticationContext authenticationContext) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                false,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                authenticationContext,
                DEFAULT_TIMEOUT);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            AuthenticationContext authenticationContext,
            long timeout) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                false,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                authenticationContext,
                timeout);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            boolean removePrecedence,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            AuthenticationContext authenticationContext,
            long timeout) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                removePrecedence,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                authenticationContext,
                timeout,
                null,
                null);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            AuthenticationContext authenticationContext,
            long timeout,
            String socksHost,
            String socksPort) {
        this(
                id,
                name,
                description,
                from,
                prefix,
                isSession,
                false,
                protocol,
                location,
                smtpPort,
                tlsRequired,
                authenticationContext,
                timeout,
                socksHost,
                socksPort);
    }

    /**
     * @deprecated since 7.1.0. Use {@link SMTPMailServerImpl.Builder} instead.
     */
    @Deprecated
    public SMTPMailServerImpl(
            Long id,
            String name,
            String description,
            String from,
            String prefix,
            boolean isSession,
            boolean removePrecedence,
            MailProtocol protocol,
            String location,
            String smtpPort,
            boolean tlsRequired,
            AuthenticationContext authenticationContext,
            long timeout,
            String socksHost,
            String socksPort) {
        super(
                id,
                name,
                description,
                protocol,
                location,
                smtpPort,
                authenticationContext,
                timeout,
                socksHost,
                socksPort);
        setDefaultFrom(from);
        setPrefix(prefix);
        setSessionServer(isSession);
        setRemovePrecedence(removePrecedence);
        setTlsRequired(tlsRequired);

        if (isSession) {
            setJndiLocation(location);
            setHostname(null);
        }
        initTransportCache();
    }

    @Override
    public String getJndiLocation() {
        return jndiLocation;
    }

    @Override
    public void setJndiLocation(String jndiLocation) {
        if (StringUtils.isNotBlank(jndiLocation) && !StringUtils.startsWithIgnoreCase(jndiLocation, JNDI_JAVA_SCHEME)) {
            throw new IllegalArgumentException("Only the java URL scheme(java:) is allowed for the jndiLocation");
        }
        this.jndiLocation = jndiLocation;
        propertyChanged();
    }

    /**
     * get the mail session
     */
    @Override
    public Session getSession() throws NamingException, MailException {
        if (session == null) {
            if (isSessionServer()) {
                log.debug("Getting session from JNDI");
                Object jndiSession = getJndiSession();
                if (jndiSession instanceof javax.mail.Session) {
                    session = (Session) jndiSession;
                } else {
                    log.error("Mail server at location [" + getJndiLocation()
                            + "] is not of required type javax.mail.Session, or is in different classloader. "
                            + "It is of type '"
                            + (jndiSession != null ? jndiSession.getClass().getName() : null) + "' in classloader '"
                            + jndiSession.getClass().getClassLoader() + "' instead");
                    throw new IllegalArgumentException("Mail server at location [" + getJndiLocation()
                            + "] is not of required type javax.mail.Session. ");
                }
            } else {
                final Properties props = loadSystemProperties(getProperties());

                if (isTlsRequired() && isTlsHostnameCheckRequired()) {
                    props.put(
                            getMailProtocol() == MailProtocol.SECURE_SMTP
                                    ? "mail.smtps.ssl.checkserveridentity"
                                    : "mail.smtp.ssl.checkserveridentity",
                            "true");
                }

                final Authenticator auth = isAuthenticating ? getAuthenticator() : null;
                session = getSessionFromServerManager(props, auth);
                if (auth != null) {
                    session.setPasswordAuthentication(
                            new URLName(
                                    getMailProtocol().getProtocol(),
                                    getHostname(),
                                    Integer.parseInt(getPort()),
                                    null,
                                    null,
                                    null),
                            new PasswordAuthentication(getUsername(), getPassword()));
                }
            }
        }
        return session;
    }

    protected Object getJndiSession() throws NamingException {
        Context ctx = new InitialContext();
        return ctx.lookup(getJndiLocation());
    }

    @Override
    public void send(Email email) throws MailException {
        sendWithMessageId(email, null);
    }

    @Override
    public void sendWithMessageId(final Email email, final String messageId) throws MailException {
        try {
            Session thisSession = getSession();

            MimeMessage message = new ExtendedMimeMessage(thisSession, messageId);
            MessageCreator messageCreator = new MessageCreator();

            messageCreator.updateMimeMessage(email, getDefaultFrom(), getDefaultReplyTo(), prefix, message);

            log.debug("Getting transport for protocol [" + getMailProtocol().getProtocol() + "]");
            // Send the message using the transport configured for this mail server - JRA-24549/MAIL-61
            final Transport transport = getTransport(thisSession);
            try {
                if (log.isDebugEnabled()) {
                    log.debug("Got transport: [" + transport + "]. Connecting");
                }
                if (!transport.isConnected()) {
                    getAuthenticationContext().connectService(transport);
                }
                log.debug("Sending message");
                sendMimeMessage(message, transport);
            } finally {
                if (!isTransportCachingEnabled() && transport != null) {
                    try {
                        transport.close();
                    } catch (MessagingException ex) {
                        log.warn(
                                "An error has occurred whilst closing the connection to the outgoing mail server: "
                                        + getName()
                                        + ". This could be caused by a time-out on closing this connection, increase the timeout in smtp server configuration."
                                        + " Alternatively, set the 'mail.smtp.quitwait' (or 'mail.smtps.quitwait' for ssl) system property to false.",
                                ex);
                    }
                }
            }
            // If Message-Id is null or blank, then the MimeMessage may have created one automatically - lets retrieve
            // it.
            final String[] messageHeaders = message.getHeader("Message-Id");
            if (!ArrayUtils.isEmpty(messageHeaders)) {
                final String actualMessageId = messageHeaders[0];
                if (log.isDebugEnabled()) {
                    log.debug("Message was sent with Message-Id " + actualMessageId);
                }
                email.setMessageId(actualMessageId);
            }
        } catch (NamingException | MessagingException e) {
            throw new MailException(e);
        } catch (UnsupportedEncodingException e) {
            log.error("Error setting the 'from' address with an email and the user's fullname", e);
        }
    }

    private Transport getTransport(Session thisSession) throws MailException {
        try {
            if (isTransportCachingEnabled()) {
                log.debug("Obtaining transport object with cached approach.");
                return transportCache.get(thisSession);
            }
            log.debug("Obtaining transport object directly (no caching).");
            return thisSession.getTransport(getMailProtocol().getProtocol());
        } catch (ExecutionException | NoSuchProviderException e) {
            throw new MailException(e);
        }
    }

    /**
     * Extracted method, it can be used for testing purposes.
     * @param message the message to send
     * @param transport the transport protocol (probably SMTP or SSMTP)
     * @throws MessagingException see {@link Transport#sendMessage(Message, Address[])} and {@link Message#getAllRecipients()}
     */
    protected void sendMimeMessage(MimeMessage message, Transport transport) throws MessagingException {
        // {@link Transport#sendMessage(Message, Address[])} accesses a mailcap configuration file in the Java mail jar,
        // and requires the webapp ClassLoader to do so. If the current thread is in the context of a plugin, the
        // current
        // ClassLoader may not be the correct one, so we temporarily switch to the correct context while executing that
        // method.
        final Thread currentThread = Thread.currentThread();
        final ClassLoader originalClassLoader = currentThread.getContextClassLoader();
        try {
            currentThread.setContextClassLoader(SMTPMailServerImpl.class.getClassLoader());
            transport.sendMessage(message, message.getAllRecipients());
        } finally {
            currentThread.setContextClassLoader(originalClassLoader);
        }
    }

    /**
     * Send a message - but don't throw exceptions, just log the errors
     */
    @Override
    public void quietSend(Email email) throws MailException {
        try {
            send(email);
        } catch (Exception e) {
            log.error(
                    "Error sending mail. to:" + email.getTo() + ", cc:" + email.getCc() + ", bcc:" + email.getBcc()
                            + ", subject:" + email.getSubject() + ", body:" + email.getBody() + ", mimeType:"
                            + email.getMimeType() + ", encoding:" + email.getEncoding() + ", multipart:"
                            + email.getMultipart() + ", error:" + e,
                    e);
        }
    }

    @Override
    public String getType() {
        return MailServerManager.SERVER_TYPES[1];
    }

    @Override
    public String getDefaultFrom() {
        return defaultFrom;
    }

    @Override
    public void setDefaultFrom(String defaultFrom) {
        this.defaultFrom = defaultFrom;
        propertyChanged();
    }

    @Override
    public String getDefaultReplyTo() {
        return defaultReplyTo;
    }

    @Override
    public void setDefaultReplyTo(String defaultReplyTo) {
        this.defaultReplyTo = StringUtils.isBlank(defaultReplyTo) ? null : defaultReplyTo;
        propertyChanged();
    }

    @Override
    public String getPrefix() {
        return prefix;
    }

    @Override
    public void setPrefix(String prefix) {
        this.prefix = prefix;
    }

    @Override
    public boolean isRemovePrecedence() {
        return removePrecedence;
    }

    @Override
    public void setRemovePrecedence(boolean precedence) {
        this.removePrecedence = precedence;
        propertyChanged();
    }

    @Override
    public boolean isSessionServer() {
        return isSessionServer;
    }

    @Override
    public void setSessionServer(boolean sessionServer) {
        isSessionServer = sessionServer;
        propertyChanged();
    }

    @Override
    public boolean isTlsHostnameCheckRequired() {
        return tlsHostnameCheckRequired;
    }

    @Override
    public void setTlsHostnameCheckRequired(boolean tlsHostnameCheckRequired) {
        this.tlsHostnameCheckRequired = tlsHostnameCheckRequired;
    }

    private class MyAuthenticator extends Authenticator {
        public PasswordAuthentication getPasswordAuthentication() {
            return new PasswordAuthentication(getUsername(), getPassword());
        }
    }

    /// CLOVER:OFF
    @SuppressWarnings("RedundantIfStatement")
    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof SMTPMailServerImpl)) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }

        final SMTPMailServerImpl smtpMailServer = (SMTPMailServerImpl) o;

        if (isSessionServer != smtpMailServer.isSessionServer) {
            return false;
        }
        if (defaultFrom != null
                ? !defaultFrom.equals(smtpMailServer.defaultFrom)
                : smtpMailServer.defaultFrom != null) {
            return false;
        }
        if (prefix != null ? !prefix.equals(smtpMailServer.prefix) : smtpMailServer.prefix != null) {
            return false;
        }
        if (removePrecedence != smtpMailServer.removePrecedence) {
            return false;
        }

        return true;
    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 29 * result + (isSessionServer ? 1 : 0);
        result = 29 * result + (defaultFrom != null ? defaultFrom.hashCode() : 0);
        result = 29 * result + (prefix != null ? prefix.hashCode() : 0);
        result = 29 * result + (removePrecedence ? 1 : 0);
        return result;
    }

    @Override
    public String toString() {
        return new ToStringBuilder(this)
                .append("id", getId())
                .append("name", getName())
                .append("description", getDescription())
                .append("server name", getHostname())
                .append("username", getUsername())
                .append("password", getPassword())
                .append("isSessionServer", isSessionServer)
                .append("defaultFrom", defaultFrom)
                .append("prefix", prefix)
                .append("smtpPort", getPort())
                .toString();
    }

    /**
     * Discard the cached session when a property of the server changes.
     */
    @Override
    protected void propertyChanged() {
        super.propertyChanged();
        session = null;
    }

    /**
     * Initiates transport cache.
     *
     * Cached transport allows to reuse connection to mail server. Such approach may solve problems with bulk
     * mail sending (no reconnection on each mail) and improve performance. Transport cache validity is limited.
     * Once the validity time expires, the connection is closed.
     */
    private void initTransportCache() {
        transportCache = CacheBuilder.newBuilder()
                .expireAfterWrite(getTransportCacheTtl())
                .removalListener(closeTransport())
                .build(new CacheLoader<Session, Transport>() {
                    @Override
                    public Transport load(@Nonnull Session session) throws NoSuchProviderException {
                        log.debug("Obtaining a new connection (transport) to mail server.");
                        return session.getTransport(ofNullable(getMailProtocol())
                                .map(MailProtocol::getProtocol)
                                .orElse(null));
                    }
                });
    }

    public Duration getTransportCacheTtl() {
        final String configuredTtl = getProperties().getProperty(TRANSPORT_CACHE_TTL_PROPERTY);

        if (configuredTtl != null && !configuredTtl.isEmpty()) {
            try {
                log.info("'{}' is set to '{}'", TRANSPORT_CACHE_TTL_PROPERTY, configuredTtl);
                return Duration.of(Integer.parseInt(configuredTtl), MILLIS);
            } catch (DateTimeParseException | NumberFormatException ex) {
                log.warn(
                        "The value of '{}' is not a valid duration in ms: '{}', defaulting to {}ms!",
                        TRANSPORT_CACHE_TTL_PROPERTY,
                        configuredTtl,
                        DEFAULT_TRANSPORT_CACHE_TTL.toMillis(),
                        ex);
            }
        }

        return DEFAULT_TRANSPORT_CACHE_TTL;
    }

    private RemovalListener<Object, Object> closeTransport() {
        return notification -> {
            try {
                Transport transport = (Transport) notification.getValue();
                transport.close();
                log.debug("Completed closing the connection (transport) to mail server.");
            } catch (MessagingException | RuntimeException ex) {
                log.warn(
                        "An error has occurred whilst closing the connection to the outgoing mail server: " + getName()
                                + ". This could be caused by a time-out on closing this connection, increase the timeout in smtp server configuration."
                                + " Alternatively, set the 'mail.smtp.quitwait' (or 'mail.smtps.quitwait' for ssl) system property to false.",
                        ex);
            }
        };
    }

    public static class Builder<T extends Builder<T>> extends AbstractMailServer.Builder<T> {
        private boolean sessionServer;
        private String defaultFrom;
        private String defaultReplyTo;
        private String prefix;
        private String location;
        private boolean removePrecedence;
        private boolean tlsHostnameCheckRequired;

        public Builder() {
            this.mailProtocol(DEFAULT_SMTP_PROTOCOL);
            this.port(MailConstants.DEFAULT_SMTP_PORT);
        }

        @SuppressWarnings("unchecked")
        @Override
        protected T self() {
            return (T) this;
        }

        public T sessionServer(boolean sessionServer) {
            this.sessionServer = sessionServer;
            return self();
        }

        public T defaultFrom(String defaultFrom) {
            this.defaultFrom = defaultFrom;
            return self();
        }

        public T defaultReplyTo(String defaultReplyTo) {
            this.defaultReplyTo = defaultReplyTo;
            return self();
        }

        public T prefix(String prefix) {
            this.prefix = prefix;
            return self();
        }

        public T location(String location) {
            this.location = location;
            return self();
        }

        public T removePrecedence(boolean removePrecedence) {
            this.removePrecedence = removePrecedence;
            return self();
        }

        public T tlsHostnameCheckRequired(boolean tlsHostnameCheckRequired) {
            this.tlsHostnameCheckRequired = tlsHostnameCheckRequired;
            return self();
        }

        @Override
        public SMTPMailServerImpl build() {
            return new SMTPMailServerImpl(this);
        }
    }
}
