package com.atlassian.mail.server;

import java.io.Serializable;
import java.util.Objects;
import java.util.Properties;
import javax.mail.MessagingException;
import javax.mail.Service;

import com.atlassian.mail.server.auth.AuthenticationContext;
import com.atlassian.mail.server.auth.Credentials;
import com.atlassian.mail.server.auth.UserPasswordCredentials;

/**
 * Internal backward-compatible authentication context which supports default user/password authentication
 *
 */
final class InternalAuthenticationContext implements AuthenticationContext, Serializable {

    interface MutableUserPasswordCredentials extends UserPasswordCredentials {
        void setPassword(String password);

        void setUserName(String userName);
    }

    private MutableUserPasswordCredentials credentials;

    public InternalAuthenticationContext(final MutableUserPasswordCredentials credentials) {
        super();
        Objects.requireNonNull(credentials, "Not null credentials required");
        this.credentials = credentials;
    }

    @Override
    public void connectService(Service service) throws MessagingException {
        service.connect(this.credentials.getUserName(), this.credentials.getPassword());
    }

    @Override
    public Credentials getCredentials() {
        return credentials;
    }

    @Override
    public boolean isAuthenticating() {
        return this.credentials.getUserName() != null;
    }

    MutableUserPasswordCredentials getUserPasswordCredentials() {
        return this.credentials;
    }

    @Override
    public Properties preparePropertiesForSession(Properties properties) {
        String protocol = properties.getProperty("mail.transport.protocol");
        properties.put("mail." + protocol + ".auth", Boolean.toString(isAuthenticating()));
        return properties;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        InternalAuthenticationContext that = (InternalAuthenticationContext) o;
        return credentials.equals(that.credentials);
    }

    @Override
    public int hashCode() {
        return Objects.hash(credentials);
    }
}
