package com.atlassian.mail.server;

import java.util.Optional;
import java.util.Properties;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

import com.atlassian.mail.server.auth.AuthenticationContext;
import com.atlassian.mail.server.auth.AuthenticationContextFactory;
import com.atlassian.mail.server.auth.Credentials;
import com.atlassian.mail.server.auth.UserPasswordCredentials;

/**
 * Default factory for authentication context - it supports user/password credentials only
 */
public class DefaultAuthContextFactory implements AuthenticationContextFactory {

    private static volatile DefaultAuthContextFactory instance;

    public static DefaultAuthContextFactory getInstance() {
        if (instance == null) {
            synchronized (DefaultAuthContextFactory.class) {
                if (instance == null) {
                    instance = new DefaultAuthContextFactory();
                }
            }
        }
        return instance;
    }

    @Override
    public AuthenticationContext createAuthenticationContext(final Credentials credentials) {
        if (credentials instanceof UserPasswordCredentials) {
            final UserPasswordCredentials userPwdCredentials = (UserPasswordCredentials) credentials;
            return new InternalAuthenticationContext(new FactoryUserPasswordCredentials(
                    userPwdCredentials.getUserName(),
                    userPwdCredentials.getPassword(),
                    userPwdCredentials.getProperties().orElse(null)));
        }
        return null;
    }

    AuthenticationContext createAuthenticationContext(
            final InternalAuthenticationContext.MutableUserPasswordCredentials credentials) {
        return new InternalAuthenticationContext(credentials);
    }

    /**
     * Provides backward compatibility for externally created authentication context based on UserPassword credentials.
     * This allows AbstractMailServer to manage user and password via its get and set methods
     */
    private static final class FactoryUserPasswordCredentials
            implements InternalAuthenticationContext.MutableUserPasswordCredentials {
        private String userName;
        private String password;
        private final Properties properties;

        public FactoryUserPasswordCredentials(String userName, String password, final Properties properties) {
            this.userName = userName;
            this.password = password;
            this.properties = properties;
        }

        @Override
        public void setPassword(String password) {
            this.password = password;
        }

        @Override
        public void setUserName(String userName) {
            this.userName = userName;
        }

        @Override
        public String getUserName() {
            return this.userName;
        }

        @Override
        public String getPassword() {
            return this.password;
        }

        @Override
        public Optional<Properties> getProperties() {
            return Optional.ofNullable(this.properties);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof UserPasswordCredentials)) return false;
            final UserPasswordCredentials credentials = (UserPasswordCredentials) o;
            return new EqualsBuilder()
                    .append(getUserName(), credentials.getUserName())
                    .append(getPassword(), credentials.getPassword())
                    .append(getProperties(), credentials.getProperties())
                    .isEquals();
        }

        @Override
        public int hashCode() {
            return new HashCodeBuilder()
                    .append(getUserName())
                    .append(getPassword())
                    .append(getProperties())
                    .toHashCode();
        }
    }
}
