/*
 * Created by IntelliJ IDEA.
 * User: owen
 * Date: Nov 22, 2002
 * Time: 3:33:27 PM
 * CVS Revision: $Revision: 1.6 $
 * Last CVS Commit: $Date: 2006/09/29 02:48:01 $
 * Author of last CVS Commit: $Author: cowen $
 * To change this template use Options | File Templates.
 */
package com.atlassian.mail.server;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.PrintStream;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import javax.mail.Authenticator;
import javax.mail.MessagingException;
import javax.mail.PasswordAuthentication;
import javax.mail.Service;
import javax.mail.Session;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.atlassian.mail.MailConstants;
import com.atlassian.mail.MailException;
import com.atlassian.mail.MailFactory;
import com.atlassian.mail.MailProtocol;
import com.atlassian.mail.server.auth.AuthenticationContext;
import com.atlassian.mail.server.auth.AuthenticationContextAware;
import com.atlassian.mail.server.auth.Credentials;
import com.atlassian.mail.server.auth.UserPasswordCredentials;

public abstract class AbstractMailServer implements MailServer, AuthenticationContextAware, Serializable {
    private static final long serialVersionUID = 8520787806355214860L;
    protected transient Logger log = LoggerFactory.getLogger(this.getClass());
    private Long id;
    private String name;
    private String description;
    private String hostname;
    private String username = null;
    private String password = null;
    private MailProtocol mailProtocol = null;
    private String port = null;
    private long timeout;
    private long connectionTimeout;
    private boolean debug;
    private boolean tlsRequired;
    private transient PrintStream debugStream;
    private Properties props = new Properties();
    protected boolean isAuthenticating;
    private String socksHost;
    private String socksPort;
    private transient AuthenticationContext authenticationContext = null;

    public AbstractMailServer(Builder<?> builder) {
        this.id = builder.id;
        this.name = builder.name;
        this.description = builder.description;
        this.hostname = builder.hostname;
        this.mailProtocol = builder.mailProtocol;
        this.port = builder.port;
        this.timeout = builder.timeout;
        this.connectionTimeout = builder.connectionTimeout;
        this.debug = builder.debug;
        this.tlsRequired = builder.tlsRequired;
        this.debugStream = builder.debugStream;
        this.socksHost = builder.socksHost;
        this.socksPort = builder.socksPort;
        if (builder.authenticationContext != null) {
            this.setAuthenticationContext(builder.authenticationContext);
        } else {
            InternalMutableUserPasswordCredentials mutableCredentials = new InternalMutableUserPasswordCredentials();
            mutableCredentials.setUserName(builder.username);
            mutableCredentials.setPassword(builder.password);
            this.setAuthenticationContext(
                    DefaultAuthContextFactory.getInstance().createAuthenticationContext(mutableCredentials));
        }
        this.setInitialProperties();
        this.props = loadSystemProperties(props);
    }

    /**
     * @deprecated since 7.1.0. Use {@link AbstractMailServer.Builder} instead.
     */
    @Deprecated
    protected AbstractMailServer(long timeout) {
        this.authenticationContext = DefaultAuthContextFactory.getInstance()
                .createAuthenticationContext(new InternalMutableUserPasswordCredentials());
        this.timeout = timeout;
        this.connectionTimeout = timeout;
        this.setInitialProperties();
    }

    /**
     * @deprecated since 7.1.0. Use {@link AbstractMailServer.Builder} instead.
     */
    @Deprecated
    public AbstractMailServer() {
        this.authenticationContext = DefaultAuthContextFactory.getInstance()
                .createAuthenticationContext(new InternalMutableUserPasswordCredentials());
        this.setInitialProperties();
    }

    /**
     * @deprecated since 7.1.0. Use {@link AbstractMailServer.Builder} instead.
     */
    @Deprecated
    public AbstractMailServer(
            Long id,
            String name,
            String description,
            MailProtocol protocol,
            String hostName,
            String port,
            String username,
            String password,
            long timeout,
            String socksHost,
            String socksPort) {
        final InternalMutableUserPasswordCredentials mutableCredentials = new InternalMutableUserPasswordCredentials();
        mutableCredentials.setUserName(username);
        mutableCredentials.setPassword(password);
        this.authenticationContext =
                DefaultAuthContextFactory.getInstance().createAuthenticationContext(mutableCredentials);
        this.id = id;
        this.name = name;
        this.description = description;
        this.hostname = hostName;
        this.mailProtocol = protocol;
        this.port = port;
        this.timeout = timeout;
        this.connectionTimeout = timeout;
        this.socksHost = socksHost;
        this.socksPort = socksPort;
        this.setInitialProperties();
        // MAIL-60: Need to ensure that system properties get set for mail servers.
        this.props = loadSystemProperties(props);
    }

    /**
     * @deprecated since 7.1.0. Use {@link AbstractMailServer.Builder} instead.
     */
    @Deprecated
    public AbstractMailServer(
            Long id,
            String name,
            String description,
            MailProtocol protocol,
            String hostName,
            String port,
            AuthenticationContext authenticationContext,
            long timeout,
            String socksHost,
            String socksPort) {
        Objects.requireNonNull(authenticationContext, "not null authentication context required");
        this.authenticationContext = authenticationContext;
        this.id = id;
        this.name = name;
        this.description = description;
        this.hostname = hostName;
        this.mailProtocol = protocol;
        this.port = port;
        this.timeout = timeout;
        this.connectionTimeout = timeout;
        this.socksHost = socksHost;
        this.socksPort = socksPort;
        this.setInitialProperties();
        // MAIL-60: Need to ensure that system properties get set for mail servers.
        this.props = loadSystemProperties(props);
    }

    private void setInitialProperties() {
        final MailProtocol mailProtocol = getMailProtocol();
        if (mailProtocol != null) {
            final String protocol = mailProtocol.getProtocol();
            props.put("mail.store.protocol", protocol);
            props.put("mail." + protocol + ".host", "" + getHostname());
            props.put("mail." + protocol + ".port", "" + getPort());
            props.put("mail." + protocol + ".timeout", "" + getTimeout());
            props.put("mail." + protocol + ".connectiontimeout", "" + getConnectionTimeout());
            props.put("mail.transport.protocol", "" + protocol);
            if (mailProtocol == MailProtocol.SMTP || mailProtocol == MailProtocol.SECURE_SMTP) {
                final String mailSmtpQuitWaitPropertyName = "mail." + protocol + ".quitwait";
                // by default it's set to false but allow overriding to true with system property
                props.put(
                        mailSmtpQuitWaitPropertyName,
                        Boolean.toString(Boolean.getBoolean(mailSmtpQuitWaitPropertyName)));
            }
            if (isTlsRequired()) {
                props.put("mail." + protocol + ".starttls.enable", "true");
            }

            isAuthenticating = getAuthenticationContext().isAuthenticating();

            if (StringUtils.isNotBlank(getSocksHost())) {
                props.put("mail." + protocol + ".socks.host", getSocksHost());
            }

            if (StringUtils.isNotBlank(getSocksPort())) {
                props.put("mail." + protocol + ".socks.port", getSocksPort());
            }
        }
        props.put("mail.debug", "" + getDebug());
        if (Boolean.getBoolean("mail.debug")) {
            props.put("mail.debug", "true");
        }

        props = getAuthenticationContext().preparePropertiesForSession(props);
    }

    protected Authenticator getAuthenticator() {
        final Credentials credentials = this.getAuthenticationContext().getCredentials();
        if (credentials instanceof UserPasswordCredentials) {
            final UserPasswordCredentials userPwdCred = (UserPasswordCredentials) credentials;
            return new Authenticator() {
                @Override
                protected PasswordAuthentication getPasswordAuthentication() {
                    return new PasswordAuthentication(userPwdCred.getUserName(), userPwdCred.getPassword());
                }
            };
        }
        return null;
    }

    public void smartConnect(Service service) throws MessagingException {
        getAuthenticationContext().connectService(service);
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
        propertyChanged();
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
        propertyChanged();
    }

    public String getDescription() {
        return description;
    }

    public void setDescription(String description) {
        this.description = description;
        propertyChanged();
    }

    public String getHostname() {
        return hostname;
    }

    public void setHostname(String serverName) {
        this.hostname = serverName;
        propertyChanged();
    }

    public AuthenticationContext getAuthenticationContext() {
        return this.authenticationContext;
    }

    public void setAuthenticationContext(AuthenticationContext context) {
        this.authenticationContext = context;
    }

    public String getUsername() {
        return getInternalContext()
                .map(InternalAuthenticationContext::getUserPasswordCredentials)
                .map(UserPasswordCredentials::getUserName)
                .orElse(null);
    }

    public void setUsername(String username) {
        getInternalContext().ifPresent(ctx -> {
            ctx.getUserPasswordCredentials().setUserName(username);
            propertyChanged();
        });
    }

    public String getPassword() {
        return getInternalContext()
                .map(InternalAuthenticationContext::getUserPasswordCredentials)
                .map(UserPasswordCredentials::getPassword)
                .orElse(null);
    }

    public void setPassword(String password) {
        getInternalContext().ifPresent(ctx -> {
            ctx.getUserPasswordCredentials().setPassword(password);
            propertyChanged();
        });
    }

    private Optional<InternalAuthenticationContext> getInternalContext() {
        return Optional.ofNullable(getAuthenticationContext())
                .filter(c -> (c instanceof InternalAuthenticationContext))
                .map(c -> (InternalAuthenticationContext) c);
    }

    public MailProtocol getMailProtocol() {
        return mailProtocol;
    }

    public void setMailProtocol(final MailProtocol protocol) {
        this.mailProtocol = protocol;
        propertyChanged();
    }

    public String getPort() {
        return port;
    }

    public void setPort(final String port) {
        this.port = port;
        propertyChanged();
    }

    public long getTimeout() {
        return timeout;
    }

    public long getConnectionTimeout() {
        return connectionTimeout;
    }

    public void setTimeout(long timeout) {
        this.timeout = timeout;
        propertyChanged();
    }

    public void setConnectionTimeout(long timeout) {
        this.connectionTimeout = timeout;
    }

    public String getSocksHost() {
        return socksHost;
    }

    public void setSocksHost(String socksHost) {
        this.socksHost = socksHost;
        propertyChanged();
    }

    public String getSocksPort() {
        return socksPort;
    }

    public void setSocksPort(String socksPort) {
        this.socksPort = socksPort;
        propertyChanged();
    }

    public boolean isTlsRequired() {
        return tlsRequired;
    }

    public void setTlsRequired(final boolean tlsRequired) {
        this.tlsRequired = tlsRequired;
        propertyChanged();
    }

    public Properties getProperties() {
        return props;
    }

    public void setProperties(Properties props) {
        this.props = props;
        propertyChanged();
    }

    public void setDebug(boolean debug) {
        this.debug = debug;
        propertyChanged();
    }

    public void setDebugStream(PrintStream debugStream) {
        this.debugStream = debugStream;
        propertyChanged();
    }

    public boolean getDebug() {
        return this.debug;
    }

    public PrintStream getDebugStream() {
        return this.debugStream;
    }

    /// CLOVER:OFF
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof AbstractMailServer)) return false;
        final AbstractMailServer abstractMailServer = (AbstractMailServer) o;
        return new EqualsBuilder()
                .append(id, abstractMailServer.id)
                .append(name, abstractMailServer.name)
                .append(description, abstractMailServer.description)
                .append(hostname, abstractMailServer.hostname)
                .append(getAuthenticationContext(), abstractMailServer.getAuthenticationContext())
                .append(mailProtocol, abstractMailServer.mailProtocol)
                .append(port, abstractMailServer.port)
                .append(socksHost, abstractMailServer.socksHost)
                .append(socksPort, abstractMailServer.socksPort)
                .isEquals();
    }

    public int hashCode() {
        return new HashCodeBuilder()
                .append(id)
                .append(name)
                .append(description)
                .append(hostname)
                .append(getAuthenticationContext())
                .append(mailProtocol)
                .append(port)
                .append(socksHost)
                .append(socksPort)
                .toHashCode();
    }

    public String toString() {
        return new ToStringBuilder(this)
                .append("id", id)
                .append("name", name)
                .append("description", description)
                .append("server name", hostname)
                .append("username", getUsername())
                .append("password", getPassword() != null ? "***" : "<unset>")
                .append("authenticationContext", getAuthenticationContext())
                .append("mail protocol", mailProtocol)
                .append("port", port)
                .append("socks host", socksHost)
                .append("socks port", socksPort)
                .toString();
    }

    /**
     * Call this method whenever a property of the server changes.
     * Subclasses should override it to clear any cached information.
     */
    protected void propertyChanged() {
        setInitialProperties();
    }

    private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException {
        ois.defaultReadObject();
        log = LoggerFactory.getLogger(this.getClass());
        this.authenticationContext = DefaultAuthContextFactory.getInstance()
                .createAuthenticationContext(new InternalMutableUserPasswordCredentials());
    }

    /**
     * This allows users of atlassian mail to add command line properties to modify session defaults
     * See JRA-11452
     * <p>
     * The hierarchy now is - default properties, then from the command line, then properties added via setProperties
     *
     * @param p the default properties for the current mail session
     * @return the properties with the system properties loaded
     */
    protected synchronized Properties loadSystemProperties(Properties p) {
        Properties props = new Properties();
        props.putAll(p);
        props.putAll(System.getProperties());
        if (this.props != null) {
            props.putAll(this.props);
        }
        return props;
    }

    public void setLogger(Logger logger) {
        this.log = logger;
    }

    // This whole ugly code below is just to log some internal Session details, which are sometimes
    // necessary for debugging purposes by our support, but they are not logged where they should be
    // namely because Session logs a lot of stuff in its constructor before you can pass it a desired
    // debug stream.
    protected void getMoreDebugInfoAboutCreatedSession(Session session) {
        log.debug("Session providers: [" + Arrays.toString(session.getProviders()) + "]");
        try {
            final Field addressMapField = Session.class.getDeclaredField("addressMap");
            final boolean originalAccessibility = addressMapField.isAccessible();
            addressMapField.setAccessible(true);
            try {
                log.debug("Session addressMap: [" + addressMapField.get(session) + "]");
            } finally {
                addressMapField.setAccessible(originalAccessibility);
            }

        } catch (Exception e) {
            log.debug("Cannot retrieve Session details via reflections: " + e.getMessage(), e);
        }
    }

    protected Session getSessionFromServerManager(Properties props, final Authenticator authenticator)
            throws MailException {
        log.debug("Getting session");
        if (getDebug()) {
            log.debug("Debug messages from JavaMail session initialization will not appear in this log."
                    + " These messages are sent to standard out.");
        }

        props = getAuthenticationContext().preparePropertiesForSession(props);
        final Session session = getSessionFromServerManagerInternal(props, authenticator);

        if (log.isDebugEnabled()) {
            getMoreDebugInfoAboutCreatedSession(session);
        }
        if (getDebugStream() != null) {
            try {
                session.setDebugOut(getDebugStream());
            } catch (NoSuchMethodError nsme) {
                // JRA-8543
                log.error(
                        "Warning: An old (pre-1.3.2) version of the JavaMail library (javamail.jar or mail.jar) bundled with your app server, is in use. Some functions such as IMAPS/POPS/SMTPS will not work. Consider upgrading the app server's javamail jar to the version JIRA provides.");
            }
        }
        return session;
    }

    protected Session getSessionFromServerManagerInternal(Properties props, Authenticator authenticator)
            throws MailException {
        return MailFactory.getServerManager().getSession(props, authenticator);
    }

    /**
     *  Internal implementation of user password credential makes server instances
     *  where user and password are stored in AbstractMailServer fields backward compatible with AuthenticationContext interface
     */
    private final class InternalMutableUserPasswordCredentials
            implements InternalAuthenticationContext.MutableUserPasswordCredentials {
        @Override
        public String getUserName() {
            return AbstractMailServer.this.username;
        }

        @Override
        public String getPassword() {
            return AbstractMailServer.this.password;
        }

        @Override
        public void setPassword(String password) {
            if (StringUtils.isNotBlank(password)) AbstractMailServer.this.password = password;
            else AbstractMailServer.this.password = null;
        }

        @Override
        public void setUserName(String userName) {
            if (StringUtils.isNotBlank(userName)) AbstractMailServer.this.username = userName;
            else AbstractMailServer.this.username = null;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (!(o instanceof UserPasswordCredentials)) return false;
            final UserPasswordCredentials credentials = (UserPasswordCredentials) o;
            return new EqualsBuilder()
                    .append(getUserName(), credentials.getUserName())
                    .append(getPassword(), credentials.getPassword())
                    .append(getProperties(), credentials.getProperties())
                    .isEquals();
        }

        @Override
        public int hashCode() {
            return new HashCodeBuilder()
                    .append(getUserName())
                    .append(getPassword())
                    .append(getProperties())
                    .toHashCode();
        }

        @Override
        public Optional<Properties> getProperties() {
            return Optional.ofNullable(null);
        }
    }

    public abstract static class Builder<T extends Builder<T>> {
        private Long id;
        private String name;
        private String description;
        private String hostname;
        private String username;
        private String password;
        private MailProtocol mailProtocol;
        private String port;
        private long timeout = MailConstants.DEFAULT_TIMEOUT;
        private long connectionTimeout = MailConstants.DEFAULT_TIMEOUT;
        private boolean debug;
        private boolean tlsRequired;
        private PrintStream debugStream;
        private String socksHost;
        private String socksPort;
        private AuthenticationContext authenticationContext;

        protected abstract T self();

        public abstract AbstractMailServer build();

        public T id(Long id) {
            this.id = id;
            return self();
        }

        public T name(String name) {
            this.name = name;
            return self();
        }

        public T description(String description) {
            this.description = description;
            return self();
        }

        public T hostname(String hostname) {
            this.hostname = hostname;
            return self();
        }

        public T username(String username) {
            this.username = username;
            return self();
        }

        public T password(String password) {
            this.password = password;
            return self();
        }

        public T mailProtocol(MailProtocol mailProtocol) {
            this.mailProtocol = mailProtocol;
            return self();
        }

        public T port(String port) {
            this.port = port;
            return self();
        }

        public T timeout(long timeout) {
            this.timeout = timeout;
            return self();
        }

        public T connectionTimeout(long timeout) {
            this.connectionTimeout = timeout;
            return self();
        }

        public T debug(boolean debug) {
            this.debug = debug;
            return self();
        }

        public T tlsRequired(boolean tlsRequired) {
            this.tlsRequired = tlsRequired;
            return self();
        }

        public T debugStream(PrintStream debugStream) {
            this.debugStream = debugStream;
            return self();
        }

        public T socksHost(String socksHost) {
            this.socksHost = socksHost;
            return self();
        }

        public T socksPort(String socksPort) {
            this.socksPort = socksPort;
            return self();
        }

        public T authenticationContext(AuthenticationContext authenticationContext) {
            this.authenticationContext = authenticationContext;
            return self();
        }
    }
}
