package com.atlassian.mail.options;

import javax.annotation.ParametersAreNonnullByDefault;

import com.atlassian.mail.converters.HtmlConverter;
import com.atlassian.mail.converters.basic.HtmlToTextConverter;

/**
 * @since v2.7.11
 */
@ParametersAreNonnullByDefault
public class GetBodyOptions {

    private static final HtmlConverter HTML_TO_TEXT_CONVERTER = new HtmlToTextConverter();

    /**
     * Options that will prefer the text part for the body, and not strip any whitespace.
     * <p>
     * If only a HTML part is present, then will use {@link HtmlToTextConverter} to convert to text.
     */
    public static final GetBodyOptions PREFER_TEXT_BODY_NO_STRIP_WHITESPACE =
            new GetBodyOptions(false, false, HTML_TO_TEXT_CONVERTER);

    /**
     * Options that will prefer the text part for the body, and will strip any whitespace.
     * <p>
     * If only a HTML part is present, then will use {@link HtmlToTextConverter} to convert to text.
     */
    public static final GetBodyOptions PREFER_TEXT_BODY_STRIP_WHITESPACE =
            new GetBodyOptions(true, false, HTML_TO_TEXT_CONVERTER);

    private final boolean stripWhitespace;
    private final boolean preferHtmlPart;
    private final HtmlConverter htmlConverter;

    private GetBodyOptions(boolean stripWhitespace, boolean preferHtmlPart, HtmlConverter htmlConverter) {
        this.stripWhitespace = stripWhitespace;
        this.preferHtmlPart = preferHtmlPart;
        this.htmlConverter = htmlConverter;
    }

    public boolean isStripWhitespace() {
        return stripWhitespace;
    }

    public boolean isPreferHtmlPart() {
        return preferHtmlPart;
    }

    public HtmlConverter getHtmlConverter() {
        return htmlConverter;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(GetBodyOptions template) {
        return new Builder(template);
    }

    public static class Builder {

        private boolean stripWhitespace;
        private boolean preferHtmlPart;
        private HtmlConverter htmlConverter;

        private Builder() {
            this.htmlConverter = HTML_TO_TEXT_CONVERTER;
        }

        private Builder(GetBodyOptions template) {
            this.stripWhitespace = template.stripWhitespace;
            this.preferHtmlPart = template.preferHtmlPart;
            this.htmlConverter = template.htmlConverter;
        }

        /**
         * Remove all whitespace from the start and end of text
         * <p>
         * Whitespace is defined by {@link Character#isWhitespace(char)}.
         * @return the builder with the "stripWhitespace" toggled on
         */
        public Builder stripWhitespace() {
            return setStripWhitespace(true);
        }

        /**
         * Set whether to remove all whitespace from the start and end of text
         * <p>
         * Whitespace is defined by {@link Character#isWhitespace(char)}.
         *
         * @param stripWhitespace to toggle stripping whitespace on or off
         * @return the builder with the "stripWhitespace" flag set appropriately
         * @since v2.7.13
         */
        public Builder setStripWhitespace(boolean stripWhitespace) {
            this.stripWhitespace = stripWhitespace;
            return this;
        }

        /**
         * Prefer processing of the HTML content type if available, over the Text content type of a multipart/alternative
         * message
         * @return the builder with te "preferHtmlPart" flag toggled on
         */
        public Builder preferHtmlPart() {
            return setPreferHtmlPart(true);
        }

        /**
         * Set whether to prefer processing of the HTML content type if available, over the Text content type of
         * a multipart/alternative message
         *
         * @param preferHtmlPart to toggle "preferHtmlPart" on or off
         * @return the builder with the "preferHtmlPart" flag set appropriately
         * @since v2.7.13
         */
        public Builder setPreferHtmlPart(boolean preferHtmlPart) {
            this.preferHtmlPart = preferHtmlPart;
            return this;
        }

        /**
         * If the message is HTML only or {@link #preferHtmlPart()} then use this {@link HtmlConverter} to convert from
         * the html to desired output.
         * <p>
         * By default will use {@link HtmlToTextConverter} if not provided here.
         *
         * @param htmlConverter The html converter to produce desired output
         * @return the builder with the {@link HtmlConverter} set
         */
        public Builder setHtmlConverter(HtmlConverter htmlConverter) {
            this.htmlConverter = htmlConverter;
            return this;
        }

        public GetBodyOptions build() {
            return new GetBodyOptions(stripWhitespace, preferHtmlPart, htmlConverter);
        }
    }
}
