package com.atlassian.mail.converters.wiki;

import java.util.ArrayDeque;
import java.util.Deque;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jsoup.internal.StringUtil;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.repeat;

@ParametersAreNonnullByDefault
final class ListHandler {

    public static final String HTML_OL = "ol";
    public static final String HTML_UL = "ul";
    public static final String HTML_LI = "li";
    public static final String HTML_DL = "dl";
    public static final String HTML_DT = "dt";
    public static final String HTML_DD = "dd";

    private static final String WIKI_OL_LI = "#";
    private static final String WIKI_UL_LI = "*";
    private static final String WIKI_DT = EMPTY;
    private static final String WIKI_DD = ".... ";
    private static final String NEWLINE = "\n";

    private final BlockStyleHandler blockStyleHandler;

    private boolean inDescription;

    private final Deque<String> listWikiStack = new ArrayDeque<String>();

    public ListHandler(final BlockStyleHandler blockStyleHandler) {
        this.blockStyleHandler = blockStyleHandler;
    }

    public String enter(final String name) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        if (HTML_OL.equals(name)) {
            listWikiStack.addFirst(WIKI_OL_LI);
            return NEWLINE;
        } else if (HTML_UL.equals(name)) {
            listWikiStack.addFirst(WIKI_UL_LI);
            return NEWLINE;
        } else if (HTML_DL.equals(name)) {
            inDescription = true;
            return NEWLINE;
        } else if (!listWikiStack.isEmpty() && HTML_LI.equals(name)) {
            return repeat(listWikiStack.peekFirst(), listWikiStack.size()) + " ";
        } else if (inDescription) {
            if (HTML_DD.equals(name)) {
                return WIKI_DD;
            } else if (HTML_DT.equals(name)) {
                return WIKI_DT;
            }
        }

        return EMPTY;
    }

    public String exit(final String name) {
        if (!this.blockStyleHandler.isFormattingPossible()) {
            return EMPTY;
        }

        if (HTML_OL.equals(name)) {
            listWikiStack.pollFirst();
            return NEWLINE;
        } else if (HTML_UL.equals(name)) {
            listWikiStack.pollFirst();
            return NEWLINE;
        } else if (HTML_DL.equals(name)) {
            inDescription = false;
            return NEWLINE;
        } else if (StringUtil.in(name, HTML_LI, HTML_DD, HTML_DT)) {
            return NEWLINE;
        }

        return EMPTY;
    }
}
