package com.atlassian.mail.converters.wiki;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.jsoup.internal.StringUtil.isWhitespace;

@ParametersAreNonnullByDefault
final class DocumentUtilities {

    private DocumentUtilities() {}

    @Nonnull
    static DocumentElement parseHtml(final String html) {
        final Document document = Jsoup.parse(html);

        document.outputSettings().prettyPrint(false);

        return new DocumentElement(document);
    }

    @Nonnull
    static BodyElement getBody(final DocumentElement document) {
        final Element body = document.getDocument().body();
        return new BodyElement(body);
    }

    @Nonnull
    static String removeTrailingWhitespace(final StringBuilder sb) {
        // need to wrap tightly
        StringBuilder whiteSpace = new StringBuilder();
        while (sb.length() > 0 && isWhitespace(sb.charAt(sb.length() - 1))) {
            whiteSpace.append(sb.charAt(sb.length() - 1));
            sb.deleteCharAt(sb.length() - 1);
        }

        return whiteSpace.reverse().toString();
    }

    static class DocumentElement {
        private final Document document;

        private DocumentElement(Document document) {
            this.document = checkNotNull(document, "Document can not be null");
        }

        public Document getDocument() {
            return document;
        }
    }

    static class BodyElement {
        private final Element body;

        private BodyElement(Element body) {
            this.body = checkNotNull(body, "Body can not be null");
            checkArgument(body.tag().getName().equals("body"), "Element must be the body");
        }

        public Element getBody() {
            return body;
        }
    }
}
