/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package com.atlassian.labs.httpservice;


import com.atlassian.labs.httpservice.descriptors.HttpServiceServletFilterModuleDescriptor;
import com.atlassian.labs.httpservice.descriptors.LegacyHttpServiceServletFilterModuleDescriptor;
import com.atlassian.labs.httpservice.resource.ResourceFilter;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.servlet.ServletModuleManager;
import com.atlassian.plugin.servlet.descriptors.ServletFilterModuleDescriptor;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.osgi.framework.Bundle;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.http.HttpContext;
import org.osgi.service.http.HttpService;
import org.osgi.service.http.NamespaceException;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import java.util.Dictionary;
import java.util.Hashtable;


public class HttpServiceImpl implements HttpService
{
    private final ServletManager servletManager;
    private final ResourceFilter resourceFilter;

    /** Bundle which "got" this service instance from the service factory */
    private final Bundle bundle;
    private static final Log log = LogFactory.getLog(HttpServiceImpl.class);
    private final ServiceRegistration filterReg;
    private ServletFilterModuleDescriptor filterDescriptor;

    public HttpServiceImpl(Plugin plugin, Bundle bundle, ServletModuleManager servletModuleManager)
    {
        this.bundle = bundle;
        this.servletManager = new ServletManager(plugin, bundle, servletModuleManager);
        this.resourceFilter = new ResourceFilter(servletManager);
        // todo: check if already exists
        filterDescriptor = createServletDescriptor(plugin, servletModuleManager);
        filterDescriptor.enabled();
        filterReg = bundle.getBundleContext().registerService(ModuleDescriptor.class.getName(), filterDescriptor, null);
    }

    private ServletFilterModuleDescriptor createServletDescriptor(Plugin plugin, ServletModuleManager servletModuleManager)
    {
        try
        {
            getClass().getClassLoader().loadClass("com.atlassian.plugin.module.ModuleFactory");
            return new HttpServiceServletFilterModuleDescriptor(plugin, resourceFilter, "/*", new Hashtable(), servletModuleManager);
        }
        catch (ClassNotFoundException e)
        {
            return new LegacyHttpServiceServletFilterModuleDescriptor(plugin, resourceFilter, "/*", new Hashtable(), servletModuleManager);
        }

    }

    public HttpContext createDefaultHttpContext()
    {
        return new DefaultContextImpl(bundle);
    }


    public void registerServlet( String alias, Servlet servlet, Dictionary params, HttpContext osgiHttpContext )
        throws ServletException, NamespaceException
    {
        log.debug( "http register servlet :" + bundle + ", alias: " + alias );

        validateAlias( alias );

        // todo: check if already exists

        servletManager.addServlet(alias, servlet, params);
    }

    public void registerResources( String alias, String name, HttpContext osgiHttpContext ) throws NamespaceException
    {
        log.debug( "** http register resource :" + bundle + ", alias: " + alias );

        validateAlias( alias );
        validateName( name );

        if ( osgiHttpContext == null )
        {
            osgiHttpContext = createDefaultHttpContext();
        }

        resourceFilter.addResource(alias, name, osgiHttpContext);
    }


    public void unregister( String alias )
    {
        servletManager.removeServlet(alias);
        resourceFilter.destroy();
    }


    void unregisterAll()
    {
        servletManager.removeAll();
        resourceFilter.destroy();
        filterDescriptor.disabled();
        filterReg.unregister();
    }

    private void validateAlias( String alias )
    {
        if (alias == null || (!alias.equals( "/" ) && ( !alias.startsWith( "/" ) || alias.endsWith( "/" ) ) ))
        {
            throw new IllegalArgumentException("Invalid alias: " + alias);
        }
    }
    
    
    private void validateName( String name )
    {
        if (name == null || name.endsWith( "/" ))
        {
            throw new IllegalArgumentException("Invalid name: " + name);
        }
    }
}