/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.fastdev.maven;

import com.atlassian.fastdev.maven.MavenTaskError;
import com.atlassian.fastdev.util.Either;
import com.atlassian.fastdev.util.Option;
import com.atlassian.fastdev.util.Pair;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.MapMaker;
import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class MavenTaskManager
implements DisposableBean {
    private static final long DEFAULT_AVERAGE_BUILD_TIME = 0L;
    private final Logger LOG = LoggerFactory.getLogger(MavenTaskManager.class);
    private final ExecutorService executorService;
    private final ConcurrentMap<File, MavenTask> inFlightTasks;
    private final ConcurrentMap<UUID, MavenTask> tasks;
    private final ConcurrentMap<UUID, MavenTask> completedTasks;
    private final ConcurrentMap<File, RunningAverage> averageBuildTimes;
    private final Iterable<Pair<String, String>> mavenCommands = ImmutableList.of(Pair.pair("ATLAS_HOME", "/apache-maven/bin/mvn"), Pair.pair("M2_HOME", "/bin/mvn"));

    public MavenTaskManager() {
        this(Executors.newFixedThreadPool(8, new MavenThreadFactory()));
    }

    public MavenTaskManager(ExecutorService executorService) {
        this.executorService = executorService;
        this.tasks = new MapMaker().makeMap();
        this.inFlightTasks = new MapMaker().makeMap();
        this.averageBuildTimes = new MapMaker().makeMap();
        this.completedTasks = new MapMaker().expiration(30L, TimeUnit.SECONDS).makeMap();
    }

    private String getMavenCommand() {
        for (String string : Option.option(System.getProperty("fastdev.mvn.command"))) {
            if (new File(string).exists()) {
                return string;
            }
            this.LOG.warn("Cannot use maven command " + string + " from property " + "fastdev.mvn.command" + " because it does not exist");
        }
        for (Pair pair : this.mavenCommands) {
            for (String path : Option.option(System.getenv((String)pair.first()))) {
                File cmd = new File(path, (String)pair.second());
                if (cmd.exists()) {
                    return cmd.getAbsolutePath();
                }
                this.LOG.warn("Cannot use maven command " + cmd.getAbsolutePath() + " from property " + (String)pair.first() + " because it does not exist");
            }
        }
        return "mvn";
    }

    public Either<MavenTaskError, MavenTask> createTask(File buildRoot, List<String> commands) {
        if (!buildRoot.isDirectory() || !new File(buildRoot, "pom.xml").exists()) {
            this.LOG.error("Failed to execute build: " + buildRoot.getPath() + " is not a maven root");
            return Either.left(MavenTaskError.INVALID_BUILD_ROOT);
        }
        UUID uuid = UUID.randomUUID();
        MavenTask task = new MavenTask(uuid, buildRoot, commands);
        return this.add(task);
    }

    public MavenTask getTask(UUID uuid) {
        return (MavenTask)this.tasks.get(uuid);
    }

    public MavenTask getTaskForRoot(File file) {
        return (MavenTask)this.inFlightTasks.get(file);
    }

    public MavenTask getCompletedTask(UUID uuid) {
        return (MavenTask)this.completedTasks.get(uuid);
    }

    public Iterable<MavenTask> getAllTasks() {
        return ImmutableList.copyOf(this.tasks.values());
    }

    private Either<MavenTaskError, MavenTask> add(MavenTask task) {
        MavenTask existing = this.inFlightTasks.putIfAbsent(task.getBuildRoot(), task);
        if (existing != null) {
            this.LOG.warn("Ignoring build: " + task.getBuildRoot().getAbsolutePath() + " as a plugin install is currently in progress");
            return Either.left(MavenTaskError.PLUGIN_INSTALL_IN_PROGRESS);
        }
        this.tasks.put(task.getUuid(), task);
        return Either.right(task);
    }

    private void remove(MavenTask task) {
        this.inFlightTasks.remove(task.getBuildRoot());
        this.completedTasks.put(task.getUuid(), task);
        this.tasks.remove(task.getUuid());
    }

    private long averageBuildTime(File buildRoot) {
        RunningAverage ave = this.averageBuildTimes.putIfAbsent(buildRoot, new RunningAverage(0, 0L));
        if (ave == null) {
            return 0L;
        }
        return ave.average();
    }

    private void recordBuildTime(File buildRoot, long latest) {
        boolean replaced = false;
        while (!replaced) {
            RunningAverage oldAverage = (RunningAverage)this.averageBuildTimes.get(buildRoot);
            replaced = this.averageBuildTimes.replace(buildRoot, oldAverage, oldAverage.adjust(latest));
        }
    }

    public void destroy() {
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.LOG.warn("Timed out while waiting for Maven executor to shutdown.");
            }
        }
        catch (InterruptedException e) {
            this.LOG.warn("Interrupted while waiting for Maven executor to shutdown.");
        }
    }

    private static class MavenThreadFactory
    implements ThreadFactory {
        private final AtomicLong id = new AtomicLong();

        private MavenThreadFactory() {
        }

        public Thread newThread(Runnable r) {
            Thread thread = new Thread(r);
            thread.setName(String.format("FastDev-MavenTask-%d", this.id.getAndIncrement()));
            thread.setDaemon(true);
            return thread;
        }
    }

    private static class RunningAverage {
        private final int n;
        private final long total;

        public RunningAverage(int n, long total) {
            this.n = n;
            this.total = total;
        }

        public long average() {
            if (this.n == 0) {
                return 0L;
            }
            return this.total / (long)this.n;
        }

        public RunningAverage adjust(long latest) {
            return new RunningAverage(this.n + 1, this.total + latest);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            RunningAverage that = (RunningAverage)o;
            if (this.n != that.n) {
                return false;
            }
            return this.total == that.total;
        }

        public int hashCode() {
            int result = this.n;
            result = 31 * result + (int)(this.total ^ this.total >>> 32);
            return result;
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public class MavenTask {
        private final File buildRoot;
        private final ProcessBuilder builder;
        private final UUID uuid;
        private final ConcurrentLinkedQueue<String> output;
        private volatile Integer exitCode;
        private final long currentAverage;
        private volatile long startTime;

        private MavenTask(UUID uuid, File buildRoot, List<String> commands) {
            ImmutableList.Builder commandsBuilder = ImmutableList.builder();
            if (SystemUtils.IS_OS_WINDOWS) {
                commandsBuilder.add((Object[])new String[]{"cmd", "/c"});
            }
            commandsBuilder.add((Object)MavenTaskManager.this.getMavenCommand());
            commandsBuilder.addAll(commands);
            this.builder = new ProcessBuilder((List<String>)commandsBuilder.build());
            this.builder.directory(buildRoot);
            this.builder.environment().remove("MAVEN_COLOR");
            this.uuid = uuid;
            this.output = new ConcurrentLinkedQueue();
            this.buildRoot = buildRoot;
            this.currentAverage = MavenTaskManager.this.averageBuildTime(buildRoot);
        }

        public File getBuildRoot() {
            return this.buildRoot;
        }

        public long getAverageTaskTime() {
            return this.currentAverage;
        }

        public UUID getUuid() {
            return this.uuid;
        }

        public Iterable<String> getOutput() {
            return ImmutableList.copyOf(this.output);
        }

        public Integer getExitCode() {
            return this.exitCode;
        }

        public long getElapsedTime() {
            return System.currentTimeMillis() - this.startTime;
        }

        public Future<Integer> start() {
            return MavenTaskManager.this.executorService.submit(new Callable<Integer>(){

                @Override
                public Integer call() throws Exception {
                    try {
                        String lineRead;
                        Process process = MavenTask.this.builder.start();
                        MavenTaskManager.this.LOG.info("Maven process executed as : " + Joiner.on((String)" ").join(MavenTask.this.builder.command()));
                        MavenTask.this.startTime = System.currentTimeMillis();
                        InputStreamReader isr = new InputStreamReader(process.getInputStream());
                        BufferedReader br = new BufferedReader(isr);
                        while ((lineRead = br.readLine()) != null) {
                            MavenTask.this.output.add(lineRead);
                        }
                        process.waitFor();
                        MavenTask.this.exitCode = process.exitValue();
                        MavenTaskManager.this.remove(MavenTask.this);
                        MavenTaskManager.this.recordBuildTime(MavenTask.this.buildRoot, System.currentTimeMillis() - MavenTask.this.startTime);
                        return process.exitValue();
                    }
                    catch (Exception e) {
                        MavenTaskManager.this.LOG.error("Maven execution failed", (Throwable)e);
                        throw e;
                    }
                }
            });
        }
    }
}

