package com.atlassian.johnson.spring.web;

import org.springframework.util.StringUtils;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;

/**
 * Constants related to determining which {@link com.atlassian.johnson.event.EventType EventType} to use for Spring-
 * related {@link com.atlassian.johnson.event.Event Event}s.
 * 
 * @since 2.0
 */
public class SpringEventType
{
    /**
     * Defines the {@code init-param} which may be used for controlling whether an event is added when a portion of
     * Spring initialisation is bypassed due to previous errors.
     * <p/>
     * Note: This flag does not control whether an event is added when Spring initialisation is not bypassed and fails.
     *
     * @see #addEventOnBypass(javax.servlet.ServletContext)
     */
    public static final String ADD_EVENT_ON_BYPASS_PARAM = "johnson.spring.addEventOnBypass";
    /**
     * Defines the {@code init-param} which may be used for controlling the event type added when Spring events occur.
     * Where the value must be set depends on the type being initialised.
     *
     * @see #getContextEventType(javax.servlet.ServletContext)
     * @see #getServletEventType(javax.servlet.ServletConfig)
     */
    public static final String EVENT_TYPE_PARAM = "johnson.spring.eventType";

    /**
     * Defines the default context event type which will be used if one is not explicitly set.
     */
    public static final String SPRING_CONTEXT_EVENT_TYPE = "spring";
    /**
     * Defines the default servlet event type which will be used if one is not explicitly set.
     */
    public static final String SPRING_SERVLET_EVENT_TYPE = "spring-mvc";
    
    private SpringEventType()
    {
        
    }

    /**
     * Retrieves a flag indicating whether a Johnson event should be added when Spring initialisation is bypassed
     * due to previous fatal errors.
     * <p/>
     * By default, an event is <i>not</i> added. If a {@code context-param} named {@link #ADD_EVENT_ON_BYPASS_PARAM}
     * exists with the value {@code true}, then an {@link #getContextEventType(javax.servlet.ServletContext) event}
     * will be added when initialisation is bypassed.
     * <p/>
     * To set this value, add the following to {@code web.xml}:
     * <pre><code>
     *     &lt;context-param&gt;
     *         &lt;param-name&gt;johnson.spring.addEventOnBypass&lt;/param-name&gt;
     *         &lt;param-value&gt;true&lt;/param-value&gt;
     *     &lt;/context-param&gt;
     * </code></pre>
     * Note: If initialisation is not bypassed and fails, this flag <i>does not</i> control whether an event will be
     * added at that time.
     * 
     * @param context the servlet context
     * @return {@code true} if an event has been explicitly requested; otherwise, {@code false}
     */
    public static boolean addEventOnBypass(ServletContext context)
    {
        return "true".equals(context.getInitParameter(ADD_EVENT_ON_BYPASS_PARAM));
    }

    /**
     * Retrieves a flag indicating whether a Johnson event should be added when SpringMVC initialisation is bypassed
     * due to previous fatal Spring errors.
     * <p/>
     * By default, an event is <i>not</i> added. If an {@code init-param} named {@link #ADD_EVENT_ON_BYPASS_PARAM}
     * exists, its value ({@code true} or {@code false}) controls whether an event is added. Otherwise, a fallback check
     * is made {@link #addEventOnBypass(javax.servlet.ServletContext) to the context} for a {@code context-param}.
     * This means if an event is explicitly requested at the context level, by default it will also be requested at
     * the servlet level. However, individual servlets can explicitly disable that by setting their {@code init-param}
     * to {@code false}.
     * <p/>
     * To set this value, add the following to the declaration for the servlet in {@code web.xml}:
     * <pre><code>
     *     &lt;init-param&gt;
     *         &lt;param-name&gt;johnson.spring.addEventOnBypass&lt;/param-name&gt;
     *         &lt;param-value&gt;true&lt;/param-value&gt;
     *     &lt;/init-param&gt;
     * </code></pre>
     * Note: If initialisation is not bypassed and fails, this flag <i>does not</i> control whether an event will be
     * added at that time.
     *
     * @param config the servlet configuration
     * @return {@code true} if an event has been specifically requested, either at the servlet level or at the context
     *         level; otherwise, {@code false}
     */
    public static boolean addEventOnBypass(ServletConfig config)
    {
        String value = config.getInitParameter(ADD_EVENT_ON_BYPASS_PARAM);
        if (value == null)
        {
            //If parameter was found at the servlet level, look at the context level.
            return addEventOnBypass(config.getServletContext());
        }
        //If any value is found at the servlet level, be it true or false, that value always overrides any value set
        //at the context level.
        return "true".equals(value);
    }

    /**
     * Examines the provided {@code ServletContext} for a {@code context-param} named {@link #EVENT_TYPE_PARAM} and, if
     * one is found, returns its value; otherwise the default {@link #SPRING_CONTEXT_EVENT_TYPE} is returned.
     * <p/>
     * To set this value, add the following to {@code web.xml}:
     * <pre><code>
     *     &lt;context-param&gt;
     *         &lt;param-name&gt;johnson.spring.eventType&lt;/param-name&gt;
     *         &lt;param-value&gt;my-spring-context-event-type&lt;/param-value&gt;
     *     &lt;/context-param&gt;
     * </code></pre>
     * 
     * @param context the servlet context
     * @return the context event type
     */
    public static String getContextEventType(ServletContext context)
    {
        String value = context.getInitParameter(EVENT_TYPE_PARAM);
        if (!StringUtils.hasText(value))
        {
            value = SPRING_CONTEXT_EVENT_TYPE;
        }
        return value;
    }

    /**
     * Examines the provided {@code ServletConfig} for an {@code init-param} named {@link #EVENT_TYPE_PARAM} and, if
     * one is found, returns its value; otherwise, the default {@link #SPRING_SERVLET_EVENT_TYPE} is returned.
     * <p/>
     * To set this value, add the following to the declaration for the servlet in {@code web.xml}:
     * <pre><code>
     *     &lt;init-param&gt;
     *         &lt;param-name&gt;johnson.spring.eventType&lt;/param-name&gt;
     *         &lt;param-value&gt;my-spring-servlet-event-type&lt;/param-value&gt;
     *     &lt;/init-param&gt;
     * </code></pre>
     *
     * @param config the servlet configuration
     * @return the servlet event type
     */
    public static String getServletEventType(ServletConfig config)
    {
        String value = config.getInitParameter(EVENT_TYPE_PARAM);
        if (!StringUtils.hasText(value))
        {
            value = SPRING_SERVLET_EVENT_TYPE;
        }
        return value;
    }
}
