AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView", {
    /**
     * Create a `WorkflowModel` seeded with test data.
     *
     * @returns {JIRA.WorkflowDesigner.WorkflowModel} The model.
     */
    createWorkflowModel: function () {
        var workflowModel;

        workflowModel = new JIRA.WorkflowDesigner.WorkflowModel();
        workflowModel.get("statuses").add([{
            name: "Open",
            stepId: 1
        }, {
            name: "In Progress",
            stepId: 2
        }, {
            name: "Closed",
            stepId: 3
        }]);

        return workflowModel;
    },

    setup: function () {
        var sandbox = this.sandbox = sinon.sandbox.create();

        this.getScreensDeferred = jQuery.Deferred().resolve([]);
        this.getScreensStub = sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.ScreensAJAXManager, "getScreens").returns(this.getScreensDeferred);
        this.newTransitionFormViewSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "NewTransitionFormView");
        this.newTransitionFormViewSubmitDeferred = jQuery.Deferred();
        this.newTransitionFormViewSubmitStub = this.sandbox.stub(JIRA.WorkflowDesigner.NewTransitionFormView.prototype, "submit").returns(this.newTransitionFormViewSubmitDeferred);
        this.reuseTransitionFormViewSpy = sandbox.spy(JIRA.WorkflowDesigner, "ReuseTransitionFormView");
        this.reuseTransitionFormViewSubmitDeferred = jQuery.Deferred();
        this.reuseTransitionFormViewSubmitStub = this.sandbox.stub(JIRA.WorkflowDesigner.ReuseTransitionFormView.prototype, "submit").returns(this.reuseTransitionFormViewSubmitDeferred);
        this.showErrorMessageSpy = this.sandbox.spy(JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView.prototype, "showErrorMessage");
    },

    /**
     * Create and show an `AddTransitionDialogView`.
     *
     * @param {object} [options] Options to pass to the constructor.
     * @returns {JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView} The view.
     */
    showDialog: function (options) {
        options = _.defaults({}, options, {
            transitionModel: new JIRA.WorkflowDesigner.TransitionModel(),
            workflowModel: this.createWorkflowModel()
        });

        options.canvasModel || (options.canvasModel = new JIRA.WorkflowDesigner.CanvasModel({}, {
            workflowModel: options.workflowModel
        }));

        return new JIRA.WorkflowDesigner.Dialogs.AddTransitionDialogView(options).show();
    },

    teardown: function () {
        JIRA.WorkflowDesigner.TestUtilities.removeDialogs();
        this.sandbox.restore();
    }
});

test("Automatically selects the transition's source and target statuses", function () {
    var newTransitionFormView, reuseTransitionFormView, statuses, workflowModel;

    workflowModel = this.createWorkflowModel();
    statuses = workflowModel.get("statuses");

    this.showDialog({
        transitionModel: new JIRA.WorkflowDesigner.TransitionModel({
            source: statuses.at(1),
            target: statuses.at(2)
        })
    });

    newTransitionFormView = this.newTransitionFormViewSpy.thisValues[0];
    equal(newTransitionFormView.ui.sourceStepId.val(), 2, "The transition's source is selected");
    equal(newTransitionFormView.ui.targetStepId.val(), 3, "The transition's target is selected");

    reuseTransitionFormView = this.reuseTransitionFormViewSpy.thisValues[0];
    equal(reuseTransitionFormView.ui.sourceStepId.val(), 2, "The transition's source is selected");
    equal(reuseTransitionFormView.ui.targetStepId.val(), 3, "The transition's target is selected");
});

test("Creates and shows a NewTransitionFormView and a ReuseTransitionFormView", function () {
    var dialog = this.showDialog(),
        expectedNewOptions,
        expectedReuseOptions,
        expectedStatusOptions;

    expectedStatusOptions = [{
        selected: false,
        text: "Closed",
        value: 3
    }, {
        selected: false,
        text: "In Progress",
        value: 2
    }, {
        selected: false,
        text: "Open",
        value: 1
    }];

    expectedNewOptions = {
        model: dialog.options.transitionModel,
        screens: [],
        sourceStatuses: expectedStatusOptions,
        targetStatuses: expectedStatusOptions,
        workflowModel: dialog.options.workflowModel
    };

    ok(this.newTransitionFormViewSpy.calledWithExactly(expectedNewOptions), "A NewTransitionFormView was created");
    ok(dialog.newTransitionPane.currentView === this.newTransitionFormViewSpy.thisValues[0],
            "It was shown in the newTransitionPane region");

    expectedReuseOptions = _.omit(expectedNewOptions, "screens");
    ok(this.reuseTransitionFormViewSpy.calledWithExactly(expectedReuseOptions), "A ReuseTransitionFormView was created");
    ok(dialog.reuseTransitionPane.currentView === this.reuseTransitionFormViewSpy.thisValues[0],
            "It was shown in the reuseTransitionPane region");
});

test("Doesn't show the initial status in the status fields", function () {
    var workflowModel;

    workflowModel = this.createWorkflowModel();
    workflowModel.get("statuses").reset([{initial: true}]);

    this.showDialog({workflowModel: workflowModel});
    deepEqual(this.newTransitionFormViewSpy.args[0][0].sourceStatuses, [], "No source statuses were passed to NewTransitionFormView");
    deepEqual(this.newTransitionFormViewSpy.args[0][0].targetStatuses, [], "No target statuses were passed to NewTransitionFormView");
    deepEqual(this.reuseTransitionFormViewSpy.args[0][0].sourceStatuses, [], "No source statuses were passed to ReuseTransitionFormView");
    deepEqual(this.reuseTransitionFormViewSpy.args[0][0].targetStatuses, [], "No target statuses were passed to ReuseTransitionFormView");
});

test("Failing to create a new transition", function () {
    var dialog;

    dialog = this.showDialog();
    dialog.$("form").submit();
    this.newTransitionFormViewSubmitDeferred.reject("Nope");

    equal(dialog.$(":input:disabled").length, 0, "All inputs are enabled");
    ok(this.showErrorMessageSpy.calledWithExactly("Nope"), "The correct error message was shown");
});

test("Failing to reuse a transition", function () {
    var dialog;

    dialog = this.showDialog();
    AJS.tabs.change(dialog.$(".aui-tabs a:last"));
    dialog.$("form").submit();
    this.reuseTransitionFormViewSubmitDeferred.reject("Nope");

    equal(dialog.$(":input:disabled").length, 0, "All inputs are enabled");
    ok(this.showErrorMessageSpy.calledWithExactly("Nope"), "The correct error message was shown");
});

test("Hides and shows an error message when loading screen information fails", function () {
    var dialog, getScreensDeferred, showErrorMessageStub;

    getScreensDeferred = jQuery.Deferred();
    showErrorMessageStub = this.sandbox.stub(JIRA.WorkflowDesigner.Messages, "showErrorMessage");
    this.getScreensStub.returns(getScreensDeferred);

    dialog = this.showDialog();
    getScreensDeferred.reject("Error");

    ok(dialog.$el.is(":hidden"), "The dialog isn't visible");
    ok(showErrorMessageStub.calledWithExactly("Error"),
            "showErrorMessage was called with the correct error message");
});

test("Repositions on tab change", function () {
    var dialog, repositionSpy;

    dialog = this.showDialog();
    repositionSpy = this.sandbox.spy(JIRA.WorkflowDesigner.Dialogs.FormDialogView.prototype, "reposition");

    dialog.$(".aui-tabs a:last").click();
    equal(repositionSpy.callCount, 1, "FormDialogView#reposition was called");
});

test("Requests screen information on show", function () {
    var dialog, getScreensDeferred;

    getScreensDeferred = jQuery.Deferred();
    this.getScreensStub.returns(getScreensDeferred);

    dialog = this.showDialog();
    ok(dialog.$el.is(":hidden"), "The dialog isn't visible");

    getScreensDeferred.resolve([]);
    ok(dialog.$el.is(":visible"), "The dialog is visible");
});

test("Shows the new transition pane when ReuseTransitionFormView triggers a createNewTransition event", function () {
    var dialog, newTransitionLink, reuseTransitionFormView, tabsChangeSpy;

    dialog = this.showDialog();
    newTransitionLink = dialog.$(".aui-tabs .new-transition > a");
    tabsChangeSpy = this.sandbox.spy(AJS.tabs, "change");

    reuseTransitionFormView = this.reuseTransitionFormViewSpy.thisValues[0];
    reuseTransitionFormView.trigger("createNewTransition");
    ok(tabsChangeSpy.calledWithExactly(newTransitionLink),
            "The new transition pane was shown");
});

test("Successfully creates a new transition", function () {
    var dialog = this.showDialog(),
        transitionModel = dialog.options.transitionModel;

    dialog.$("form").submit();
    equal(dialog.$(":input:disabled").length, dialog.$(":input").length, "All inputs are disabled");
    equal(this.newTransitionFormViewSubmitStub.callCount, 1, "NewTransitionFormView#submit was called");

    this.newTransitionFormViewSubmitDeferred.resolve();
    ok(dialog.options.canvasModel.get("selectedModel") === transitionModel, "The TransitionModel is selected");
    ok(dialog.options.workflowModel.get("transitions").contains(transitionModel),
            "The TransitionModel was added to the WorkflowModel's transitions collection");
    ok(!JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog was hidden");
});

test("Successfully reuses an existing transition", function () {
    var dialog = this.showDialog(),
        transitionModel = dialog.options.transitionModel;

    AJS.tabs.change(dialog.$(".aui-tabs a:last"));
    dialog.$("form").submit();
    equal(dialog.$(":input:disabled").length, dialog.$(":input").length, "All inputs are disabled");
    equal(this.reuseTransitionFormViewSubmitStub.callCount, 1, "ReuseTransitionFormView#submit was called");

    this.reuseTransitionFormViewSubmitDeferred.resolve();
    ok(dialog.options.canvasModel.get("selectedModel") === transitionModel, "The TransitionModel is selected");
    ok(dialog.options.workflowModel.get("transitions").contains(transitionModel),
            "The TransitionModel was added to the WorkflowModel's transitions collection");
    ok(!JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog was hidden");
});