AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView", {
    setup: function () {
        this.sandbox = sinon.sandbox.create();

        this.getScreensStub = this.sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.ScreensAJAXManager, "getScreens");
        this.getScreensStub.returns(jQuery.Deferred().resolve([{
            id: 1,
            name: "Screen"
        }]).promise());

        this.updateTransitionStub = this.sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.WorkflowTransitionsAJAXManager, "updateTransition");
        this.workflowModel = new JIRA.WorkflowDesigner.WorkflowModel({name: "Workflow Name"});
    },

    /**
     * Create and show an <tt>EditTransitionDialogView</tt>.
     *
     * @return {object} The dialog's element and deferred result.
     */
    showDialog: function () {
        this.transitionModel = this.workflowModel.addTransition({
            actionId: "1",
            description: "This is a description.",
            name: "Transition",
            source: this.workflowModel.addStatus({id:100, stepId: 1}),
            target: this.workflowModel.addStatus({id:200, stepId: 2})
        });

        new JIRA.WorkflowDesigner.Dialogs.EditTransitionDialogView({
            transitionModel: this.transitionModel,
            workflowModel: this.workflowModel
        }).show();

        return JIRA.Dialog.current && JIRA.Dialog.current.get$popup();
    },

    teardown: function () {
        JIRA.WorkflowDesigner.TestUtilities.removeDialogs();
        this.sandbox.restore();
    }
});

test("An error message is shown if editing the transition fails", function () {
    var element = this.showDialog(),
        errorElement,
        errorMessage = "Oh noes!";

    this.updateTransitionStub.returns(jQuery.Deferred().reject(errorMessage).promise());
    JIRA.WorkflowDesigner.TestUtilities.submit(element);

    errorElement = element.find(".aui-message.error");
    equal(errorElement.length, 1, "An error message is shown in the dialog");
    ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
    equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

    this.updateTransitionStub.returns(jQuery.Deferred().promise());
    JIRA.WorkflowDesigner.TestUtilities.submit(element);
    equal(element.find(".error").length, 0, "Errors are removed on submit");
});

test("Doesn't contain source or target fields", function () {
    var element = this.showDialog();

    equal(element.find("#transition-source-step-id").length, 0, "The dialog doesn't contain a source field");
    equal(element.find("#transition-target-step-id").length, 0, "The dialog doesn't contain a target field");
});

test("Editing a transition", function () {
    var attributes,
        deferred = jQuery.Deferred(),
        element = this.showDialog(),
        expectedArgumentsToAjaxManager,
        expectedAttributes;

    this.updateTransitionStub.returns(deferred.promise());
    this.sandbox.stub(JIRA.WorkflowDesigner.Analytics, "triggerUpdateTransition");

    JIRA.WorkflowDesigner.TestUtilities.submit(element, {
        "#transition-description": "Awesome new description!",
        "#transition-name": "New Name"
    });

    equal(element.find(":input:disabled").length, element.find(":input").length, "All inputs are disabled while submitting");
    equal(this.updateTransitionStub.callCount, 1, "TransitionsAJAXManager.updateTransition() was called");

    deferred.resolve(100);

    attributes = this.workflowModel.get("transitions").at(0).attributes;
    expectedAttributes = {
        description: "Awesome new description!",
        name: "New Name"
    };
    deepEqual(_.pick(attributes, _.keys(expectedAttributes)), expectedAttributes,
        "The TransitionModel's attributes were updated");

    expectedArgumentsToAjaxManager = {
        description: "Awesome new description!",
        name: "New Name",
        screenId: 1,
        sourceStepId: 1,
        transitionId: "1",
        workflowName: "Workflow Name"
    };

    deepEqual(this.updateTransitionStub.args[0][0], expectedArgumentsToAjaxManager,
        "The correct arguments were passed to the AJAX Manager");

    ok(JIRA.WorkflowDesigner.Analytics.triggerUpdateTransition.calledOnce, "Analytics event for update/transition has been triggered");
});

test("Fields are populated with the transition's attributes", function () {
    var element = this.showDialog();

    equal(element.find("#transition-description").val(), this.transitionModel.get("description"), "The transition's description was used");
    equal(element.find("#transition-name").val(), this.transitionModel.get("name"), "The transition's name was used");
});