AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.Dialogs.DeleteStatusDialogView", {
    setup: function () {
        this.sandbox = sinon.sandbox.create();

        this.removeStatusStub = this.sandbox.stub(JIRA.WorkflowDesigner.IO.AJAX.WorkflowStatusesAJAXManager, "removeStatus");
        this.workflowModel = new JIRA.WorkflowDesigner.WorkflowModel({name: "Workflow Name", liveStepIds: [1]});

        this.statusModel = this.workflowModel.addStatus({ stepId: 1, statusId: 100 });
        this.transitionModel = this.workflowModel.addTransition({
            source: new JIRA.WorkflowDesigner.StatusModel({ stepId: 2, statusId: 101, initial: true }),
            target: this.statusModel
        });

        this.statusModel.destroy = this.sandbox.spy();
        this.workflowModel.destroy = this.sandbox.spy();
    },

    /**
     * Create and show an <tt>DeleteStatusDialogView</tt>.
     *
     * @return {object} The dialog's element and deferred result.
     */
    showDialog: function () {
        new JIRA.WorkflowDesigner.Dialogs.DeleteStatusDialogView({
            workflowModel: this.workflowModel,
            statusModel: this.statusModel
        }).show();

        return JIRA.Dialog.current && JIRA.Dialog.current.get$popup();
    },

    teardown: function () {
        JIRA.WorkflowDesigner.TestUtilities.removeDialogs();
        this.sandbox.restore();
    }
});

test("Deleting a status reloads the new workflow layout", function () {
    var element = this.showDialog(),
        expectedArguments,
        layoutData = {},
        workflowModelResetStub = this.sandbox.stub(this.workflowModel, "reset");

    ok(JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog is visible");

    this.removeStatusStub.returns(jQuery.Deferred().resolve(layoutData).promise());
    this.sandbox.stub(JIRA.WorkflowDesigner.Analytics, "triggerRemoveStep");

    JIRA.WorkflowDesigner.TestUtilities.submit(element);

    expectedArguments = [{
        statusId: 100,
        workflowName: "Workflow Name"
    }];

    equal(this.removeStatusStub.callCount, 1, "WorkflowStatusesAJAXManager.removeStatus() was called");
    deepEqual(this.removeStatusStub.args[0], expectedArguments, "It was passed the correct arguments");

    ok(!JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "No dialog is visible");
    equal(workflowModelResetStub.callCount, 1, "WorkflowModel.reset() was called");
    deepEqual(workflowModelResetStub.args[0], [layoutData], "It was passed the layout data returned from removeStatus()");

    ok(JIRA.WorkflowDesigner.Analytics.triggerRemoveStep.calledOnce, "Analytics event for remove/step has been triggered");
});

test("An error message is shown if deleting the status fails", function () {
    var element = this.showDialog(),
        errorElement,
        errorMessage = "Oh noes!";

    this.removeStatusStub.returns(jQuery.Deferred().reject(errorMessage).promise());
    JIRA.WorkflowDesigner.TestUtilities.submit(element);

    errorElement = element.find(".aui-message.error");
    equal(errorElement.length, 1, "An error message is shown in the dialog");
    ok(errorElement.text().indexOf(errorMessage) > -1, "It contains the correct text");
    equal(element.find(":input:disabled").length, 0, "No elements in the dialog are disabled");

    this.removeStatusStub.returns(jQuery.Deferred().promise());
    JIRA.WorkflowDesigner.TestUtilities.submit(element);
    equal(element.find(".error").length, 0, "Errors are removed on submit");
});

test("Closing the dialog doesn't destroy workflow model", function () {
    var element = this.showDialog();

    ok(JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog is visible");

    element.find(".cancel").click();

    equal(this.workflowModel.destroy.callCount, 0, "Workflow model was not destroyed");
    equal(this.statusModel.destroy.callCount, 0, "Status model was not destroyed");
    ok(!JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog closed");
});

test("Can't delete initial status", function () {
    this.statusModel.set("initial", true);

    var element = this.showDialog();

    ok(JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog is visible");

    ok(!element.find(":submit").length, "The dialog does not have a submit button");
});

test("Can't delete status connected to the initial transition", function () {
    this.transitionModel.get("source").set("initial", true);

    var element = this.showDialog();

    ok(JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog is visible");

    ok(!element.find(":submit").length, "The dialog does not have a submit button");
});

test("Can't delete a non-deletable status", function () {
    this.statusModel = this.workflowModel.addStatus({stepId: 1});
    this.workflowModel.statusIsDeletable = sinon.stub().returns(false);

    var element = this.showDialog();

    ok(JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog is visible");
    ok(!element.find(":submit").length, "The dialog does not have a submit button");
});

test("Can delete a deletable status", function () {
    this.statusModel = this.workflowModel.addStatus({stepId: 1});
    this.workflowModel.statusIsDeletable = sinon.stub().returns(true);

    var element = this.showDialog();

    ok(JIRA.WorkflowDesigner.TestUtilities.dialogIsVisible(), "The dialog is visible");
    ok(element.find(":submit").length, "The dialog has a submit button");
});