AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.CanvasView", {
    setup: function () {
        this.sandbox = sinon.sandbox.create();
        this.workflowModel = new JIRA.WorkflowDesigner.WorkflowModel();
        this.canvasModel = new JIRA.WorkflowDesigner.CanvasModel({}, {
            workflowModel: this.workflowModel
        });
        this.canvasView = JIRA.WorkflowDesigner.TestUtilities.testCanvasView({
            canvasModel: this.canvasModel,
            workflowModel: this.workflowModel
        });
        this.canvas = this.canvasView.canvas;
    },

    teardown: function () {
        JIRA.WorkflowDesigner.TestUtilities.removeDialogs();
        this.canvasView.remove();
        this.sandbox.restore();
    },

    triggerKeyDown: function (keyCode, properties) {
        properties = _.extend({which: keyCode}, properties);
        jQuery(document).trigger(jQuery.Event("keydown", properties));
    }
});

test("Adding a connection displays an AddTransitionDialogView", function () {
    var connection = new draw2d.Connection(),
        dialog = {show: sinon.spy()},
        dialogStub = this.sandbox.stub(JIRA.WorkflowDesigner.Dialogs, "AddTransitionDialogView").returns(dialog),
        expectedArguments,
        transition;

    transition = new JIRA.WorkflowDesigner.TransitionModel({
        source: this.workflowModel.addStatus(),
        target: this.workflowModel.addStatus()
    });

    connection.setSource(this.canvasView.statusViews.at(0).getPorts()[0]);
    connection.setTarget(this.canvasView.statusViews.at(1).getPorts()[0]);

    sinon.spy(this.canvas, "removeFigure");
    this.sandbox.stub(JIRA.WorkflowDesigner, "TransitionModel").returns(transition);
    this.canvasView.canvas.onNewConnection(connection);

    expectedArguments = [{
        transitionModel: transition,
        workflowModel: this.workflowModel
    }];

    equal(dialogStub.callCount, 1, "An AddTransitionDialogView was created");
    ok(dialogStub.args[0][0].canvasModel === this.canvasModel, "It was passed the correct canvas model");
    ok(dialogStub.args[0][0].transitionModel === transition, "It was passed the correct transition model");
    ok(dialogStub.args[0][0].workflowModel === this.workflowModel, "It was passed the correct workflow model");
    equal(dialog.show.callCount, 1, "It was shown");
    ok(this.canvas.removeFigure.calledWith(connection), "The connection was removed from the canvas");
    equal(this.workflowModel.get("transitions").length, 0, "No TransitionModel was added to the WorkflowModel");
});

test("addStatus()", function () {
    var initialStatusViewSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "InitialStatusView"),
        statusModel = new JIRA.WorkflowDesigner.StatusModel(),
        statusViewSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "StatusView");

    this.canvasView.addStatus(statusModel);
    equal(initialStatusViewSpy.callCount, 0, "No InitialStatusView was created for a normal status");
    equal(statusViewSpy.callCount, 1, "A StatusView was created for a normal status");
    equal(statusViewSpy.args[0][0].model, statusModel, "It is associated with the given model");

    statusModel.set("initial", true);
    this.canvasView.addStatus(statusModel);
    equal(initialStatusViewSpy.callCount, 1, "An InitialStatusView was created for an initial status");
    equal(initialStatusViewSpy.args[0][0].model, statusModel, "It is associated with the given model");
    equal(statusViewSpy.callCount, 1, "No StatusView was created for an initial status");
});

test("addStatus() selects the new status if necessary", function () {
    var selectSpy = this.sandbox.spy(JIRA.WorkflowDesigner.StatusView.prototype, "select"),
        status;

    status = new JIRA.WorkflowDesigner.StatusModel();
    this.canvasView.addStatus(status);
    equal(selectSpy.callCount, 0, "The status wasn't selected");

    status = new JIRA.WorkflowDesigner.StatusModel();
    this.canvasModel.set("selectedModel", status);
    this.canvasView.addStatus(status);
    equal(selectSpy.callCount, 1, "The status was selected");
});

test("addTransition()", function () {
    var statusModel = this.workflowModel.addStatus({}),
        transitionModel,
        transitionViewSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "TransitionView");

    transitionModel = this.workflowModel.addTransition({
        name: "Transition",
        source: statusModel,
        target: statusModel
    });

    this.canvasView.addTransition(transitionModel);
    equal(transitionViewSpy.callCount, 1, "A new TransitionView was created");
    equal(transitionViewSpy.args[0][0].model, transitionModel, "It is associated with the given model");

    this.canvasView.addTransition(transitionModel);
    equal(transitionViewSpy.callCount, 1, "Attempting to add duplicate transitions does nothing");
});

test("addTransition() selects the new transition if necessary", function () {
    var selectSpy = this.sandbox.spy(JIRA.WorkflowDesigner.TransitionView.prototype, "select"),
        transition;

    transition = new JIRA.WorkflowDesigner.TransitionModel({
        source: this.workflowModel.addStatus({}),
        target: this.workflowModel.addStatus({})
    });

    this.canvasView.addTransition(transition);
    equal(selectSpy.callCount, 0, "The transition wasn't selected");

    transition = new JIRA.WorkflowDesigner.TransitionModel({
        source: this.workflowModel.addStatus({}),
        target: this.workflowModel.addStatus({})
    });

    this.canvasModel.set("selectedModel", transition);
    this.canvasView.addTransition(transition);
    equal(selectSpy.callCount, 1, "The transition was selected");
});

test("getCanvasBoundingBox() doesn't include LayerRootFigures", function () {
    var expected;

    this.canvasView.canvas.getFigures = function () {
        return new draw2d.util.ArrayList([
            new JIRA.WorkflowDesigner.Draw2DCanvas.LayerRootFigure(),
            JIRA.WorkflowDesigner.TestMocks.figure([10, 10, 10, 10])
        ]);
    };

    expected = new draw2d.geo.Rectangle(10, 10, 10, 10);
    ok(expected.equals(this.canvasView.getCanvasBoundingBox()),
            "The bounding box was calculated correctly");
});

test("getCanvasBoundingBox() includes relevant figures and lines", function () {
    var expected;

    this.canvasView.canvas.getFigures = function () {
        return new draw2d.util.ArrayList(_.map([
            [-10, -10, 5, 5],
            [  0,   0, 5, 5],
            [ 10,  10, 5, 5]
        ], JIRA.WorkflowDesigner.TestMocks.figure));
    };

    this.canvasView.canvas.getLines = function () {
        return new draw2d.util.ArrayList(_.map([
            [-8, -15, 13, 18],
            [-3,   3, 13, 18],
            [-5,  -8, 23, 13]
        ], JIRA.WorkflowDesigner.TestMocks.figure));
    };

    expected = new draw2d.geo.Rectangle(-10, -15, 28, 36);
    ok(expected.equals(this.canvasView.getCanvasBoundingBox()), "The bounding box was calculated correctly");
});

test("WorkflowModel StatusCollection observation", function () {
    var removeStatusSpy = this.sandbox.spy(this.canvasView, "removeStatus"),
        statusModel,
        statusViewSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "StatusView");

    statusModel = this.workflowModel.addStatus({});
    equal(statusViewSpy.callCount, 1, "Adding a StatusModel causes a StatusView to be created");
    equal(statusViewSpy.args[0][0].model, statusModel, "It is associated with the new StatusModel");

    this.workflowModel.reset({
        statuses: [],
        transitions: []
    });

    ok(removeStatusSpy.callCount === 1 && removeStatusSpy.args[0][0] === statusModel,
            "Removing a StatusModel causes its StatusView to be removed");
});

test("WorkflowModel TransitionCollection observation", function () {
    var removeTransitionSpy = this.sandbox.spy(this.canvasView, "removeTransition"),
        statusModel,
        transitionModel,
        transitionViewSpy = this.sandbox.spy(JIRA.WorkflowDesigner, "TransitionView");

    statusModel = this.workflowModel.addStatus({});
    transitionModel = this.workflowModel.addTransition({
        name: "Transition",
        source: statusModel,
        target: statusModel
    });

    equal(transitionViewSpy.callCount, 1, "Adding a TransitionModel causes a TransitionView to be created");
    equal(transitionViewSpy.args[0][0].model, transitionModel, "It is associated with the new TransitionModel");

    this.workflowModel.reset({
        statuses: [],
        transitions: []
    });

    ok(removeTransitionSpy.callCount === 1 && removeTransitionSpy.args[0][0] === transitionModel,
            "Removing a TransitionModel causes its TransitionView to be removed");
});

test("Transition reconnection updates the TransitionModel if target status was not changed", function () {
    var closed = this.workflowModel.addStatus({x: 0, y: 100}),
        closedView = this.canvasView.addStatus(closed),
        expectedAttributes,
        open = this.workflowModel.addStatus(),
        openView = this.canvasView.addStatus(open),
        transition,
        transitionView;

    transition = this.workflowModel.addTransition({
        name: "Transition",
        source: open,
        sourceAngle: 90,
        target: closed,
        targetAngle: -90
    });

    transitionView = this.canvasView.addTransition(transition);
    transitionView._connection.setSource(openView.getPortForAngle(0));
    transitionView._connection.setTarget(closedView.getPortForAngle(0));
    transitionView.trigger("reconnect", transitionView);

    expectedAttributes = {
        source: open,
        sourceAngle: openView.getAngleToPort(transitionView._connection.getSource()),
        target: closed,
        targetAngle: closedView.getAngleToPort(transitionView._connection.getTarget())
    };

    deepEqual(_.pick(transition.attributes, "source", "sourceAngle", "target", "targetAngle"), expectedAttributes,
            "The TransitionModel's sourceAngle and targetAngle attributes were updated");
});

test("Transition reconnection shows EditTransitionTargetDialogView if target status was changed", function () {
    var closed = this.workflowModel.addStatus({x: 0, y: 100}),
        expectedAttributes,
        editTransitionTargetDialogViewStub = this.sandbox.stub(JIRA.WorkflowDesigner.Dialogs, "EditTransitionTargetDialogView"),
        open = this.workflowModel.addStatus(),
        openView = this.canvasView.addStatus(open),
        reopened = this.workflowModel.addStatus(),
        reopenedView = this.canvasView.addStatus(reopened),
        transition,
        transitionView;

    editTransitionTargetDialogViewStub.returns({show: jQuery.noop});

    transition = this.workflowModel.addTransition({
        name: "Transition",
        source: open,
        sourceAngle: 90,
        target: closed,
        targetAngle: -90
    });

    transitionView = this.canvasView.addTransition(transition);
    transitionView._connection.setSource(openView.getPortForAngle(0));
    transitionView._connection.setTarget(reopenedView.getPortForAngle(0));
    transitionView.trigger("reconnect", transitionView);

    expectedAttributes = {
        targetPort: reopenedView.getPortForAngle(0),
        targetView: reopenedView,
        transitionView: transitionView,
        workflowModel: this.workflowModel
    };

    equal(editTransitionTargetDialogViewStub.callCount, 1, "An EditTransitionTargetDialogView was created");
    ok(_.isEqual(editTransitionTargetDialogViewStub.args[0][0], expectedAttributes), "It was passed the correct arguments");
});

test("Transition reconnection shows EditTransitionSourceDialogView if source status was changed", function () {
    var closed = this.workflowModel.addStatus({x: 0, y: 100}),
        expectedAttributes,
        editTransitionSourceDialogViewStub = this.sandbox.stub(JIRA.WorkflowDesigner.Dialogs, "EditTransitionSourceDialogView"),
        open = this.workflowModel.addStatus(),
        reopened = this.workflowModel.addStatus(),
        reopenedView = this.canvasView.addStatus(reopened),
        transition,
        transitionView;

    editTransitionSourceDialogViewStub.returns({show: jQuery.noop});

    transition = this.workflowModel.addTransition({
        name: "Transition",
        source: open,
        sourceAngle: 90,
        target: closed,
        targetAngle: -90
    });

    transitionView = this.canvasView.addTransition(transition);
    transitionView._connection.setSource(reopenedView.getPortForAngle(0));
    transitionView.trigger("reconnect", transitionView);

    expectedAttributes = {
        newSourcePort: reopenedView.getPortForAngle(0),
        newSourceView: reopenedView,
        originalSourceStatus: open,
        transitionView: transitionView,
        workflowModel: this.workflowModel
    };

    equal(editTransitionSourceDialogViewStub.callCount, 1, "An EditTransitionSourceDialogView was created");
    ok(_.isEqual(editTransitionSourceDialogViewStub.args[0][0], expectedAttributes), "It was passed the correct arguments");
});

test("Statuses without coordinates are positioned automatically", function () {
    var layoutChangedSpy = sinon.spy(),
        statusModelOne = new JIRA.WorkflowDesigner.StatusModel({x: 10, y: 10}),
        statusModelTwo = new JIRA.WorkflowDesigner.StatusModel({x: 200, y: 200}),
        statusModelThree = new JIRA.WorkflowDesigner.StatusModel(),
        statusModelFour = new JIRA.WorkflowDesigner.StatusModel();

    this.workflowModel.on("layoutChanged", layoutChangedSpy);
    this.workflowModel.reset({
        statuses: [statusModelOne, statusModelTwo, statusModelThree, statusModelFour],
        transitions: []
    });

    ok(_.isNumber(statusModelThree.get("x")), "The first model has an x coordinate");
    ok(_.isNumber(statusModelThree.get("y")), "The first model has a y coordinate");
    ok(_.isNumber(statusModelFour.get("x")), "The second model has an x coordinate");
    ok(_.isNumber(statusModelFour.get("y")), "The second model has a y coordinate");
    ok(layoutChangedSpy.called, "Triggered a layoutChanged event");
});

test("Autocalculated transition ports must trigger autosaving", function () {
    var layoutChangedSpy = sinon.spy(),
        source = new JIRA.WorkflowDesigner.StatusModel({x: 10, y:10}),
        target = new JIRA.WorkflowDesigner.StatusModel({x: 200, y: 200}),
        transitionModel;

    transitionModel = new JIRA.WorkflowDesigner.TransitionModel({
        name: "Transition",
        source: source,
        target: target,
        ports: null
    });

    this.workflowModel.on("layoutChanged", layoutChangedSpy);
    this.workflowModel.reset({
        statuses: [source, target],
        transitions: [transitionModel]
    });

    ok(layoutChangedSpy.called, "Triggered a layoutChanged event");
});

test("Zooming with the mouse wheel hides the current inline dialog", function () {
    this.sandbox.stub(AJS, "InlineDialog");
    AJS.InlineDialog.current = {hide: sinon.spy()};

    sinon.stub(this.canvasView, "getCanvasBoundingBox").returns(new draw2d.geo.Rectangle(0, 0, 0, 0));
    this.canvasView._zoomHandler.trigger('zoom', {
        clientX: 10,
        clientY: 10,
        factor: 1.05
    });

    equal(AJS.InlineDialog.current.hide.callCount, 1, "The current inline dialog was hidden");
});

test("The canvas is resized after its container is resized", function () {
    JIRA.WorkflowDesigner.TestUtilities.fakeTimer(function (clock) {
        var canvasView = JIRA.WorkflowDesigner.TestUtilities.testCanvasView({model: this.workflowModel});

        var originalContainerWidth = canvasView.$el.width();
        var originalContainerHeight = canvasView.$el.height();

        clock.tick(150);
        equal(originalContainerWidth, canvasView.canvas.svgElement.width());
        equal(originalContainerHeight, canvasView.canvas.svgElement.height());

        var assertDimensions = function () {
            equal(canvasView.$el.width(), canvasView.canvas.svgElement.width());
            equal(canvasView.$el.height(), canvasView.canvas.svgElement.height());
        };

        canvasView.$el.width(originalContainerWidth * 0.5);
        canvasView.$el.height(originalContainerHeight * 0.5);
        clock.tick(150);
        assertDimensions();

        canvasView.$el.width(originalContainerWidth);
        canvasView.$el.height(originalContainerHeight);
        clock.tick(150);
        assertDimensions();

        canvasView.remove();
    }, this);
});

test("Deleting the selected transition", function () {
    var destroyStub,
        keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
        transitionView;

    this.workflowModel.addTransition({
        name: "Transition",
        source: this.workflowModel.addStatus({}),
        target: this.workflowModel.addStatus({})
    });

    transitionView = this.canvasView.transitionViews.at(0);
    this.canvas.selectFigure(transitionView._connection);
    destroyStub = this.sandbox.stub(transitionView, "destroy");

    _.each(keyCodes, function (keyCode) {
        this.triggerKeyDown(keyCode);
        equal(destroyStub.callCount, 1, "TransitionView.destroy() was called for key code " + keyCode);
        destroyStub.reset();
    }, this);
});

test("Deleting the selected status", function () {
    var destroyStub,
        keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
        statusView;

    this.workflowModel.addStatus({x: 0, y: 100});
    statusView = this.canvasView.statusViews.at(0);
    this.canvas.selectFigure(statusView._figure);
    destroyStub = this.sandbox.stub(statusView, "destroy");

    _.each(keyCodes, function (keyCode) {
        this.triggerKeyDown(keyCode);
        equal(destroyStub.callCount, 1, "StatusView.destroy() was called for key code " + keyCode);
        destroyStub.reset();
    }, this);
});

test("Deletion is disabled when a dialog is visible", function () {
    var destroyStub,
        keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
        statusModel,
        statusView,
        transitionModel,
        transitionView;

    // "Show" a dialog
    jQuery("#qunit-fixture").append("<div class=\"aui-blanket\">");

    transitionModel = this.workflowModel.addTransition({
        name: "Transition",
        source: this.workflowModel.addStatus({}),
        target: this.workflowModel.addStatus({})
    });

    transitionView = this.canvasView.addTransition(transitionModel).render();
    this.canvas.selectFigure(transitionView._connection);
    destroyStub = this.sandbox.stub(transitionView, "destroy");

    _.each(keyCodes, function (keyCode) {
        this.triggerKeyDown(keyCode);
        equal(destroyStub.callCount, 0, "TransitionView.destroy() was not called");
    }, this);

    statusModel = this.workflowModel.addStatus({x: 0, y: 100});
    statusView = this.canvasView.addStatus(statusModel).render();
    statusView.select();
    destroyStub = this.sandbox.stub(statusView, "destroy");

    _.each(keyCodes, function (keyCode) {
        this.triggerKeyDown(keyCode);
        equal(destroyStub.callCount, 0, "StatusView.destroy() was not called");
    }, this);
});

test("Deletion is disabled when the view is locked", function () {
    var statusView;

    this.canvasView.setLocked(true);
    this.workflowModel.addStatus({x: 0, y: 0});
    statusView = this.canvasView.statusViews.at(0);
    statusView.select();
    this.sandbox.stub(statusView, "destroy");

    this.triggerKeyDown(jQuery.ui.keyCode.BACKSPACE);
    equal(statusView.destroy.callCount, 0, "StatusView#destroy wasn't called");
});

test("Pressing delete key when target is an input does not display a DeleteTransitionDialogView", function () {
    var destroyStub,
        keyCodes = [jQuery.ui.keyCode.BACKSPACE, jQuery.ui.keyCode.DELETE],
        transitionModel,
        transitionView;

    transitionModel = this.workflowModel.addTransition({
        name: "Transition",
        source: this.workflowModel.addStatus({}),
        target: this.workflowModel.addStatus({})
    });

    transitionView = this.canvasView.addTransition(transitionModel).render();
    this.canvas.selectFigure(transitionView._connection);
    destroyStub = this.sandbox.stub(transitionView, "destroy");

    _.each(keyCodes, function (keyCode) {
        this.triggerKeyDown(keyCode, {target: jQuery("<input>")});
        equal(destroyStub.callCount, 0, "TransitionView.destroy() was not called");
    }, this);
});

test("Auto-fitting fits the SVG element to its container", function () {
    sinon.spy(this.canvasView.canvas, "fitToContainer");

    this.canvasView.autoFit();
    equal(this.canvasView.canvas.fitToContainer.callCount, 1,
            "Canvas#fitToContainer() was called");
});

test("Auto-fitting centers the diagram if it fits into default bounds", function () {
    var setViewBoxSpy = sinon.spy(this.canvasView.canvas, "setViewBox");

    this.canvasView.$el.width(500);
    this.canvasView.$el.height(200);

    this.canvasView.canvas.getFigures = function () {
        return new draw2d.util.ArrayList([JIRA.WorkflowDesigner.TestMocks.figure([-10, -10, 25, 25])]);
    };

    this.canvasView.autoFit();

    ok(setViewBoxSpy.args[0][0].equals(new draw2d.geo.Rectangle(-247.5, -97.5, 500, 200)), "View box dimensions were not changed and the diagram was centered");
});

test("Auto-fitting scales and centers the diagram if it doesn't fit into default bounds", function () {
    var setViewBoxSpy = sinon.spy(this.canvasView.canvas, "setViewBox");

    this.canvasView.$el.width(500);
    this.canvasView.$el.height(200);

    sinon.stub(this.canvasView.canvas, "getViewBox").returns(new draw2d.geo.Rectangle(-20, -20, 620, 350));

    this.canvasView.canvas.getFigures = function () {
        return new draw2d.util.ArrayList([JIRA.WorkflowDesigner.TestMocks.figure([-10, -10, 600, 300])]);
    };

    this.canvasView.autoFit();

    ok(setViewBoxSpy.args[1][0].equals(new draw2d.geo.Rectangle(-20, -35, 620, 350)), "View box was scaled and the diagram was centered");
});

test("Auto-fitting sets the CanvasModel's zoomLevel property", function () {
    // Add a status so the canvas has a bounding box.
    this.canvasView.addStatus(new JIRA.WorkflowDesigner.StatusModel());

    ok(!this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property isn't set");

    this.canvasView.autoFit();
    ok(this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property is set");
    equal(this.canvasModel.get("zoomLevel"), this.canvasView.canvas.getZoom(), "It is set to the canvas zoom level");
});

test("setResizeInterval()", function () {
    var setResizeIntervalSpy = this.sandbox.spy(JIRA.WorkflowDesigner.Draw2DCanvas.prototype, "setResizeInterval");

    this.canvasView.setResizeInterval(42);
    equal(setResizeIntervalSpy.callCount, 1, "JIRA.WorkflowDesigner.Draw2DCanvas#setResizeInterval() was called");
    deepEqual(setResizeIntervalSpy.args[0], [42], "It was passed the correct arguments");
});

test("Selects views in response to the CanvasModel's selectedModel property changing", function () {
    var onSelectSpy = this.sandbox.spy(JIRA.WorkflowDesigner.StatusView.prototype, "_onSelect"),
        statusModel = this.workflowModel.addStatus({});

    this.canvasModel.set("selectedModel", statusModel);
    equal(onSelectSpy.callCount, 1, "The view was selected");
});

test("Updates the CanvasModel in response to view (de)selection", function () {
    var statusModel = this.workflowModel.addStatus({}),
        statusView = this.canvasView.statusViews.at(0);

    statusView.select();
    ok(this.canvasModel.get("selectedModel") === statusModel, "The CanvasModel's selectedModel property was updated");
    ok(this.canvasModel.get("selectedView") === statusView, "The CanvasModel's selectedView property was updated");

    statusView.deselect();
    ok(!this.canvasModel.has("selectedModel"), "The CanvasModel's selectedModel property was cleared");
    ok(!this.canvasModel.has("selectedView"), "The CanvasModel's selectedView property was cleared");
});

test("Source view is updated when model source is updated", function () {
    var newSourceStatusModel, transitionModel, transitionView;

    transitionModel = this.workflowModel.addTransition({
        name: "Transition",
        source: this.workflowModel.addStatus({}),
        target: this.workflowModel.addStatus({})
    });

    transitionView = this.canvasView._getTransitionViewWithModel(transitionModel);
    this.sandbox.spy(transitionView, "resetConnection");

    newSourceStatusModel = this.workflowModel.addStatus({});
    transitionModel.set("source", newSourceStatusModel);

    ok(transitionView._getSourceView() === this.canvasView._getStatusViewWithModel(newSourceStatusModel), "Source view was updated");
    equal(transitionView.resetConnection.callCount, 1, "Connection was reset");
});

test("Target view is updated when model target is updated", function () {
    var newTargetStatusModel, transitionModel, transitionView;

    transitionModel = this.workflowModel.addTransition({
        name: "Transition",
        source: this.workflowModel.addStatus({}),
        target: this.workflowModel.addStatus({})
    });

    transitionView = this.canvasView._getTransitionViewWithModel(transitionModel);
    this.sandbox.spy(transitionView, "resetConnection");

    newTargetStatusModel = this.workflowModel.addStatus({});
    transitionModel.set("target", newTargetStatusModel);

    ok(transitionView._getTargetView() === this.canvasView._getStatusViewWithModel(newTargetStatusModel), "Target view was updated");
    equal(transitionView.resetConnection.callCount, 1, "Connection was reset");
});

test("Zooming sets the CanvasModel's zoomLevel property", function () {
    // Add a status so the canvas has a bounding box.
    this.canvasView.addStatus(new JIRA.WorkflowDesigner.StatusModel());

    ok(!this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property isn't set");

    this.canvasView.zoom(1.0);
    ok(this.canvasModel.has("zoomLevel"), "The CanvasModel's zoomLevel property is set");
    equal(this.canvasModel.get("zoomLevel"), this.canvasView.canvas.getZoom(), "It is set to the canvas zoom level");
});

test("When a transition is (de)selected, its siblings are (de)selected", function () {
    var closed = this.workflowModel.addStatus({id: 1}),
        inProgress = this.workflowModel.addStatus({id: 2}),
        open = this.workflowModel.addStatus({id: 3}),
        otherTransition,
        otherTransitionView,
        siblingTransition,
        siblingTransitionView,
        transition,
        transitionView;

    otherTransition = this.workflowModel.addTransition({
        actionId: "2",
        source: open,
        target: inProgress
    });

    siblingTransition = this.workflowModel.addTransition({
        actionId: "1",
        source: open,
        target: closed
    });

    transition = this.workflowModel.addTransition({
        actionId: "1",
        source: inProgress,
        target: closed
    });

    otherTransitionView = this.canvasView._getTransitionViewWithModel(otherTransition);
    siblingTransitionView = this.canvasView._getTransitionViewWithModel(siblingTransition);
    transitionView = this.canvasView._getTransitionViewWithModel(transition);

    this.sandbox.spy(otherTransitionView, "appearSelected");
    this.sandbox.spy(siblingTransitionView, "appearSelected");
    transitionView.trigger("selected", transitionView);
    equal(otherTransitionView.appearSelected.callCount, 0, "Non-sibling transitions aren't selected");
    equal(siblingTransitionView.appearSelected.callCount, 1, "Sibling transitions are selected");

    this.sandbox.spy(otherTransitionView, "unhighlight");
    this.sandbox.spy(siblingTransitionView, "unhighlight");
    transitionView.trigger("deselected", transitionView);
    equal(otherTransitionView.unhighlight.callCount, 0, "Non-sibling transitions aren't deselected");
    equal(siblingTransitionView.unhighlight.callCount, 1, "Sibling transitions are deselected");
});