(function() {
    AJS.namespace("JIRA.WorkflowDesigner.StatusPort");

    JIRA.WorkflowDesigner.StatusPort = draw2d.HybridPort.extend(
    /** @lends JIRA.WorkflowDesigner.StatusPort# */
    {
        /**
         * Radius of the port
         *
         * @type {number}
         * @constant
         * @default
         */
        RADIUS: 8,

        /**
         * Stroke width of the port
         *
         * @type {number}
         * @constant
         * @default
         */
        STROKE_WIDTH: 1,

        /**
         * Initialise the port.
         *
         * @classdesc A Draw2D port shown on {@link JIRA.WorkflowDesigner.StatusView}s.
         * @constructs
         * @extends draw2d.HybridPort
         * @param {object} options
         * @param {draw2d.Canvas} options.canvas The canvas on which the port will be rendered.
         * @param {JIRA.WorkflowDesigner.Direction} options.connectionDirection The direction in which transitions leave the port.
         * @param {function} [options.isPortDragged] Function returning true if some port is currently being dragged.
         * @param {function} [options.getAllTargetPorts] Function returning all possible target ports.
         */
        init: function (options) {
            this._super();

            this.connectionDirection = options.connectionDirection;
            this.canvas = options.canvas;

            this.isPortDragged = options.isPortDragged;
            this.getAllTargetPorts = options.getAllTargetPorts;

            this.lighterBgColor = this.bgColor.hash();
            this.setBackgroundColor("#ffffff");
            this.setColor("#1a8cff");
            this.setCoronaWidth(6);
            this.setDimension(this.RADIUS, this.RADIUS);
            this.setStroke(this.STROKE_WIDTH);

            this.hide();

            this._resetEditPolicies();
            this.installEditPolicy(new JIRA.WorkflowDesigner.Policy.Feedback.StatusPortsFeedbackPolicy());
        },

        /**
         * Reset the edit policies
         * @private
         */
        _resetEditPolicies: function() {
            var instance = this;
            this.editPolicy.each(function(index, policy) {
                policy.onUninstall(instance);
            });

            this.editPolicy = new draw2d.util.ArrayList();
        },

        /**
         * @returns {JIRA.WorkflowDesigner.Direction}
         */
        getConnectionDirection: function () {
            return this.connectionDirection;
        },

        /**
         * @param {number} x
         * @param {number} y
         * @returns {draw2d.Port|null}
         */
        getDropTarget: function (x, y) {
            return JIRA.WorkflowDesigner.StatusPort.getHitPort(x, y, this.getDropTargets().asArray());
        },

        /**
         * @returns {draw2d.util.ArrayList}
         */
        getDropTargets: function() {
            var ports = _.without(this.getAllTargetPorts(), this);

            return new draw2d.util.ArrayList(ports).trimToSize();
        },

        /**
         * @param {draw2d.Figure} previousHoverFigure
         */
        onMouseEnter: function(previousHoverFigure) {
            this.trigger("mouse:enter", previousHoverFigure);
        },

        /**
         * @param {draw2d.Figure} newHoverFigure
         */
        onMouseLeave: function(newHoverFigure) {
            this.trigger("mouse:leave", newHoverFigure);
        },

        /**
         * @method
         */
        onDragStart: function() {
            this.triggerDragStart();
            this.trigger("select");
            this._super();
        },

        /**
         * @method
         */
        onDragEnd: function() {
            this._super();
            this.triggerDragEnd();
        },

        /**
         * @method
         */
        triggerDragStart: function() {
            this.trigger("drag:start");
        },

        /**
         * @method
         */
        triggerDragEnd: function() {
            this.trigger("drag:end");
        },

        /**
         * @param {number} x
         * @param {number} y
         * @returns {boolean}
         */
        hitTest: function(x, y) {
            // Port drag and drop: consider corona width.
            if (this.isPortDragged()) {
                return this._super(x, y);
            }

            var absoluteX = this.getAbsoluteX();
            var absoluteY = this.getAbsoluteY();

            var xMin = absoluteX - this.getWidth()/2;
            var yMin = absoluteY - this.getHeight()/2;
            var xMax = absoluteX + this.getWidth()/2;
            var yMax = absoluteY + this.getHeight()/2;

            return (x >= xMin && x <= xMax && y >= yMin && y <= yMax);
        },

        /**
         * @method
         */
        show: function() {
            this.sendToFront().setVisible(true);
        },

        /**
         * @method
         */
        hide: function() {
            this.setVisible(false);
        },

        /**
         * @method
         */
        repaint:function() {
            this._super.apply(this, arguments);

            this._setCursor(this);
        },

        /**
         * Highlight this port with a corona.
         *
         * @param {boolean} enable Indicator if the figure should glow.
         */
        setGlow: function(enable) {
            var coronaColor = "#3b7fc4";

            this._super(enable);

            if (enable) {
                this.corona.setBackgroundColor(coronaColor);
                this.corona.setColor(coronaColor);

                // Maximum allowed value of Corona.setAlpha() is 0.3, set the attribute directly.
                this.corona.shape.attr("opacity", 0.45);

                this._setCursor(this.corona);

                // Remove the border.
                this.sendToFront();
                this.setStroke(0);
            } else {
                this.setStroke(this.STROKE_WIDTH);
            }
        },

        /**
         * Sets the cursor
         *
         * @param {draw2d.Figure} figure
         * @private
         */
        _setCursor: function(figure) {
            JIRA.WorkflowDesigner.Draw2DUtilities.setCursor(figure, "crosshair");
        }
    });

    /**
     * Finds the best hit target from the list of specified candidates at the specified location.
     *
     * @param {number} x The X coordinate of the location.
     * @param {number} y The Y coordinate of the location.
     * @param {JIRA.WorkflowDesigner.StatusPort[]} candidates The list of ports to consider when looking for the hit target.
     * @returns {JIRA.WorkflowDesigner.StatusPort|null} The best hit match or null if there is no matching candidate.
     */
    JIRA.WorkflowDesigner.StatusPort.getHitPort = function (x, y, candidates) {
        var draggedPoint = new draw2d.geo.Point(x, y),
            target;

        function hitsPoint(port) {
            return port.hitTest(draggedPoint.x, draggedPoint.y);
        }

        function getDistance(port) {
            // The center of a port is determined by the absolute coordinates, we don't need to
            // mess around with width/height
            return port.getAbsolutePosition().getDistance(draggedPoint);
        }

        target = _.chain(candidates)
            .filter(hitsPoint)
            .min(getDistance)
            .value();

        return target instanceof JIRA.WorkflowDesigner.StatusPort ? target : null;
    };
}());