(function () {
    /**
     * @param {string} RGB An RGB colour string (e.g. "rgb(255, 0, 0)").
     * @returns {string} The hexadecimal representation of `RGB` (e.g. "#FF0000").
     */
    function getHexadecimalFromRGB(RGB) {
        RGB = RGB.match(/\d+/g);
        return "#" + _.map(RGB, function (component) {
            component = parseInt(component, 10).toString(16);
            return ("0" + component).slice(-2);
        }).join("");
    }

    /**
     * Get a CSS colour property from an issue status lozenge.
     *
     * @param {object} options
     * @param {string} options.colourName The name of an issue status colour.
     * @param {boolean} [options.isSubtle=false] Whether to render the subtle version of the lozenge.
     * @param {string} options.property The name of the CSS colour property to read.
     * @returns {string} The value of the named CSS colour property.
     */
    function getStatusLozengeColour(options) {
        var element = renderStatusLozenge(options),
            result = element.css(options.property);

        element.remove();
        return getHexadecimalFromRGB(result);
    }

    /**
     * @param {object} options
     * @param {string} options.colourName The name of an issue status colour.
     * @param {boolean} [options.isSubtle=false] Whether to render the subtle version of the lozenge.
     * @returns {jQuery} An issue status lozenge appended to the body.
     */
    function renderStatusLozenge(options) {
        return jQuery(JIRA.Template.Util.Issue.Status.issueStatus({
            isSubtle: !!options.isSubtle,
            issueStatus: {
                statusCategory: {
                    colorName: options.colourName
                }
            }
        })).hide().appendTo("body");
    }

    AJS.namespace("JIRA.WorkflowDesigner.StatusLozengeColours");

    /**
     * Returns CSS colours of issue status lozenges.
     *
     * @namespace
     */
    JIRA.WorkflowDesigner.StatusLozengeColours = {
        /**
         * @param {object} options
         * @param {string} options.colourName The name of a lozenge colour.
         * @param {boolean} [options.isSubtle=false] Whether to render the subtle version of the lozenge.
         * @returns {string} The CSS background colour of the described status lozenge.
         */
        getBackgroundColour: function (options) {
            options.property = "background-color";
            return getStatusLozengeColour(options);
        },

        /**
         * @param {object} options
         * @param {string} options.colourName The name of a lozenge colour.
         * @param {boolean} [options.isSubtle=false] Whether to render the subtle version of the lozenge.
         * @returns {string} The CSS border colour of the described status lozenge.
         */
        getBorderColour: function (options) {
            options.property = "border-top-color";
            return getStatusLozengeColour(options);
        },

        /**
         * @param {object} options
         * @param {string} options.colourName The name of a lozenge colour.
         * @param {boolean} [options.isSubtle=false] Whether to render the subtle version of the lozenge.
         * @returns {string} The CSS text colour of the described status lozenge.
         */
        getColour: function (options) {
            options.property = "color";
            return getStatusLozengeColour(options);
        }
    };
}());