(function () {
    AJS.namespace("JIRA.WorkflowDesigner.BrowserZoomChecker");

    /**
     * @constructor
     * @classdesc Monitors browser zoom level and notifies the user when it is not 100%.
     */
    JIRA.WorkflowDesigner.BrowserZoomChecker = function() {
        _.bindAll(this, "_startWindowResizeSession", "_endWindowResizeSession");

        jQuery(window).on("resize", this._startWindowResizeSession);
        this._debouncedEndWindowResizeSession = _.debounce(this._endWindowResizeSession, 250);

        this._intervalId = setInterval(_.bind(this._check, this), 1000);
    };

    _.extend(JIRA.WorkflowDesigner.BrowserZoomChecker.prototype,
    /** @lends JIRA.WorkflowDesigner.BrowserZoomChecker# */
    {
        /**
         * Whether the user agent is running on Mac
         *
         * @type {boolean}
         * @readonly
         */
        _isMac: JIRA.WorkflowDesigner.BrowserUtilities.isMac(),

        /**
         * Displays an error message if the current browser zoom level is not supported.
         * If the level is supported and a message was previously shown then clears the message.
         *
         * @private
         */
        _check: function() {
            var message = this._isMac ? AJS.I18n.getText("workflow.designer.reset.zoom.level.mac") : AJS.I18n.getText("workflow.designer.reset.zoom.level.pc"),
                zoomLevel;

            // WebKit browsers seem to update window.outerWidth at a slower rate than window.innerWidth while window is being rapidly re-sized.
            // Disable zoom checking until window resizing is finished.
            if (this._windowIsResizing) {
                return;
            }

            zoomLevel = JIRA.WorkflowDesigner.BrowserUtilities.getZoomLevel();

            if (!zoomLevel) {
                return;
            }

            if (zoomLevel !== 1) {
                this._browserZoomMessage = JIRA.WorkflowDesigner.Messages.showErrorMessage(message);
            } else if (this._browserZoomMessage) {
                this._browserZoomMessage.remove();
            }
        },

        /**
         * Stops the browser zoom level checking.
         */
        destroy: function() {
            clearInterval(this._intervalId);
            jQuery(window).off("resize", this._startWindowResizeSession);

            this._browserZoomMessage && this._browserZoomMessage.remove();
        },

        /**
         * Handler for end window resize session.
         *
         * @private
         */
        _endWindowResizeSession: function() {
            this._windowIsResizing = false;
        },

        /**
         * Handler for start window resize session.
         *
         * @private
         */
        _startWindowResizeSession: function() {
            this._windowIsResizing = true;
            this._debouncedEndWindowResizeSession();
        }
    });
}());