(function () {
    AJS.namespace("JIRA.WorkflowDesigner.BrowserUtilities");

    /**
     * Browser-related utility methods.
     *
     * @namespace
     */
    JIRA.WorkflowDesigner.BrowserUtilities = {
        /**
         * Escapes a value so that it can be used as a URL query parameter.
         *
         * @param {string} value The query parameter to escape.
         * @return {string} Escaped query parameter.
         */
        escapeQueryParameter: function (value) {
            // Encode according to RFC 3986 syntax.
            return encodeURIComponent(value)
                    .replace(/[!'()]/g, escape)
                    .replace(/\*/g, "%2A")
                    .replace(/%20/g, "+");
        },

        /**
         * Construct a query string from an object.
         *
         * @param {object} parameters The keys/values to convert to a query string.
         * @return {string} A query string constructed from <tt>parameters</tt>.
         */
        makeQueryString: function (parameters) {
            function makeKeyValue(pair) {
                var encode = JIRA.WorkflowDesigner.BrowserUtilities.escapeQueryParameter;
                return _.map(pair, encode).join("=");
            }

            return _.map(_.pairs(parameters), makeKeyValue).join("&");
        },

        /**
         * Returns the current browser zoom level as a floating point number rounded to 1 decimal place.
         * E.g., the value of 1.5 means that the browser is zoomed to 150%.
         * Returns null in Firefox.
         *
         * @returns {number} Current browser zoom level.
         */
        getZoomLevel: function() {
            var ratio;

            if (jQuery.browser.mozilla) {
                // Currently we don't need to know browser zoom level in Firefox.
                return null;
            } else if (jQuery.browser.msie) {
                ratio = screen.deviceXDPI / screen.logicalXDPI;
            } else if (jQuery.browser.webkit) {
                // Outer width is in device pixels (browser window width). Inner width is in CSS pixels (viewport width including scrollbar).
                // They are roughly equal if zoom level is 100%, but inner width changes with zooming.
                ratio = window.outerWidth / window.innerWidth;
            }

            return Math.round(ratio * 10) / 10;
        },

        /**
         * @returns {Boolean} Whether the current operating system is Mac.
         */
        isMac: function() {
            return navigator.platform.toUpperCase().indexOf("MAC") !== -1;
        }
    };
}());