AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:test-resources");
AJS.test.require("com.atlassian.jira.plugins.jira-workflow-designer:workflow-designer");

module("JIRA.WorkflowDesigner.WorkflowModel", {
    setup: function () {
        this.workflowModel = new JIRA.WorkflowDesigner.WorkflowModel();
        this.sandbox = sinon.sandbox.create();
    },

    teardown: function() {
        this.sandbox.restore();
    }
});

test("addStatus()", function () {
    var expectedAttributes,
        status,
        statuses = this.workflowModel.get("statuses");

    expectedAttributes = {
        description: "",
        id: "id",
        stepId: 42,
        statusCategory: null,
        statusId: "42",
        name: "Status",
        initial: false,
        x: 42,
        y: 1337
    };

    status = this.workflowModel.addStatus({
        id: "id",
        stepId: 42,
        statusId: "42",
        name: "Status",
        x: 42,
        y: 1337
    });

    ok(status instanceof JIRA.WorkflowDesigner.StatusModel, "The new StatusModel was returned");
    deepEqual(expectedAttributes, status.attributes, "Its attributes are correct");
    ok(statuses.length === 1 && statuses.at(0) === status, "It was added to the StatusCollection");
});

test("addTransition()", function () {
    var expectedAttributes,
        source = new JIRA.WorkflowDesigner.StatusModel({id: "source"}),
        target = new JIRA.WorkflowDesigner.StatusModel({id: "target"}),
        transition,
        transitions = this.workflowModel.get("transitions");

    expectedAttributes = {
        id: "A<1:source:target>",
        actionId: 1,
        description: null,
        globalTransition: false,
        name: "Transition",
        source: source,
        sourceAngle: 90,
        target: target,
        targetAngle: -90,
        screenId: null,
        screenName: null,
        propertiesCount: 1,
        conditionsCount: 2,
        validatorsCount: 3,
        postFunctionsCount: 4
    };

    transition = this.workflowModel.addTransition({
        actionId: 1,
        globalTransition: false,
        name: "Transition",
        source: source,
        sourceAngle: 90,
        target: target,
        targetAngle: -90,
        propertiesCount: 1,
        conditionsCount: 2,
        validatorsCount: 3,
        postFunctionsCount: 4
    });

    ok(transition instanceof JIRA.WorkflowDesigner.TransitionModel, "The new TransitionModel was returned");
    deepEqual(transition.attributes, expectedAttributes, "Its attributes are correct");
    ok(transitions.length === 1 && transitions.at(0) === transition, "It was added to the WorkflowModel's TransitionCollection");
});

test("addTransition() global transition", function () {
    var expectedAttributes,
        status = new JIRA.WorkflowDesigner.StatusModel({id: "status"}),
        transition,
        transitions = this.workflowModel.get("transitions");

    expectedAttributes = {
        id: "A<1:status:status>",
        actionId: 1,
        description: "Resolve the issue!",
        globalTransition: true,
        name: "Resolve",
        source: status,
        sourceAngle: null,
        target: status,
        targetAngle: null,
        screenName: "Resolve View",
        screenId: 2,
        propertiesCount: 1,
        conditionsCount: 2,
        validatorsCount: 3,
        postFunctionsCount: 4
    };

    transition = this.workflowModel.addTransition({
        actionId: 1,
        description: "Resolve the issue!",
        globalTransition: true,
        name: "Resolve",
        source: status,
        target: status,
        screenName: "Resolve View",
        screenId: 2,
        propertiesCount: 1,
        conditionsCount: 2,
        validatorsCount: 3,
        postFunctionsCount: 4
    });

    ok(transition instanceof JIRA.WorkflowDesigner.TransitionModel, "The new TransitionModel was returned");
    deepEqual(transition.attributes, expectedAttributes, "Its attributes are correct");
    ok(transitions.length === 1 && transitions.at(0) === transition, "It was added to the WorkflowModel's TransitionCollection");
});

test("getGlobalTransitionForStatus()", function () {
    var globalTransition,
        status = this.workflowModel.addStatus({});

    ok(this.workflowModel.getGlobalTransitionForStatus(status) === undefined,
            "Returns undefined when the status doesn't have a global transition");

    globalTransition = this.workflowModel.addTransition({
        globalTransition: true,
        target: status
    });

    ok(this.workflowModel.getGlobalTransitionForStatus(status) === globalTransition,
            "Returns the correct TransitionModel when the status has a global transition");
});

test("getStatus()", function () {
    var status;

    ok(this.workflowModel.getStatus("open") === undefined,
            "Returns undefined when no status is found");

    status = this.workflowModel.addStatus({id: "open"});
    ok(this.workflowModel.getStatus("open") === status,
            "Returns the correct StatusModel when a status is found");
});

test("reset() doesn't trigger new:* events on initial load", function () {
    var spy = sinon.spy();

    this.workflowModel.on("new:status new:transition", spy);
    this.workflowModel.reset({
        statuses: [new JIRA.WorkflowDesigner.StatusModel()],
        transitions: [new JIRA.WorkflowDesigner.TransitionModel()]
    });

    equal(spy.callCount, 0, "No new:* events were triggered");
});

test("reset() resets the model to a given state", function () {
    var closed,
        open,
        statuses = this.workflowModel.get("statuses"),
        transition,
        transitions = this.workflowModel.get("transitions");

    equal(statuses.length, 0, "The model contains no statuses");
    equal(transitions.length, 0, "The model contains no transitions");

    closed = new JIRA.WorkflowDesigner.StatusModel({name: "Closed"});
    open = new JIRA.WorkflowDesigner.StatusModel({name: "Open"});
    transition = new JIRA.WorkflowDesigner.TransitionModel({
        name: "Close",
        source: open,
        target: closed
    });

    this.workflowModel.reset({
        statuses: [closed, open],
        transitions: [transition]
    });

    equal(statuses.length, 2, "The model contains the correct number of statuses");
    equal(transitions.length, 1, "The model contains the correct number of transitions");

    this.workflowModel.reset({
        draft: true,
        name: "Different Name",
        statuses: [],
        transitions: []
    });

    equal(this.workflowModel.get("draft"), true, "The model's draft status is correct");
    equal(this.workflowModel.get("name"), "Different Name", "The model's name is correct");
    equal(statuses.length, 0, "The model contains the correct number of statuses");
    equal(transitions.length, 0, "The model contains the correct number of transitions");
});

test("reset() triggers a new:status event if there is a single new status", function () {
    var closed = new JIRA.WorkflowDesigner.StatusModel({id: "Closed"}),
        inProgress = new JIRA.WorkflowDesigner.StatusModel({id: "In Progress"}),
        open = new JIRA.WorkflowDesigner.StatusModel({id: "Open"}),
        pullRequest = new JIRA.WorkflowDesigner.StatusModel({id: "Pull Request"}),
        spy = sinon.spy();

    this.workflowModel.addStatus(closed);
    this.workflowModel.on("new:status", spy);
    this.workflowModel.reset({
        statuses: [closed, inProgress],
        transitions: []
    });

    equal(spy.callCount, 1, "A new:status event is triggered when there is a single new status");
    ok(spy.args[0][0] === inProgress, "The new status was passed as an argument");

    this.workflowModel.reset({
        statuses: [closed, inProgress, open, pullRequest],
        transitions: []
    });

    equal(spy.callCount, 1, "No new:status event is triggered when there are multiple new statuses");
});

test("reset() triggers a new:transition event if there is a single new transition", function () {
    var close = new JIRA.WorkflowDesigner.TransitionModel({id: "Closed"}),
        reopen = new JIRA.WorkflowDesigner.TransitionModel({id: "Pull Request"}),
        spy = sinon.spy(),
        startProgress = new JIRA.WorkflowDesigner.TransitionModel({id: "In Progress"}),
        stopProgress = new JIRA.WorkflowDesigner.TransitionModel({id: "Open"});

    this.workflowModel.addTransition(close);
    this.workflowModel.on("new:transition", spy);
    this.workflowModel.reset({
        statuses: [],
        transitions: [close, reopen]
    });

    equal(spy.callCount, 1, "A new:transition event is triggered when there is a single new transition");
    ok(spy.args[0][0] === reopen, "The new transition was passed as an argument");

    this.workflowModel.reset({
        statuses: [],
        transitions: [close, reopen, startProgress, stopProgress]
    });

    equal(spy.callCount, 1, "No new:transition event is triggered when there are multiple new transitions");
});

test("Triggers a layoutChanged event when status coordinates change", function () {
    var layoutChangedSpy = sinon.spy(),
        status;

    status = this.workflowModel.addStatus({
        id: "id",
        name: "Status",
        stepId: 1,
        x: 0,
        y: 0
    });

    this.workflowModel.bind("layoutChanged", layoutChangedSpy);

    // Same value
    status.set({x: 0, y: 0});
    ok(!layoutChangedSpy.called, "A layoutChanged event wasn't triggered");

    // Different value
    status.set({x: 10, y: 10});
    ok(layoutChangedSpy.called, "A layoutChanged event was triggered");
});

test("Triggers a layoutChanged event when transition action ID changes value", function () {
    var layoutChangedSpy = sinon.spy(),
        transition = this.workflowModel.addTransition({actionId: 1});

    this.workflowModel.bind("layoutChanged", layoutChangedSpy);

    // Same value
    transition.set({actionId: 1});
    ok(!layoutChangedSpy.called, "A layoutChanged event wasn't triggered");

    // Different value
    transition.set({actionId: 2});
    ok(layoutChangedSpy.called, "A layoutChanged event was triggered");
});

test("Triggers a layoutChanged event when transition angles change", function () {
    var layoutChangedSpy = sinon.spy(),
        transition = this.workflowModel.addTransition({sourceAngle: 90, targetAngle: -90});

    this.workflowModel.bind("layoutChanged", layoutChangedSpy);

    // Same values
    transition.set({sourceAngle: 90, targetAngle: -90});
    ok(!layoutChangedSpy.called, "A layoutChanged event wasn't triggered");

    // Different values
    transition.set({sourceAngle: 0, targetAngle: 180});
    ok(layoutChangedSpy.called, "A layoutChanged event was triggered");
});

test("Triggers a layoutChanged event when transition target changes", function () {
    var layoutChangedSpy = sinon.spy(),
        status = this.workflowModel.addStatus(),
        transition = this.workflowModel.addTransition({target: status});

    this.workflowModel.bind("layoutChanged", layoutChangedSpy);

    // Same values
    transition.set("target", status);
    ok(!layoutChangedSpy.called, "A layoutChanged event wasn't triggered");

    // Different values
    transition.set("target", this.workflowModel.addStatus());
    ok(layoutChangedSpy.called, "A layoutChanged event was triggered");
});

test("Triggers a layoutChanged event when transition model is destroyed", function () {
    var layoutChangedSpy  = sinon.spy(),
        transition = this.workflowModel.addTransition({ actionId: 1 });

    this.workflowModel.bind("layoutChanged", layoutChangedSpy);

    transition.destroy();

    ok(layoutChangedSpy.called, "A layoutChanged event was triggered");
});

test("statusHasGlobalTransition() returns false when status has no transitions", function() {
    var inProgress = new JIRA.WorkflowDesigner.StatusModel({id: 1});

    this.workflowModel.reset({
        statuses: [inProgress]
    });

    ok(!this.workflowModel.statusHasGlobalTransition(inProgress), "Should return false when no transitions");
});

test("statusHasGlobalTransition() returns false when status has only normal transitions", function() {
    var close, closed, inProgress, open, startProgress;

    closed = new JIRA.WorkflowDesigner.StatusModel({id: 1});
    inProgress = new JIRA.WorkflowDesigner.StatusModel({id: 2});
    open = new JIRA.WorkflowDesigner.StatusModel({id: 3});

    close = new JIRA.WorkflowDesigner.TransitionModel({actionId: 1, source: open, target: closed});
    startProgress = new JIRA.WorkflowDesigner.TransitionModel({actionId: 2, source: open, target: inProgress});

    this.workflowModel.reset({
        statuses: [closed, inProgress, open],
        transitions: [close, startProgress]
    });

    ok(!this.workflowModel.statusHasGlobalTransition(inProgress), "Should return false when only normal transitions");
});

test("statusHasGlobalTransition() returns true when status has only global transition", function() {
    var globalStartProgress, inProgress;

    inProgress = new JIRA.WorkflowDesigner.StatusModel({id: 1});

    globalStartProgress = new JIRA.WorkflowDesigner.TransitionModel({actionId: 1, target: inProgress, globalTransition: true});

    this.workflowModel.reset({
        statuses: [inProgress],
        transitions: [globalStartProgress]
    });

    ok(this.workflowModel.statusHasGlobalTransition(inProgress), "Should return true when only a global transition");
});

test("statusHasGlobalTransition() returns true when status has global transition and normal transitions", function() {
    var close, closed, globalStartProgress, inProgress, open, startProgress;

    closed = new JIRA.WorkflowDesigner.StatusModel({id: 1});
    inProgress = new JIRA.WorkflowDesigner.StatusModel({id: 2});
    open = new JIRA.WorkflowDesigner.StatusModel({id: 3});

    close = new JIRA.WorkflowDesigner.TransitionModel({actionId: 1, source: open, target: closed});
    startProgress = new JIRA.WorkflowDesigner.TransitionModel({actionId: 2, source: open, target: inProgress});

    globalStartProgress = new JIRA.WorkflowDesigner.TransitionModel({actionId: 3, target: inProgress, globalTransition: true});

    this.workflowModel.reset({
        statuses: [closed, inProgress, open],
        transitions: [close, startProgress, globalStartProgress]
    });

    ok(this.workflowModel.statusHasGlobalTransition(inProgress), "Should return true when normal transitions and a global transition");
});

test("Draft workflow status that exists on the live workflow is not deletable", function() {
    var status = this.workflowModel.addStatus({stepId: 1});

    this.workflowModel.set({
        draft: true,
        liveStepIds: [1]
    });

    ok(!this.workflowModel.statusIsDeletable(status), "Status is non-deletable");
});

test("Draft workflow status that doesn't exists on the live workflow is deletable", function() {
    var status = this.workflowModel.addStatus({stepId: 1});

    this.workflowModel.set({
        draft: true,
        liveStepIds: [2]
    });

    ok(this.workflowModel.statusIsDeletable(status), "Status is deletable");
});

test("All live workflow statuses are deletable", function () {
    var status = this.workflowModel.addStatus({stepId: 1});
    this.workflowModel.set({liveStepIds: [1]});

    ok(this.workflowModel.statusIsDeletable(status), "Status is deletable");
});

test("updateTransitionTargets sets the passed step on all transitions with the passed action ID", function () {
    var status = this.workflowModel.addStatus({ stepId: 1 }),
        transitionOne = this.workflowModel.addTransition({ actionId: 2 }),
        transitionTwo = this.workflowModel.addTransition({ actionId: 2 });

    this.sandbox.spy(transitionOne, "setTarget");
    this.sandbox.spy(transitionTwo, "setTarget");

    this.workflowModel.updateTransitionTargets({
        targetAngle: 0,
        targetStepId: 1,
        transition: transitionOne
    });

    equal(transitionOne.setTarget.callCount, 1, "setTarget of the first transition was called");
    equal(transitionOne.setTarget.args[0][0], status, "It was passed the correct target");
    equal(transitionOne.setTarget.args[0][1], 0, "It was passed the correct target angle");
    equal(transitionTwo.setTarget.callCount, 1, "setTarget of the second transition was called");
    equal(transitionTwo.setTarget.args[0][0], status, "It was passed the correct target");
    ok(!transitionTwo.setTarget.args[0][1], "It was passed the correct target angle");
});

test("updateTransitionSource sets the new source", function () {
    var status = this.workflowModel.addStatus({ stepId: 1 }),
        transitionOne = this.workflowModel.addTransition({ actionId: 2 });

    this.sandbox.spy(transitionOne, "setSource");

    this.workflowModel.updateTransitionSource({
        sourceAngle: 0,
        sourceStepId: 1,
        transition: transitionOne
    });

    equal(transitionOne.setSource.callCount, 1, "setSource of the first transition was called");
    equal(transitionOne.setSource.args[0][0], status, "It was passed the correct source");
    equal(transitionOne.setSource.args[0][1], 0, "It was passed the correct source angle");
});