(function() {

    JPT.AddProjectControllerImpl = Class.extend({

        postProjectCreationCallbacks: [],
        projectKeyValidationCallbacks: [],
        projectNamesUpperCased: [],

        init: function(options) {
            this._window = options.window || window; // used for mocking 'window' in QUnit
            _.bindAll(this, "projectCreatedHandler", "projectValidationFailedHandler", "nameTimeout", "keyTimeout", "submit");
        },

        initProjectTemplate: function(webItemData) {
            this.desiredKeyLength = 4;
            this.lastKeyValidated = "";

            this._getExistingProjects();
            JPT.ProjectKeyGenerator.init(this.desiredKeyLength, JPT.ConfigModel.maxKeyLength);
            this.webItemData = webItemData;
            JPT.AddProjectView.prepareDialog(webItemData.name);
            JPT.AddProjectView.draw({
                webItemData: this.webItemData,
                maxKeyLength: JPT.ConfigModel.maxKeyLength,
                maxNameLength: JPT.ConfigModel.maxNameLength,
                currentUserName: JPT.ConfigModel.currentUserName,
                currentUserDisplayName: JPT.ConfigModel.currentUserDisplayName,
                currentUserAvatarUrl: JPT.ConfigModel.currentUserAvatarUrl,
                errors: {}
            });
        },

        _resetProjectLeadValues: function() {
            JPT.ConfigModel.currentUserDisplayName = "";
            JPT.ConfigModel.currentUserName = "";
            JPT.ConfigModel.currentUserAvatarUrl = "";
        },

        _hasFullyConfiguredProjectLead: function() {
            return JPT.AddProjectView.getLeadDisplayName() && JPT.AddProjectView.getAvatarUrlOfSelectedLead() != "";
        },

        submit: function(event) {
            // This data is retrieved before the input fields are disabled (in enterLoadingState()) to avoid null-pointer exceptions when the fields are empty.
            var formData = JPT.AddProjectView.getAddProjectForm().serialize();

            if (JPT.AddProjectView.hasInlineErrors()) {
                return;
            }

            if (JPT.AddProjectView.enterLoadingState()) {
                // store lead so we can repopulate the user picker for when the submit fails
                if (this._hasFullyConfiguredProjectLead()) {
                    JPT.ConfigModel.currentUserDisplayName = JPT.AddProjectView.getLeadDisplayName();
                    JPT.ConfigModel.currentUserName = JPT.ConfigModel.currentUserDisplayName ? JPT.AddProjectView.getLeadUserName() : "";
                    JPT.ConfigModel.currentUserAvatarUrl = JPT.AddProjectView.getAvatarUrlOfSelectedLead();
                } else {
                    this._resetProjectLeadValues();
                }

                var projectCreateDeferred = AJS.$.ajax({
                    url: contextPath + "/rest/project-templates/1.0/templates",
                    type: "POST",
                    data: formData,
                    timeout: 50000,
                    headers: {
                        "X-Atlassian-Token": "nocheck"
                    }
                });

                projectCreateDeferred.then(this.projectCreatedHandler, this.projectValidationFailedHandler);
            }
        },

        registerPostProjectCreationCallback: function(callback) {
            this.postProjectCreationCallbacks.push(callback);
        },

        registerProjectKeyValidationCallback: function(callback) {
            this.projectKeyValidationCallbacks.push(callback);
        },

        localStoragePrefix: "jira.projecttemplates.",

        projectCreatedHandler: function(data) {
            JPT.AddProjectView.avoidDirtyFormWarning();

            // Put the created projects into local storage
            var localStoragePrefix = this.localStoragePrefix;
            var remoteProjectTypes = ["confluenceProject", "fisheyeProject", "crucibleProject", "bambooProject"];
            _.map(remoteProjectTypes, function(remoteProject) {
                if (data.remoteProjectLinks[remoteProject]) {
                    localStorage.setItem(localStoragePrefix + remoteProject, data.remoteProjectLinks[remoteProject]);
                }
            });


            // If there are any post Project Created callbacks registered, call these. Otherwise redirect to the desired return URL.
            if (!_.isEmpty(this.postProjectCreationCallbacks)) {
                JPT.DialogController.dialog.addPage("post-project-created-page");
                _.each(this.postProjectCreationCallbacks, function(callback) {
                    callback(JPT.DialogController.dialog, data.returnUrl, data.projectId, data.projectKey, data.projectName);
                });
            } else {
                this._window.location = contextPath + data.returnUrl;
            }
        },

        projectValidationFailedHandler: function(jqXhr, textStatus) {
            var errors = {};
            if (jqXhr.status === 400) {
                errors = JSON.parse(jqXhr.responseText);
            } else if (jqXhr.status === 500 && !_.isUndefined(JSON.parse(jqXhr.responseText).message)) {
                var errorResponse = JSON.parse(jqXhr.responseText);
                // Project template configuration threw an exception: Project has been created but the template hasn't been configured
                JIRA.Messages.showReloadErrorMsg(errorResponse.message);
                JPT.AddProjectView.avoidDirtyFormWarning();
                this._window.location = contextPath + errorResponse.returnUrl;
                // Remain in loading state while refreshing.
                return;
            } else if (textStatus === "timeout") {
                errors = {"errorMessages": [AJS.I18n.getText("add.projects.templates.error.timeout")]}
            } else {
                // Different kind of error. Let's generate an errors object with a general error that contains the right information
                errors = {"errorMessages": [AJS.I18n.getText("add.projects.templates.error.unknownError", jqXhr.responseText)]}
            }

            JPT.AddProjectView.draw({
                webItemData: this.webItemData,
                errors: errors,
                currentName: JPT.AddProjectView.getName(),
                currentKey: JPT.AddProjectView.getKey(),
                currentUserDisplayName: JPT.ConfigModel.currentUserDisplayName,
                currentUserName: JPT.ConfigModel.currentUserName,
                currentUserAvatarUrl: JPT.ConfigModel.currentUserAvatarUrl
            });

            JPT.AddProjectView.hideLoadingState();
        },

        _updateAndValidateKey: function updateKey(key) {
            JPT.AddProjectView.setKey(key);
            this._validateKey();
        },

        _shouldUpdateKey: function _shouldUpdateKey() {
            return (JPT.AddProjectView.getKeyEdited() != "true");
        },

        autofillKeyIfNeeded: function autofillKeyIfNeeded() {
            if (this._shouldUpdateKey()) {
                var key = JPT.ProjectKeyGenerator.generate(JPT.AddProjectView.getName());
                // JRADEV-10797 - Rather than validate the key,
                // we'll pretend that a key is always invalid if it's less than 1 character long.
                if (key.length > 1) {
                    this._updateAndValidateKey(key);
                } else {
                    // Blank the key without validation.
                    JPT.AddProjectView.setKey("");
                }
            }
        },

        _doesProjectNameExists: function(name) {
            var x;
            for (x in this.projectNamesUpperCased) {
                if (name.toUpperCase() == this.projectNamesUpperCased[x]) {
                    return true;
                }
            }

            return false;
        },

        _validateName: function() {
            var name = AJS.$.trim(JPT.AddProjectView.getName());

            if (!name) {
                return;
            }

            if (name.length < JPT.ConfigModel.minNameLength) {
                JPT.AddProjectView.showInlineErrorForName(AJS.I18n.getText("admin.errors.project.name.too.short", JPT.ConfigModel.minNameLength));
                return;
            }

            if (name.length > JPT.ConfigModel.maxNameLength) {
                JPT.AddProjectView.showInlineErrorForName(AJS.I18n.getText("admin.errors.project.name.too.long", JPT.ConfigModel.maxNameLength));
                return;
            }

            if (this._doesProjectNameExists(name)) {
                JPT.AddProjectView.showInlineErrorForName(AJS.I18n.getText("admin.errors.project.with.that.name.already.exists"));
                return;
            }

            JPT.AddProjectView.hideInlineErrorForName();
        },

        _performKeyValidationChecks: function(key) {
            var validationChecksDeferred = AJS.$.ajax({
                url: contextPath + "/rest/api/latest/projectvalidate/key?key=" + key.toUpperCase(),
                timeout: 5000
            });

            validationChecksDeferred.done(_.bind(function (errors) {
                if (errors.errors && errors.errors["projectKey"]) {
                    JPT.AddProjectView.showInlineErrorForKey(errors.errors["projectKey"]);
                } else {
                    var foundError = false;
                    _.each(this.projectKeyValidationCallbacks, function(callback) {
                        var errors = callback(key.toUpperCase());
                        if (errors.errors && errors.errors["projectKey"]) {
                            foundError = true;
                            JPT.AddProjectView.showInlineErrorForKey(errors.errors["projectKey"]);
                        }
                    });
                    if (!foundError) {
                        JPT.AddProjectView.hideInlineErrorForKey();
                    }
                }
            }, this));
        },

        _validateKey: function _validateKey() {
            var key = JPT.AddProjectView.getKey();

            // Only validate the key if it has changed since the last time we validated it
            if (this.lastKeyValidated === key) {
                return;
            }

            if (key) {
                this.lastKeyValidated = key;
                this._performKeyValidationChecks(key);
            } else {
                JPT.AddProjectView.hideInlineErrorForKey();
            }
        },

        nameTimeout: function nameTimeout() {
            this._validateName();
            this.autofillKeyIfNeeded();
        },

        keyTimeout: function keyTimeout() {
            JPT.AddProjectView.setKeyEdited();
            this._validateKey();
        },

        _getExistingProjects: function() {
            // Avoid retrieving the list twice (e.g. after hitting the 'Back' button and selecting new template)
            if (this.projectNamesUpperCased.length > 0) {
                return this.projectNamesUpperCased;
            }

            var existingProjectsDeferred = AJS.$.ajax({
                url: contextPath + "/rest/api/latest/project",
                timeout: 5000
            });

            existingProjectsDeferred.done(_.bind(function(projects) {
                this.projectNamesUpperCased = _.map(projects, function(project) {
                    return project.name.toUpperCase();
                });
            }, this));
        }
    });

    JPT.AddProjectController = new JPT.AddProjectControllerImpl({ el: AJS.$(document) });

})();
