package com.atlassian.jira.plugins.importer.github.web;

import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.config.ConstantsManager;
import com.atlassian.jira.issue.fields.config.FieldConfig;
import com.atlassian.jira.issue.fields.config.FieldConfigScheme;
import com.atlassian.jira.issue.fields.config.manager.IssueTypeSchemeManager;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.issue.resolution.Resolution;
import com.atlassian.jira.plugins.importer.github.config.ConfigBean;
import com.atlassian.jira.plugins.importer.tracking.UsageTrackingService;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.security.xsrf.RequiresXsrfCheck;
import com.atlassian.jira.util.ParameterUtils;
import com.atlassian.jira.workflow.WorkflowSchemeManager;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import org.eclipse.egit.github.core.Label;
import webwork.action.ActionContext;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Enables the user to map GitHub issue labels to JIRA issue types and issue resolutions.
 */
public class GithubLabelMappingPage extends GithubImportProcessSupport {

	public GithubLabelMappingPage(@ComponentImport UsageTrackingService usageTrackingService,
								  @ComponentImport WebInterfaceManager webInterfaceManager,
								  @ComponentImport PluginAccessor pluginAccessor,
								  @ComponentImport ProjectManager projectManager,
								  @ComponentImport IssueTypeSchemeManager issueTypeSchemeManager,
								  @ComponentImport WorkflowSchemeManager workflowSchemeManager) {
		super(usageTrackingService, webInterfaceManager, pluginAccessor, projectManager, issueTypeSchemeManager, workflowSchemeManager);
	}

	public boolean displayIssueTypeWarning() {
		return getFieldConfigScheme().isPresent();
	}

	public String getIssueTypeSchemeName() {
		final Optional<FieldConfigScheme> fieldConfigScheme = getFieldConfigScheme();
		return fieldConfigScheme.isPresent() ? fieldConfigScheme.get().getName() : "???";
	}

	private Optional<FieldConfigScheme> getFieldConfigScheme() {
		final Iterable<FieldConfigScheme> schemes = Iterables.transform(getExistingDestinationProjects(), new Function<Project, FieldConfigScheme>() {
			@Override
			public FieldConfigScheme apply(Project input) {
				return issueTypeSchemeManager.getConfigScheme(input);
			}
		});
		return Optional.fromNullable(Iterables.getFirst(Iterables.filter(schemes, Predicates.notNull()), null));
	}


	@Override
    @RequiresXsrfCheck
    protected String doExecute() throws Exception {
        if( isNextClicked() ) {
            getConfigBean().setAutoLabels( ActionContext.getParameters().containsKey("autoLabels") );
            populateLabelMappings();
			setDefaultIssueTypes();
        }
        if( isPreviousClicked() ) {
            return getRedirect(getImporterBaseUrlWithSlash()
                    + "GithubProjectMappingPage!default.jspa?externalSystem=" + getExternalSystem()
                    + "&atl_token=" + getXsrfToken());
        }

        return super.doExecute();
    }

	private void setDefaultIssueTypes() {
		final ConfigBean configBean = getConfigBean();
		Map<String, IssueType> defaultIssueTypes = Maps.newHashMap();
		for (String selectedProject : configBean.getSelectedProjects()) {
			final String projectKey = configBean.getProjectKey(selectedProject);
			final Project project = projectManager.getProjectObjByKey(projectKey);
			if (project == null) {
				issueTypeSchemeManager.getDefaultIssueTypeScheme();
				final FieldConfig defaultSchemeConfig = issueTypeSchemeManager.getDefaultIssueTypeScheme().getOneAndOnlyConfig();
				final IssueType defaultValue = issueTypeSchemeManager.getDefaultValue(defaultSchemeConfig);
				defaultIssueTypes.put(projectKey, defaultValue);
			} else {

				final IssueType defaultIssueType = issueTypeSchemeManager.getDefaultIssueType(project);
				if(defaultIssueType != null) {
					defaultIssueTypes.put(projectKey, defaultIssueType);
				} else {
					final IssueType firstIssueType = issueTypeSchemeManager.getIssueTypesForProject(project).iterator().next();
					defaultIssueTypes.put(projectKey, firstIssueType);
				}
			}
		}
		configBean.setDefaultIssueTypesForProjectKey(defaultIssueTypes);
	}

    private void populateLabelMappings() {
		final ConfigBean configBean = getConfigBean();
		final List<Label> labels = configBean.getGithubDataService().getLabels();

		final Iterable<String> labelNames = Iterables.transform(labels, new Function<Label, String>() {
			@Override
			public String apply(Label input) {
				return input.getName();
			}
		});
		configBean.setIssueTypeMapping(getMapping(labelNames, "typeMapping_"));
		configBean.setResolutionMappings(getMapping(labelNames, "resolutionMapping_"));
    }

	private Map<String, Integer> getMapping(Iterable<String> labelNames, final String mappingPrefix) {
		final HashMap<String, Integer> mapping = Maps.newHashMap();
		final Function<String, Integer> labelToId = new Function<String, Integer>() {
			@Override
			public Integer apply(String input) {
				final String typeId = ParameterUtils.getStringParam(ActionContext.getParameters(), mappingPrefix + input);
				return intOrNull(typeId);
			}

			private Integer intOrNull(String value) {
				try {
					return new Integer(value);
				} catch (NumberFormatException e) {
					return null;
				}
			}
		};
		for (String labelName : labelNames) {
			final Integer value = labelToId.apply(labelName);
			if(value != null) {
				mapping.put(labelName, value);
			}
		}
		return ImmutableMap.copyOf(mapping);
	}

	@Override
    public String getFormTitle() {
        return getText("com.atlassian.jira.plugins.importer.github.labelmapping.title");
    }

    public Map<String, String> getJiraIssueTypeValues() {
        Map<String, String> options = new LinkedHashMap<String, String>();
        for( IssueType issueType : getAllowedIssueTypes() ) {
            options.put(issueType.getId(), issueType.getNameTranslation());
        }
        return options;
    }

	private Iterable<IssueType> getAllowedIssueTypes() {
		final List<Project> existingDestinationProjects = getExistingDestinationProjects();
		if (existingDestinationProjects.size() > 0) {
			final Project project = existingDestinationProjects.get(0);
			return issueTypeSchemeManager.getNonSubTaskIssueTypesForProject(project);
		} else {
			return Iterables.filter(issueTypeSchemeManager.getIssueTypesForDefaultScheme(), new Predicate<IssueType>() {
				@Override
				public boolean apply(final IssueType input) {
					return !input.isSubTask();
				}
			});
		}
	}

    public Map<String, String> getJiraIssueResolutionValues() {
        ConstantsManager constantsManager = ComponentAccessor.getConstantsManager();
        Map<String, String> options = new LinkedHashMap<String, String>();
        for( Resolution resolution : constantsManager.getResolutionObjects() ) {
            options.put(resolution.getId(), resolution.getNameTranslation());
        }
        return options;
    }

    public String getSelectedIssueTypeMapping(String label) {
		final Integer value = getConfigBean().getIssueTypeMapping().get(label);
		return value != null ? value.toString() : "";
    }

    public String getSelectedIssueResolutionMapping(String label) {
		final Integer value = getConfigBean().getResolutionMappings().get(label);
		return value != null ? value.toString() : "";
    }

}