package com.atlassian.jira.plugins.importer.github.importer.markup;

import com.google.common.base.Function;
import com.google.common.base.Optional;

import javax.annotation.Nullable;
import java.util.regex.MatchResult;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class SimpleSyntax implements SimpleConverter {

	private static final MarkdownChanger[] CHANGERS = new MarkdownChanger[] {
		new MarkdownChanger("\r\n", new Replace("\n")),
		new MarkdownChanger("^(#+)(?=[^\\d])", Pattern.MULTILINE, new Function<MatchResult, Optional<String>>() {
			@Override
			public Optional<String> apply(MatchResult input) {
				final String group = input.group(1);
				if(group.length() <= 6) {
					return Optional.of("h" + group.length() + ".");
				}
				return Optional.absent();
			}
		}),
		new MarkdownChanger("^(.+?)\n([-=]+)$", Pattern.MULTILINE, new Function<MatchResult, Optional<String>>() {
			@Override
			public Optional<String> apply(MatchResult input) {
				int headerLevel = input.group(2).charAt(0) == '=' ? 1 : 2;
				final String result = String.format("h%d. %s", headerLevel, input.group(1));
				return Optional.of(result);
			}
		}),
		new MarkdownChanger("^[-\\*_]{3,}\\s*?$", Pattern.MULTILINE, new Replace("----")),
		new MarkdownChanger("([\\*_]+)(.+?)\\1", new Function<MatchResult, Optional<String>>() {
			@Override
			public Optional<String> apply(MatchResult input) {
				String character = input.group(1).length() == 2 ? "*" : "_";
				return Optional.of(character + "$2" + character);
			}
		}),
		new MarkdownChanger("~~(.+?)~~", new Replace("-$1-")),
		new MarkdownChanger("`(.+?)`", new Replace("{{$1}}")),
	};

	@Override
	public String convertMarkdown(String content) {
		String transformedResult = content;
		for (MarkdownChanger changer : CHANGERS) {
			final Pattern pattern = changer.pattern;
			final Function<MatchResult, Optional<String>> replaceFunction = changer.replaceFunction;
			final StringBuffer buffer = new StringBuffer();
			final Matcher matcher = pattern.matcher(transformedResult);
			while(matcher.find()) {
				final Optional<String> replacement = replaceFunction.apply(matcher);
				if(replacement.isPresent()) {
					matcher.appendReplacement(buffer, replacement.get());
				}
			}
			matcher.appendTail(buffer);
			transformedResult = buffer.toString();
		}

		return transformedResult;
	}


	private static class MarkdownChanger {
		private Pattern pattern;
		private Function<MatchResult, Optional<String>> replaceFunction;

		private MarkdownChanger(String regex, Function<MatchResult, Optional<String>> replaceFunction) {
			this.pattern = Pattern.compile(regex);
			this.replaceFunction = replaceFunction;
		}

		private MarkdownChanger(String regex, int flags, Function<MatchResult, Optional<String>> replaceFunction) {
			this.pattern = Pattern.compile(regex, flags);
			this.replaceFunction = replaceFunction;
		}
	}

	private static class Replace implements Function<MatchResult, Optional<String>> {

		private Optional<String> literal;

		private Replace(String literal) {
			this.literal = Optional.fromNullable(literal);
		}

		@Override
		public Optional<String> apply(@Nullable MatchResult input) {
			return literal;
		}
	}
}
