package com.atlassian.jira.plugins.importer.github.config;

import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.plugins.importer.external.beans.ExternalCustomField;
import com.atlassian.jira.plugins.importer.github.fetch.GithubDataService;
import com.atlassian.jira.plugins.importer.imports.config.ValueMappingDefinition;
import com.atlassian.jira.plugins.importer.imports.config.ValueMappingDefinitionsFactory;
import com.atlassian.jira.plugins.importer.imports.config.ValueMappingHelper;
import com.atlassian.jira.plugins.importer.imports.config.ValueMappingHelperImpl;
import com.atlassian.jira.plugins.importer.imports.importer.AbstractConfigBean2;
import com.atlassian.modzdetector.IOUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.codehaus.jackson.map.ObjectMapper;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ConfigBean extends AbstractConfigBean2 {

	public static final String LABEL_TO_ISSUE_TYPE = "labelToIssueType";
	public static final String LABEL_TO_RESOLUTION = "labelToResolution";
	public static final String SCHEME_STATUS = "schemeStatusMapping";

	private final GithubDataService githubDataService;
	private Map<String, Integer> issueTypeMapping = Maps.newHashMap();
	private Map<String, Integer> resolutionMappings = Maps.newHashMap();
	private boolean autoLabels = true;
	private boolean isGithubEnterprise = false;
    private SchemeStatusMapping schemeStatusMapping;
	private String githubUrl;
	private Map<String, IssueType> defaultIssueTypesForProjectKey = Maps.newHashMap();

    public ConfigBean(GithubDataService githubDataService) {
        this.githubDataService = githubDataService;
    }

    @Override
    public List<String> getExternalProjectNames() {
        try {
            return Lists.newArrayList(githubDataService.getAllProjectNames());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public List<String> getSelectedProjects() {
        List<String> selected = new ArrayList<String>();
        for( String projectName : getExternalProjectNames()) {
            if( isProjectSelected(projectName)) {
                selected.add(projectName);
            }
        }
        return selected;
    }

    @Override
    public List<ExternalCustomField> getCustomFields() {
        return Lists.newArrayList();
    }

    @Override
    public List<String> getLinkNamesFromDb() {
        return Lists.newArrayList();
    }

    @Override
    public ValueMappingHelper initializeValueMappingHelper() {
        // this would be better - but not possible if configuration needs to be saved
        // throw new UnsupportedOperationException("not supported");

        // dummy ValueMappingHelper
        final ValueMappingDefinitionsFactory mappingDefinitionFactory = new ValueMappingDefinitionsFactory() {
            public List<ValueMappingDefinition> createMappingDefinitions(ValueMappingHelper valueMappingHelper) {
                return Lists.newArrayList();

            }
        };
        return new ValueMappingHelperImpl(getWorkflowSchemeManager(), getWorkflowManager(), mappingDefinitionFactory, getConstantsManager());
    }

	@Override
	public void copyToNewProperties(Map<String, Object> configFile) {
		super.copyToNewProperties(configFile);
		configFile.put(LABEL_TO_ISSUE_TYPE, issueTypeMapping);
		configFile.put(LABEL_TO_RESOLUTION, resolutionMappings);
		configFile.put(SCHEME_STATUS, schemeStatusMapping);
	}

	@Override
	public void copyFromProperties(InputStream is) throws Exception {
		final byte[] bytes = IOUtils.toByteArray(is);
		super.copyFromProperties(new ByteArrayInputStream(bytes));
		ObjectMapper mapper = new ObjectMapper();
		Map<String, Object> configCopy = mapper.readValue(new ByteArrayInputStream(bytes), Map.class);
		issueTypeMapping = (Map<String, Integer>) configCopy.get(LABEL_TO_ISSUE_TYPE);
		resolutionMappings = (Map<String, Integer>) configCopy.get(LABEL_TO_RESOLUTION);
		schemeStatusMapping = SchemeStatusMapping.fromMap((HashMap<String, Object>) configCopy.get(SCHEME_STATUS));
	}

	public GithubDataService getGithubDataService() {
        return githubDataService;
    }

	public Map<String, Integer> getResolutionMappings() {
		return resolutionMappings;
	}

	public void setResolutionMappings(Map<String, Integer> resolutionMappings) {
		this.resolutionMappings = resolutionMappings;
	}

	public Map<String, Integer> getIssueTypeMapping() {
		return issueTypeMapping;
	}

	public void setIssueTypeMapping(Map<String, Integer> issueTypeMapping) {
		this.issueTypeMapping = issueTypeMapping;
	}

	public boolean isAutoLabels() {
        return autoLabels;
    }

    public void setAutoLabels(boolean autoLabels) {
        this.autoLabels = autoLabels;
    }

	public boolean isGithubEnterprise() {
		return isGithubEnterprise;
	}

	public void setGithubEnterprise(boolean githubEnterprise) {
		isGithubEnterprise = githubEnterprise;
	}

	public SchemeStatusMapping getSchemeStatusMapping() {
        return schemeStatusMapping;
    }

    public void setSchemeStatusMapping(SchemeStatusMapping schemeStatusMapping) {
        this.schemeStatusMapping = schemeStatusMapping;
    }

	public String getGithubUrl() {
		return githubUrl;
	}

	public void setGithubUrl(String githubUrl) {
		this.githubUrl = githubUrl;
	}

	public void setDefaultIssueTypesForProjectKey(Map<String, IssueType> defaultIssueTypesForProjectKey) {
		this.defaultIssueTypesForProjectKey = defaultIssueTypesForProjectKey;
	}

	public IssueType getDefaultIssueType(String projectKey) {
		return defaultIssueTypesForProjectKey.get(projectKey);
	}
}
