package com.atlassian.jira.plugins.importer.github.importer.markup;

import com.atlassian.jira.plugins.importer.github.config.ConfigBean;
import com.atlassian.jira.plugins.importer.github.importer.Attachment;
import com.google.common.collect.Sets;
import org.eclipse.egit.github.core.Repository;

import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class AttachmentsConverter {

	private final ConfigBean configBean;

	public AttachmentsConverter(ConfigBean configBean) {
		this.configBean = configBean;
	}

	public ConversionResult convert(Repository repository, String content) {
		Set<Attachment> attachments = Sets.newHashSet();
		Pattern pattern = Pattern.compile("(!?)\\[(.*?)\\]\\((.*?)(\\s+\\\".*?\\\")?\\)");

		Matcher matcher = pattern.matcher(content);
		while( matcher.find() ) {
			// extract link infos
			final String entireMatch = matcher.group();
			boolean isEmbeddedContent = matcher.group(1).equals("!");
			String linkId = matcher.group(2);
			final String linkTarget = matcher.group(3);

			// JIRA wiki markup format !jira.gif! (see https://jira.atlassian.com/browse/JRA-3669)
			// requires that linkId is a proper file name (i.e. add a file extension)
			if( isEmbeddedContent ) {
				final int beginIndex = linkTarget.lastIndexOf(".");
				String fileType = beginIndex > 0 ? linkTarget.substring(beginIndex) : "";
				linkId = linkId + fileType;
			}

			// decide whether to download content as attachment
			boolean importAttachment = isGithubAttachment(linkTarget) && isEmbeddedContent;
			if(importAttachment) {
				// duplicates will be filtered out with this set (but not if a different linkId was used)
				attachments.add(new Attachment(linkId, linkTarget));
			}

			// replace link markup
			String replacement;
			if(isEmbeddedContent) {
				if( importAttachment ) {
					replacement = "!"+linkId+"|thumbnail!";
				} else {
					replacement = "!"+linkTarget+"!";
				}
			} else {
				replacement = formatNonImageLink(linkId, linkTarget, repository);
			}
			content = content.replace(entireMatch, replacement);
		}

		return new ConversionResult(attachments, content);
	}

	private String formatNonImageLink(String linkId, String linkTarget, Repository repository) {
		if(linkTarget.startsWith(".")) {
			while(linkTarget.startsWith(".")) {
				linkTarget = linkTarget.replaceFirst("\\.", "");
			}
			return String.format("[%s|%s%s]", linkId, repository.getHtmlUrl(), linkTarget);
		}
		return String.format("[%s|%s]", linkId, linkTarget);
	}

	private boolean isGithubAttachment(String linkTarget) {
		if(configBean.isGithubEnterprise()) {
			return linkTarget.contains(configBean.getGithubUrl() + "/github-enterprise-assets");
		} else {
			return linkTarget.contains("://f.cloud.github.com/assets");
		}
	}

}
