package com.atlassian.jira.plugins.importer.github.fetch.auth;

import com.atlassian.jira.plugins.importer.github.fetch.RobustGitHubClient;
import com.atlassian.jira.plugins.importer.github.util.HttpClientFactory;
import com.atlassian.jira.util.I18nHelper;
import com.google.common.base.Optional;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.commons.io.IOUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.auth.BasicScheme;
import org.apache.log4j.Logger;
import org.codehaus.jackson.map.ObjectMapper;
import org.eclipse.egit.github.core.User;
import org.eclipse.egit.github.core.client.GitHubClient;
import org.eclipse.egit.github.core.service.UserService;

import javax.annotation.Nullable;
import javax.ws.rs.core.UriBuilder;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * This implementation manually authenticates with Github Enterprise
 * because Github Java API does not handle Two-factor authentication
 */
public class GithubAuthenticatorImpl implements GithubAuthenticator {

	public static final String IMPORTER_TOKEN_NAME = "JIRA-Github-Importer";
	private final Logger log = Logger.getLogger(this.getClass());

	private final static String GITHUB_CLASSIC_URL = "https://api.github.com";

	private final I18nHelper i18nHelper;
	private final HttpClient client = HttpClientFactory.createHttpClient();

	private String username;
	private String url;
	private String password;
	private String validAuthToken;
	private boolean is2FA = false;
	private boolean isEnterprise = false;
	private int authorizationId = -1;

	public GithubAuthenticatorImpl(I18nHelper i18nHelper) {
		this.i18nHelper = i18nHelper;
	}

	@Override
	public void setGithubEnterpriseUrl(final String url) {
		this.url = url;
		this.isEnterprise = true;
	}

	@Override
	public AuthorizationResponse authenticateGithub(final String username, final String password) {
		this.username = username;
		this.password = password;
		return getAccessToken(null);
	}

	@Override
	public AuthorizationResponse authenticateAndValidateGithubToken(final String token) {
		final GitHubClient gitHubClient = buildAuthenticatedClient(token);
		UserService userService = new UserService(gitHubClient);
		try {
			final User user = userService.getUser();
			if (user != null) {
				this.validAuthToken = token;
				return AuthorizationResponse.ok(token);
			}
		} catch (IOException e) {
			return AuthorizationResponse.error("Invalid token");
		}
		return AuthorizationResponse.error("Invalid token");
	}

	@Override
	public AuthorizationResponse tryAuthenticateWith2FAToken(final String key) throws IllegalStateException {
		return getAccessToken(key);
	}

	private String getText(String key) {
		return i18nHelper.getText(key);
	}

	private String getText(String key, Object... args) {
		return i18nHelper.getText(key, args);
	}

	private AuthorizationResponse getAccessToken(@Nullable String key) {
		final String requestUrl = UriBuilder.fromPath(buildAPIUrl() + "/authorizations").build().toString();

		HttpPost request = null;
		InputStream inputStream = null;
		try {
			request = new HttpPost(requestUrl);
			request.addHeader(BasicScheme.authenticate(new UsernamePasswordCredentials(username, password), "UTF-8", false));
			request.setEntity(new StringEntity(AuthorizationRequest.getDefaultAsJson()));

			if(key != null) {
				request.addHeader("X-GitHub-OTP", key);
			}

			final HttpResponse response = client.execute(request);
			inputStream = response.getEntity().getContent();

			final StatusLine statusLine = response.getStatusLine();
			final int statusCode = statusLine.getStatusCode();
			if(statusCode != HttpURLConnection.HTTP_CREATED) {
				if(statusCode == 422) {
					request.abort();
					return handleTokenAlreadyExists(key);
				}
				if(statusCode == HttpURLConnection.HTTP_UNAUTHORIZED) {
					return handleUnauthorized(response);
				}
				throw new IOException(getText("com.atlassian.jira.plugins.importer.github.error.incorrect.status", statusLine));
			}

			JsonParser parser = new JsonParser();
			final JsonElement jsonRoot = parser.parse(new InputStreamReader(inputStream, "UTF-8"));
			authorizationId = jsonRoot.getAsJsonObject().get("id").getAsInt();
			this.validAuthToken = jsonRoot.getAsJsonObject().get("token").getAsString();
			return AuthorizationResponse.ok(validAuthToken);
		} catch(RuntimeException e) {
			if(request != null) {
				request.abort();
			}
			throw e;
		} catch (IOException e) {
			final String message = getText("com.atlassian.jira.plugins.importer.github.error.cannot.connect", e.getMessage());
			log.error(message, e);
			return AuthorizationResponse.error(message);
		} finally {
			IOUtils.closeQuietly(inputStream);
		}
	}

	private AuthorizationResponse handleTokenAlreadyExists(final String key) throws IOException {
		final String requestUrl = UriBuilder.fromPath(buildAPIUrl() + "/authorizations").build().toString();
		HttpGet request = null;
		InputStream inputStream = null;
		try {
			request = new HttpGet(requestUrl);
			request.addHeader(BasicScheme.authenticate(new UsernamePasswordCredentials(username, password), "UTF-8", false));

			if(key != null) {
				request.addHeader("X-GitHub-OTP", key);
			}

			final HttpResponse response = client.execute(request);
			inputStream = response.getEntity().getContent();
			final String token = getTokenFromAuthorizations(inputStream);
			return AuthorizationResponse.ok(token);
		} catch (RuntimeException e) {
			if(request != null) {
				request.abort();
			}
			throw e;
		} finally {
			IOUtils.closeQuietly(inputStream);
		}
	}

	private String getTokenFromAuthorizations(final InputStream inputStream) throws IOException {
		JsonParser parser = new JsonParser();
		final JsonElement jsonRoot = parser.parse(new InputStreamReader(inputStream, "UTF-8"));
		final JsonArray array = jsonRoot.getAsJsonArray();
		for (int i = 0 ; i < array.size() ; i++) {
			final JsonObject authorization = array.get(i).getAsJsonObject();
			if(!IMPORTER_TOKEN_NAME.equals(authorization.get("note").getAsString())) {
				continue;
			}
			validAuthToken = authorization.get("token").getAsString();
			authorizationId = authorization.get("id").getAsInt();
			return validAuthToken;
		}

		throw new IOException("Cannot find auth token for: " + IMPORTER_TOKEN_NAME);
	}

	private AuthorizationResponse handleUnauthorized(HttpResponse response) {
		final Optional<String> faHeader = get2FAHeader(response);
		if(faHeader.isPresent()) {
			is2FA = true;
			if("app".equals(faHeader.get())) {
				return AuthorizationResponse.actionRequired(AuthState.TWO_FACTOR_AUTH_REQUIRED_APP);
			} else {
				return AuthorizationResponse.actionRequired(AuthState.TWO_FACTOR_AUTH_REQUIRED_SMS);
			}
		} else {
			return AuthorizationResponse.error(getText("com.atlassian.jira.plugins.importer.github.error.incorrect.username.or.password"));
		}
	}

	private String buildAPIUrl() {
		if(isEnterprise) {
			return String.format("%s/api/v3", url);
		} else {
			return GITHUB_CLASSIC_URL;
		}
	}

	private Optional<String> get2FAHeader(HttpResponse response) {
		final Header header = response.getFirstHeader("X-GitHub-OTP");
		if(header == null) {
			return Optional.absent();
		}
		final Iterable<String> tokens = Splitter.on(';').trimResults().split(header.getValue());
		return Optional.of(Iterables.get(tokens, 1));
	}

	private static class AuthorizationRequest {
		public String[] scopes;
		public String note;

		private AuthorizationRequest(String[] scopes, String note) {
			this.scopes = scopes;
			this.note = note;
		}

		public static AuthorizationRequest getDefault() {
			return new AuthorizationRequest(new String[] {"repo"}, IMPORTER_TOKEN_NAME);
		}

		public static String getDefaultAsJson() throws IOException {
			final ObjectMapper mapper = new ObjectMapper();
			return mapper.writeValueAsString(getDefault());
		}
	}

	@Override
	public boolean isTwoFactorAuthenticationEnabled() {
		return is2FA;
	}

	private void deleteAuthorization() {
		final String requestUrl = UriBuilder.fromPath(buildAPIUrl() + "/authorizations/" + authorizationId).build().toString();
		HttpDelete httpDelete = null;
		InputStream is = null;
		try {
			httpDelete = new HttpDelete(requestUrl);
			final HttpResponse response = client.execute(httpDelete);
			if(response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_NO_CONTENT) {
				//Authorization key is not deleted but do we really care?
				log.warn("Cannot delete github authorization with id: " + authorizationId);
			}
			is = response.getEntity().getContent();
		} catch (RuntimeException e) {
			if(httpDelete != null) {
				httpDelete.abort();
			}
			throw e;
		} catch (IOException e) {
			log.warn("Cannot delete github authorization with id: " + authorizationId);
		} finally {
			IOUtils.closeQuietly(is);
		}
	}

	@Override
	public void cleanUp() {
		if(authorizationId > 0) {
			deleteAuthorization();
		}
		this.authorizationId = -1;
		this.username = null;
		this.password = null;
		this.url = null;
		client.getConnectionManager().shutdown();
	}

	@Override
	public GitHubClient buildAuthenticatedClient() {
		return buildAuthenticatedClient(validAuthToken);
	}

	public GitHubClient buildAuthenticatedClient(String authToken) {
		final RobustGitHubClient robustGitHubClient;
		if(url != null) {
			try {
				final URL objectUrl = new URL(url);
				robustGitHubClient = new RobustGitHubClient(objectUrl.getHost(), objectUrl.getPort(), objectUrl.getProtocol());
			} catch (MalformedURLException e) {
				throw new RuntimeException(e);
			}
		} else {
			robustGitHubClient = new RobustGitHubClient();

		}
		robustGitHubClient.setOAuth2Token(authToken);
		return robustGitHubClient;
	}
}