define('jira-agile/rapid/ui/work/work-view', ["jira-agile/rapid/ui/work/work-selection-controller", "jira-agile/rapid/work/work-controls", "jira-agile/rapid/ui/work/work-controller", "jira-agile/rapid/ui/work/swimlane-view", "jira-agile/rapid/ui/work/issue-list-util", "jira-agile/rapid/ui/work/grid-data-controller", "jira-agile/rapid/layout", "jquery", "jira-agile/rapid/ui/kanplan/kanplan-acknowledge", 'jira/featureflags/feature-manager'], function (WorkSelectionController, WorkControls, WorkController, SwimlaneView, IssueListUtil, GridDataController, Layout, $, KanPlanAcknowledge, FeatureFlagManager) {
    "use strict";

    var RAPID_BOARD_BANDS = "com.atlassian.jira.agile.darkfeature.rapid.boards.bands";
    var FLEXIBLE_BOARDS = "com.atlassian.jira.agile.darkfeature.flexible.boards";

    /**
     * Work Views. Handles the work view rendering
     */

    var WorkView = {};

    WorkView.isVisible = false;

    WorkView.detailViewOpen = false;

    WorkView.isRendering = false;

    WorkView.init = function () {
        // init resize handler for updating the the details view upon resize events
        $(GH).bind(Layout.EVENT_DELAYED_RESIZE, WorkView.handleResizeEvent);

        $(document).delegate('.js-view-actions-work .js-view-action-toggle-epics-lozenge', 'click', WorkController.toggleEpicsShowOnWorkboard);
    };

    /**
     * Shows the work view
     */
    WorkView.show = function () {
        WorkView.isVisible = true;

        // show the loading transition
        WorkView.showLoading();

        // create a skeleton DOM for widgets
        var skeleton = GH.tpl.workcontroller.renderSkeleton();
        $('#ghx-work').empty().append(skeleton);

        // Configure the detail view
        GH.DetailsView.setContainerSelector('#ghx-detail-view');
        GH.DetailsView.setOptions({
            canClose: true,
            showActionsCog: true,
            showSubtaskTab: true,
            showLogWorkTrigger: true,
            closeHandler: WorkController.closeDetailsView
        });

        if (GH.RapidBoard.State.isKanbanBoard()) {
            WorkView.showKanPlanAcknowledge();
        }
    };

    /**
     *
     * @returns {jQuery}
     */
    WorkView.getFirstColumnHeader = function () {
        return $('#ghx-column-headers .ghx-column').first();
    };

    WorkView.hide = function () {
        WorkView.isVisible = false;

        // hide the loading transition
        WorkView.hideLoading();

        // hide the detail view, as it will have to be properly positioned before displaying
        $('#ghx-detail-view').css({ 'display': 'none' });

        // empty our DOM
        $('#ghx-work').empty();

        WorkView.hideKanPlanAcknowledge();
    };

    /**
     * Configures the detail view column displayed on the swimlane view
     * @param detailViewIssueKey the selected key or false if none selected
     */
    WorkView.configureUI = function (detailViewIssueKey) {
        // is the detail view opened?
        WorkView.detailViewOpen = !!detailViewIssueKey;

        // set the selected issue key in the detail view
        GH.DetailsView.setSelectedIssueKey(detailViewIssueKey);
    };

    /**
     * Renders the pool and detail view
     */
    WorkView.renderPoolAndDetailView = function () {
        WorkView.isRendering = true;

        // hide the spinner
        WorkView.hideLoadingPool();

        // render the swimlane view
        SwimlaneView.show();

        WorkView.updateAllSizings();

        // render the detail view at the beginning only if necessary
        if (WorkView.detailViewOpen) {
            WorkView.updateDetailView();
        }

        WorkView.isRendering = false;
    };

    /**
     * Scrolls the selected issue into view
     */
    WorkView.scrollIssueToView = function () {
        // ensure the issue is visible
        var issueKey = WorkSelectionController.getSelectedIssueKey();
        if (issueKey) {
            WorkSelectionController.gotoIssue(issueKey);
        }
    };

    /**
     * Opens/collapses the detail view
     */
    WorkView.openOrCloseDetailView = function (selectedIssueKey) {
        // configure the ui
        WorkView.configureUI(selectedIssueKey);

        // update the structure
        WorkView.updateAllSizings();

        // update the detail view
        WorkView.updateDetailView();
    };

    /**
     * Updates all columns marked as fixed to ensure they fit within the content view area
     */
    // TODO: hack! I have no idea what else we should call...
    WorkView.updateFixedElementPositioning = function () {
        // if not visible don't do anything
        if (!WorkView.isVisible) {
            return;
        }

        // update Quick Filter toggle spacing
        WorkControls.resizeQuickFilters();

        GH.DetailsView.updateSize();
    };

    /**
     * Updates the issue card layout and the detail view sizing
     */
    WorkView.updateAllSizings = function () {
        WorkView.updateIssueCardLayout();
        WorkView.updateDetailViewVisibilityAndSize();
    };

    /**
     * Updates the detail view
     */
    WorkView.updateDetailView = function () {
        if (WorkView.skipDetailsView) {
            delete WorkView.skipDetailsView;
            return;
        }
        // show/hide the detail view
        if (WorkView.detailViewOpen) {
            GH.DetailsView.show();
        } else {
            GH.DetailsView.hide();
        }
    };

    WorkView.skipNextDetailViewUpdate = function () {
        WorkView.skipDetailsView = true;
    };

    /**
     * Get a estimated column width
     * @return {number}
     */
    WorkView.getEstimatedColumnWidth = function () {

        var fullWidthAvailable = $('#ghx-work').width();
        var widthUsedByDetailsView = WorkView.detailViewOpen ? GH.DetailsView.minWidth : 0;
        var model = GridDataController.getModel();
        var colCount = model.getColumns().length;
        var widthUsedByGutters = (colCount + 1) * 10 + 20; // magic! each column has border-spacing (10) and the pool has a gutter (20)
        var colWidth = Math.floor((fullWidthAvailable - widthUsedByDetailsView - widthUsedByGutters) / colCount);

        return colWidth;
    };

    /**
     * Magic! Can we expect, that we have to use largest band
     */
    WorkView.isSupposedLargeBand = function () {
        return window.innerWidth > 1024 && // can't expect on small window size, 1024px is official support
        GridDataController.getModel().getColumns().length <= 5 && // too risky to expect for large column amount
        !WorkView.detailViewOpen && // space taken by Detail View if open
        $('.aui-sidebar').attr('aria-expanded') === 'false'; // space taken by sidebar if open
    };

    /**
     * Updates the min pool min height settings
     */
    WorkView.updateMinHeightsSettings = function () {
        if (WorkView.detailViewOpen) {
            $('#ghx-work').addClass('ghx-has-detail');
            $('#ghx-detail-view').css({ 'display': '' });
        } else {
            // hide the view
            $('#ghx-detail-view').css({ 'display': 'none' });

            $('#ghx-work').removeClass('ghx-has-detail');
        }
    };

    /**
     * Layout is divided into bands based on available width
     * @returns {number} - 1, 2
     */
    WorkView.getBand = function () {
        // We are doing it this in order to try avoid getEstimatedColumnWidth
        // for performance reasons connected to reflow
        if (WorkView.isSupposedLargeBand()) {
            return 2;
        }

        var columnWidth = WorkView.getEstimatedColumnWidth();
        return columnWidth <= 145 ? 1 : 2;
    };

    /**
     * Updates the detail view visibility, that is, shows/hides the detail view div as well as takes care of
     * keeping the detail view fixed on the screen
     */
    WorkView.updateDetailViewVisibilityAndSize = function () {
        // show if not yet visible
        var detailsContainer = GH.DetailsView.getContainer();
        if (WorkView.detailViewOpen) {

            // ensure the view is shown
            detailsContainer.css({ 'display': '' });

            GH.DetailsView.updateSize();
        } else {
            // hide the view
            detailsContainer.css({ 'display': 'none' });
        }
    };

    /**
     * Updates the issue card layout according to available size
     */
    WorkView.updateIssueCardLayout = function () {
        if (WorkView.usesOldLayout() && FeatureFlagManager.isFeatureEnabled(RAPID_BOARD_BANDS)) {
            var $pool = $('#ghx-pool');
            var band = WorkView.getBand();
            $pool.toggleClass('ghx-band-1', band === 1).toggleClass('ghx-band-2', band === 2);
        }
    };

    WorkView.usesOldLayout = function () {
        var isInGadget = self !== top;
        var isIE = $.browser.msie;
        var flexibleBoardsDisabled = FeatureFlagManager.isFeatureEnabled(FLEXIBLE_BOARDS);
        return isInGadget || isIE || flexibleBoardsDisabled;
    };

    /**
     * CSS Class for View Loading Transition
     */
    WorkView.showLoading = function () {
        $('body').addClass('ghx-loading-pool ghx-loading-quickfilter');
    };

    /**
     * CSS Class Removal for View Loading Transition
     */
    WorkView.hideLoading = function () {
        $('body').removeClass('ghx-loading-pool ghx-loading-quickfilter');
    };

    /**
     * CSS Class Removal for Loading Transition on QF
     */
    WorkView.hideLoadingFilter = function () {
        $('body').removeClass('ghx-loading-quickfilter');
    };

    /**
     * CSS Class for Pool Loading Transition
     */
    WorkView.showLoadingPool = function () {
        $('body').addClass('ghx-loading-pool');
        $('.ghx-throbber').spin('large');
    };

    /**
     * CSS Class Removal for Pool Loading Transition
     */
    WorkView.hideLoadingPool = function () {
        $('body').removeClass('ghx-loading-pool');
        $('.ghx-throbber').spinStop();
    };

    WorkView.handleResizeEvent = function () {
        // no-op if we are not displayed
        if (!WorkView.isVisible) {
            return;
        }

        // also skip if the model hasn't been set yet - IE triggers resize events too early
        if (!GridDataController.getModel()) {
            return;
        }

        // skip if we are rendering
        if (WorkView.isRendering) {
            return;
        }

        // update the structure
        WorkView.updateAllSizings();
    };

    WorkView.showKanPlanAcknowledge = function () {
        KanPlanAcknowledge.show();
    };

    WorkView.hideKanPlanAcknowledge = function () {
        KanPlanAcknowledge.hide();
    };

    return WorkView;
});

AJS.namespace('GH.WorkView', null, require('jira-agile/rapid/ui/work/work-view'));