define('jira-agile/rapid/ui/work/work-selection-controller', ["jira-agile/rapid/ui/work/work-controller", "jira-agile/rapid/ui/work/work-context-menu-controller", "jira-agile/rapid/ui/work/swimlane-view", "jira-agile/rapid/ui/work/ranking-model", "jira-agile/rapid/ui/work/grid-data-controller", "jira-agile/rapid/logger", "underscore", "jquery"], function (WorkController, WorkContextMenuController, SwimlaneView, RankingModel, GridDataController, Logger, _, $) {
    "use strict";

    /**
     * Issue functions
     *
     * Handles issue functionality
     */

    /** Renders the columns for the given model. */

    var WorkSelectionController = {};

    /** options not to scroll the issue into view. expensive operation. */
    WorkSelectionController.OPT_NO_SCROLL = { 'scrollToView': false };

    WorkSelectionController.selectionDataModel = {};

    /**
     * Selection manager object
     */
    WorkSelectionController.selectionManager = undefined;

    /**
     * Called upon page initialization
     */
    WorkSelectionController.init = function () {
        WorkSelectionController.initSelectionManager();

        // issue selection
        $(document).delegate('#ghx-pool .js-issue', 'click', WorkSelectionController.handleIssueClick);
        $(document).delegate('#ghx-pool .js-issue', 'focus', WorkSelectionController.handleFocus);
        // issue context menu
        $(document).delegate('#ghx-pool .js-issue', 'contextmenu', WorkSelectionController.handleIssueContextMenu);
        // swimlane header selection
        $(document).delegate('.ghx-swimlane-header', 'click', WorkSelectionController.handleSwimlaneClick);
        $(document).delegate('.ghx-swimlane-header', 'focus', WorkSelectionController.handleFocus);
        $(document).delegate('.ghx-swimlane-header', 'contextmenu', WorkSelectionController.handleSwimlaneContextMenu);
    };

    WorkSelectionController.initSelectionManager = function () {
        // create a proxy model object that binds the selectionManager to the underlying data model
        WorkSelectionController.selectionDataModel = {
            isIssueValid: function isIssueValid(issueKey) {
                return GridDataController.getModel().isIssueValid(issueKey);
            },
            getIssueIndex: function getIssueIndex(issueKey) {
                // TODO: this is actually really only used to order issues, move the ordering function into the model,
                // and call the interface getIssueKeysInOrder : function(issueKeys) instead
                // for now just return the index in a cell (which is what selections are limited to anyways)
                return GridDataController.getModel().getIssueIndexByKey(issueKey);
            },
            canAddToSelection: function canAddToSelection(selectedIssueKeys, issueKey) {
                // must be in the same cell as the first issue and compatible (parent-parent, children of same parent)
                // TODO: move function into here?
                return WorkSelectionController.canAddToSelection(selectedIssueKeys, issueKey);
            },
            getIssueRange: function getIssueRange(fromKey, toKey) {
                return RankingModel.getRankableIssuesRange(fromKey, toKey);
            }
        };

        // create the selection manager used by this object
        WorkSelectionController.selectionManager = new GH.IssueSelectionManager(WorkSelectionController.selectionDataModel, 'work.selectedIssues');
    };

    //
    // work specific logic
    //

    /**
     * Validate the current issue selection.
     */
    WorkSelectionController.validateCurrentSelection = function validateCurrentSelection() {
        // check whether the current selection is still valid
        var outcome = WorkSelectionController.selectionManager.validateCurrentSelection();

        // update if something in the selection has changed
        if (outcome) {
            // ensure the detail view state is not opened if the selection was invalid
            if (!WorkSelectionController.selectionManager.hasSelection() && WorkController.isDetailsViewOpened()) {
                // set the detail view state to closed
                GH.RapidBoard.State.setWorkViewDetailsViewOpened(false);
            }

            // the current url state is most probably invalid, so replace it with the new selection
            GH.RapidBoard.State.replaceState();

            // update the ui
            WorkSelectionController.updateUIAndState();
        }
    };

    /**
     * Go to the given issue.
     * This will select the issue if not yet selected, otherwise simply scroll the issue into view
     */
    WorkSelectionController.gotoIssue = function (issueKey) {
        if (!issueKey) {
            return;
        }
        var selectedKey = WorkSelectionController.getSelectedIssueKey();
        if (issueKey != selectedKey) {
            WorkSelectionController.selectIssue(issueKey);
        } else {
            var elem = $('.ghx-columns').find('.js-issue[data-issue-key="' + issueKey + '"]');
            if (elem.length === 0) {
                // is it a swimlane?
                elem = $("#ghx-pool").find(".ghx-swimlane-header[data-issue-key='" + issueKey + "']");
            }

            if (elem.length !== 0) {
                WorkSelectionController._scrollToView(elem);
            }
        }
    };

    WorkSelectionController.focusOnIssue = function (issueKey) {
        var elem = $('.ghx-columns').find('.js-issue[data-issue-key="' + issueKey + '"]');
        if (elem.length === 0) {
            // is it a swimlane?
            elem = $("#ghx-pool").find(".ghx-swimlane-header[data-issue-key='" + issueKey + "']");
        }
        if (elem.length !== 0) {
            elem.focus();
        }
    };

    /**
     * Selects an issue
     */
    WorkSelectionController.selectIssue = function (issueKey, options) {
        // merge default and passed in options
        var defaultOptions = {
            pushState: true,
            scrollToView: true
        };
        options = _.extend(defaultOptions, options);

        // update the model
        WorkSelectionController.selectionManager.selectIssue(issueKey);

        // push state if requested
        if (options.pushState) {
            GH.RapidBoard.State.pushState();
        }

        // update the ui
        if (!options.dontUpdateUI) {
            WorkSelectionController._updateSelectionUI(options.scrollToView ? issueKey : null);
        }
    };

    WorkSelectionController.handleSwimlaneClick = function (event) {
        // ignore if the expander is clicked
        if ($(event.target).parents().andSelf().is('.js-expander')) {
            return;
        }

        // is this an issue swimlane?
        var issueKey = WorkSelectionController.getIssueKey($(this));
        if (issueKey) {
            WorkSelectionController.handleIssueClick.call(this, event);
        } else {
            // select first issue inside the swimlane
            var swimlaneId = parseInt($(this).attr('data-swimlane-id'), 10);
            var model = GridDataController.getModel();
            var swimlane = model.getSwimlaneById(swimlaneId);
            var columns = model.getColumns();
            _.find(columns, function (column) {
                issueKey = model.getIssueKeyForColumnAtIndex(swimlane.id, column.id, 0);
                return !!issueKey;
            });
            if (issueKey) {
                WorkSelectionController.selectIssue(issueKey);
            }
        }
    };

    /**
     * Handles issue click events
     */
    WorkSelectionController.handleIssueClick = function (event) {
        // fetch the key for the given issue
        var issueKey = WorkSelectionController.getIssueKey($(this));
        var isTargetIssueKey = $(event.target).hasClass('js-key-link');
        var shouldOpenInNewTab = !GH.MouseUtils.openInSameTab(event);

        if (isTargetIssueKey && shouldOpenInNewTab) {
            return true;
        }

        //prevent any default browser click actions on the cards
        event.preventDefault();
        // handle issue selection click
        WorkSelectionController.handleIssueSelectionEvent(event, issueKey, true);
    };

    WorkSelectionController.handleSwimlaneContextMenu = function (event) {
        // only show context menu if it's an issue
        // note: epic swimlane headers don't have an issue key
        var issueKey = $(event.target).closest(".ghx-swimlane-header").attr("data-issue-key");
        if (issueKey) {
            var inSelection = WorkSelectionController.isInSelection(issueKey);
            if (!inSelection) {
                // pass through the context menu action as if it was a normal click.
                WorkSelectionController.handleSwimlaneClick.call(this, event);
            }
            WorkContextMenuController.showContextMenuAt(event, { swimlane: true });
        }
    };

    /**
     * Handles the context menu events fired on issues
     */
    WorkSelectionController.handleIssueContextMenu = function (event) {
        // When right clicking a non selected issue, it should become selected before showing the context menu
        var issueKey = WorkSelectionController.getIssueKey($(this));
        var inSelection = WorkSelectionController.isInSelection(issueKey);
        if (!inSelection) {
            // pass through the context menu action as if it was a normal click.
            WorkSelectionController.selectIssue(issueKey);
        }
        WorkContextMenuController.showContextMenuAt(event, {});
    };

    /**
     * Handles an issue selection event
     */
    WorkSelectionController.handleIssueSelectionEvent = function (event, issueKey, openDetailView) {
        var metaKeys = GH.MouseUtils.getMetaKeys(event);
        var outcome = WorkSelectionController.selectionManager.handleIssueClick(metaKeys, issueKey);
        var isMultiSelect = GH.IssueSelectionManager.getMultiSelectType(metaKeys);
        if (isMultiSelect) {
            openDetailView = false;
        }
        if (outcome) {
            // update the selection
            WorkSelectionController._updateSelectionUI(issueKey);

            // open the detail view if requested (and not yet open) - this will also push the state
            if (openDetailView && WorkSelectionController.selectionManager.hasSelection() && !WorkController.isDetailsViewOpened()) {
                WorkController.openDetailsView(issueKey);
            } else {
                // otherwise simply update the url state
                GH.RapidBoard.State.pushState();
            }
        }
    };

    /**
     * Handles an issue focus event
     */
    WorkSelectionController.handleFocus = function (event) {
        var issueKey = WorkSelectionController.getIssueKey($(this));
        var inSelection = WorkSelectionController.isInSelection(issueKey);
        if (!inSelection) {
            WorkSelectionController.handleIssueSelectionEvent(event, issueKey, false);
        }
    };

    /**
     * Updates the ui and state according to the new selection
     */
    WorkSelectionController.updateUIAndState = function () {
        // update the ui
        WorkSelectionController._updateSelectionUI();

        // update the url state
        GH.RapidBoard.State.pushState();
    };

    /**
     * Updates the selection UI
     */
    WorkSelectionController._updateSelectionUI = function (scrollToIssueKey) {
        var scrollToViewElem = null;
        $('.js-issue, .ghx-swimlane-header, #ghx-swimname').removeClass('ghx-selected ghx-selected-primary').each(function (index, elem) {
            var $elem = $(elem);
            var issueKey = $elem.attr('data-issue-key');
            if (!issueKey) {
                return;
            }

            if (WorkSelectionController.isInSelection(issueKey)) {
                $elem.addClass('ghx-selected');
                if (issueKey == selectedIssueKey) {
                    $elem.addClass('ghx-selected-primary');
                }
            }
            if (scrollToIssueKey != null && issueKey == scrollToIssueKey) {
                scrollToViewElem = $elem;
            }
        });

        // expand the swimlane to make the issue visible if currently collapsed (only if it is not a swimlane)
        if (scrollToViewElem && !scrollToViewElem.hasClass('ghx-swimlane-header')) {
            SwimlaneView.ensureSwimlaneExpandedForIssueKey(scrollToIssueKey);
        }

        // scroll the currently selected issue to view
        if (scrollToViewElem) {
            WorkSelectionController._scrollToView(scrollToViewElem);
        }

        // fire an event to inform others that the selection changed
        var issueKey = WorkSelectionController.getSelectedIssueKey();
        var selectedIssueKey = GridDataController.getModel().isIssueValid(issueKey) ? issueKey : undefined;
        Logger.log("issueSelected triggered", Logger.Contexts.ui);
        $(GH).trigger('issueSelected', selectedIssueKey);
    };

    /**
     * Ensures the passed issue element is fully visible
     */
    WorkSelectionController._scrollToView = function ($elem) {
        var $pool = $('#ghx-pool');
        var $header = $('#ghx-column-header-group');

        // bypass jQuery which is a pain for that kind of work. getBoundingClientRect is exactly what we need and is supported on all our supported browsers.
        // we assume that the document is not scrollable by design, so we don't have to manage document scroll offset
        var headerRect = $header[0].getBoundingClientRect();
        var poolRect = $pool[0].getBoundingClientRect();
        var elementRect = $elem[0].getBoundingClientRect();

        GH.ScrollUtils.scrollElementToViewPort($pool, { top: headerRect.bottom, bottom: poolRect.bottom }, elementRect);
    };

    /**
     * Ensure that when an event happens to a particular issue, that the issue is selected.
     * @param event
     * @param issueKey
     */
    WorkSelectionController.ensureIssueSelected = function (event, issueKey) {
        // if the dragged issue is not selected, we want to select it
        if (!WorkSelectionController.selectionManager.isInSelection(issueKey)) {
            // let the selection controller handle the click if the issue is compatible with the current selection
            if (WorkSelectionController.canAddToSelection(WorkSelectionController.getSelectedIssueKeys(), issueKey)) {
                // treat as if this were a normal click event
                WorkSelectionController.handleIssueSelectionEvent(event, issueKey);
            } else {
                // otherwise force a new selection
                WorkSelectionController.selectIssue(issueKey, WorkSelectionController.OPT_NO_SCROLL);
            }
        }
    };

    //
    // Getters that return data contained in the selection manager
    //

    /**
     * Returns all the currently selected issues.
     */
    WorkSelectionController.getSelectedIssueKeys = function () {
        return WorkSelectionController.selectionManager.getSelectedIssueKeys();
    };

    /** Get the main selected issue (this is the one that has been last selected) */
    WorkSelectionController.getSelectedIssueKey = function () {
        return WorkSelectionController.selectionManager.getSelectedIssueKey();
    };

    /** Is a given issue in the selection */
    WorkSelectionController.isInSelection = function (issueKey) {
        return WorkSelectionController.selectionManager.isInSelection(issueKey);
    };

    /**
     * Called by PlanUrlState to set the selection as defined in the url
     */
    WorkSelectionController.setIssueFromUrlState = function (issueKey) {
        WorkSelectionController.selectionManager.selectIssue(issueKey);
    };

    /**
     * Returns all the currently selected issues in order of rank.
     */
    WorkSelectionController.getSelectedIssuesInOrder = function () {
        return WorkSelectionController.selectionManager.getSelectedIssuesInOrder();
    };

    /**
     * Can the issueKey be added to the selection.
     */
    WorkSelectionController.canAddToSelection = function (selectedIssueKeys, issueKey) {
        // if there are no issues, adding is always allowed
        if (_.isEmpty(selectedIssueKeys)) {
            return true;
        }

        // ensure the issue is compatible with the current selection
        var firstKey = _.first(selectedIssueKeys);
        return RankingModel.canIssuesBeRankedTogether(firstKey, issueKey);
    };

    // UI stuff

    /**
     * Get the id of an issue given its element
     */
    WorkSelectionController.getIssueId = function (elem) {
        return elem.closest('.js-issue, .ghx-issue-filtered, .ghx-parent-group, .ghx-swimlane-header').attr('data-issue-id');
    };
    WorkSelectionController.getIssueKey = function (elem) {
        return elem.closest('.js-issue, .ghx-issue-filtered, .ghx-parent-group, .ghx-swimlane-header').attr('data-issue-key');
    };

    return WorkSelectionController;
});

AJS.namespace('GH.WorkSelectionController', null, require('jira-agile/rapid/ui/work/work-selection-controller'));