/**
 * Utility functions for issue lists
 * @module jira-agile/rapid/ui/work/issue-list-util
 * @requires module:underscore
 */
define('jira-agile/rapid/ui/work/issue-list-util', [
    'underscore',
    'jquery'
], function (
    _,
    $
) {

var issueListUtil = {};
/**
 * Calculates the group data for all passed list of issues
 * @param list an array of array of issues
 */
issueListUtil.calculateGroupData = function(list) {
    var data = {
        fakeParents : {},
        beginGroups: {},
        endGroups: {}
    };
    _.each(list, function(issues) {
        GH.IssueListUtil._calculateGroupData(data, issues);
    });
    return data;
};

/**
 * Calculates the group data for a single list of issues
 */
issueListUtil._calculateGroupData = function(data, issues) {
    var subTaskIssue = null, previousIssue = null;
    _.each(issues, function(issue) {
        // Set the beginGroup, fakeParent and endGroup flags, used to render parent/subtask groups
        if (issue.parentId) {
            // only set first if we are currently not in a subtask case already
            if (! subTaskIssue) {
                if (previousIssue && previousIssue.id == issue.parentId) { // parent is available, mark it as container
                    data.beginGroups[previousIssue.id] = true;
                } else {
                    // no parent available, insert placeholder parent
                    data.fakeParents[issue.id] = true;
                }
                data.beginGroups[issue.id] = true;
            } else if (subTaskIssue.parentId != issue.parentId) {
                data.endGroups[subTaskIssue.id] = true; // close previous subtask
                data.beginGroups[issue.id] = true;
                data.fakeParents[issue.id] = true;
            }
            subTaskIssue = issue;
        } else {
            if (subTaskIssue) {
                data.endGroups[subTaskIssue.id] = true;
                subTaskIssue = null;
            }
        }
        previousIssue = issue;
    });

    if (subTaskIssue) {
        data.endGroups[subTaskIssue.id] = true;
    }
};

/**
 * Calculates the display issue key for a given issue key taking into account the width of the column
 */
issueListUtil.buildDisplayIssueKey = function(issueKey, issueKeyDimensions) {
    if (issueKeyDimensions.availableWidth == null || !issueKeyDimensions.approxCharacterWidth) {
        return issueKey;
    }

    var ELLIPSIS = '\u2026';
    var issueKeySplit = issueKey.split('-');
    var issueName = issueKeySplit[0];
    var issueNumber = issueKeySplit[1];
    var availableLength = Math.floor(issueKeyDimensions.availableWidth / issueKeyDimensions.approxCharacterWidth);

    // only reduce if key needs shortening
    if (availableLength < issueKey.length) {
        // reduce the name to allow space for the number
        var issueNameLength = availableLength - issueNumber.length - 2; // subtract 1 for dash and 1 for ellipsis character
        if (issueNameLength < 0) {
            // Can't fit the number either
            return ELLIPSIS;
        }
        issueName = issueName.substr(0, issueNameLength);
        return issueName + ELLIPSIS + '-' + issueNumber;
    } else {
        return issueKey;
    }
};

/**
  * Calculates the display issue key object for a given issue key
  */
issueListUtil.buildDisplayIssueKeyObj = function(issueKey) {
    const separatorPos = issueKey.lastIndexOf('-');
    return {
        projectKey: issueKey.substring(0, separatorPos),
        issueNumber: issueKey.substring(separatorPos)
    };
};

return issueListUtil;
});

AJS.namespace('GH.IssueListUtil', null, require('jira-agile/rapid/ui/work/issue-list-util'));