/* global GH */
define('jira-agile/rapid/ui/work/work-drag-and-drop', [
    'jira/featureflags/feature-manager',
    'jira-agile/rapid/analytics-tracker',
    'jira-agile/rapid/events',
    'jira-agile/rapid/ui/notification',
    'jira-agile/rapid/ui/work/work-transition-service',
    'jira/analytics',
    'jquery'
], function (FeatureManager, AnalyticsTracker, Events, Notification, WorkTransitionService, Analytics, $) {
    "use strict";

    /**
     * RapidBoard drag and drop functionality
     */
    const WorkDragAndDrop = new Events();

    WorkDragAndDrop.OPTIMISTIC_TRANSITIONS_FLAG = "com.atlassian.jira.agile.darkfeature.optimistic.transitions";

    /**
     * Dragged dom element
     */
    WorkDragAndDrop.currentDragElement = null;

    /**
     * True if the drag was canceled (e.g. through esc key). Drop/stop handlers check this flag.
     * @type {boolean}
     */
    WorkDragAndDrop.cancel = false;

    /**
     * Is DnD enabled? Used to differentiate between normal and gadget mode (which is non-interactive)
     * @type {boolean}
     */
    WorkDragAndDrop.enabled = false;

    /**
     * Copied from plan mode, we need this variable to detect and cancel a pending rank operation (sortable) if the user
     * actually performed a column transition (drag and drop to another column).
     */
    WorkDragAndDrop.skipNextRank = false;


    WorkDragAndDrop.analytics = {
        dragAndDrop: new AnalyticsTracker('gh.rapidboard.work.dragAndDrop'),
        ranking: new AnalyticsTracker('gh.rapidboard.rankissues'),
        transitioning: new AnalyticsTracker('gh.rapidboard.transitionissues')
    };

    /**
     * Called on page load (but not for gadgets)
     */
    WorkDragAndDrop.init = function () {
        WorkDragAndDrop.enabled = true;
    };

    /**
     * Initializes all draggables (cards).
     * Delayed to allow event loop to complete and a redraw to happen beforehand as this operation is rather expensive
     */
    WorkDragAndDrop.initDraggables = function (container) {
        if (WorkDragAndDrop.enabled) {
            setTimeout(function () {
                WorkDragAndDrop._registerDraggables(container);
            }, 0);
        }
    };

    /**
     * Registers all cards as draggables / rankables
     */
    WorkDragAndDrop._registerDraggables = function (container) {
        var canUserRank = GH.RankingModel.canUserRank();
        container.find('.ghx-columns').each(function (index, rowRaw) {
            var row = $(rowRaw);
            if (canUserRank) {
                row.find(".ghx-column").sortable({
                    revert: 0,
                    zIndex: 3000,
                    start: WorkDragAndDrop.issueDragStart,
                    stop: WorkDragAndDrop.issueDragStop,
                    items: '.js-parent-drag',
                    // don't drag on the fake parent and the subtask group (activates in IE8 only)
                    cancel: '.js-fake-parent, .ghx-parent-group > .ghx-subtask-group',
                    update: WorkDragAndDrop.updateIssueHandler
                }).find('.ghx-subtask-group').sortable({
                    revert: 0,
                    zIndex: 3000,
                    start: function (event, ui) {
                        WorkDragAndDrop.issueDragStart.apply(this, [event, ui]);
                        var parent = $(ui.item).parent();
                        if (parent.children().length > 2) { // 2 includes the drag helper
                            parent.addClass('ghx-contain');
                        }
                    },
                    stop: function (event, ui) {
                        WorkDragAndDrop.issueDragStop.apply(this, [event, ui]);
                        $(ui.item).parent().removeClass('ghx-contain');
                    },
                    items: '.ghx-issue-subtask',
                    update: WorkDragAndDrop.updateIssueHandler
                });
            } else {
                WorkDragAndDrop._initIssueCardsDraggable(row.find('.js-issue'), row);
            }
        });
    };

    WorkDragAndDrop.registerSingleCardDraggable = function ($issue) {
        var canUserRank = GH.RankingModel.canUserRank();
        if (!canUserRank) {
            var $container = $issue.closest('.ghx-columns');
            WorkDragAndDrop._initIssueCardsDraggable($issue, $container);
        }
    };

    WorkDragAndDrop._initIssueCardsDraggable = function ($issues, $container) {
        $issues.draggable({
            revert: true,
            revertDuration: 0,
            zIndex: 3000,
            containment: $container,
            start: WorkDragAndDrop.issueDragStart,
            stop: WorkDragAndDrop.issueDragStop
        });
    };

    /**
     * Handles the update event of sortables which are to be ranked.
     */
    WorkDragAndDrop.updateIssueHandler = function (event, ui) {
        if (WorkDragAndDrop.skipNextRank) {
            // "this" here is the sortable container (that is also the draggable in that context) on which we want to cancel
            // the sortable operation
            $(this).sortable('cancel');
            return;
        }

        // if we're in a cancel, fail fast (strangely, this function still gets called when we call cancel...
        if (WorkDragAndDrop.cancel) {
            return;
        }

        var draggedIssue = $(ui.item);
        var issueKeys = GH.WorkSelectionController.getSelectedIssuesInOrder();
        var prevIssueKey = GH.WorkSelectionController.getIssueKey(draggedIssue.prev('.js-issue, .ghx-parent-group'));
        var nextIssueKey = GH.WorkSelectionController.getIssueKey(draggedIssue.next('.js-issue, .ghx-parent-group'));

        // do analytics before we even know if they are successful
        WorkDragAndDrop.analytics.ranking.trigger('dragAndDrop', issueKeys.length); // SAFE

        // do the ranking
        GH.WorkRanking.rankIssues(issueKeys, prevIssueKey, nextIssueKey);
    };

    /**
     * Called when issue dragging starts.
     */
    WorkDragAndDrop.issueDragStart = function (event, ui) {
        // reset flags
        WorkDragAndDrop.cancel = false;
        WorkDragAndDrop.skipNextRank = false;

        GH.WorkContextMenuController.hideContextMenu();

        // distinguish between sortable (ui.item) and draggable (this)
        var isSortable = ui.item ? true : false;
        var card = isSortable ? $(ui.item) : $(this);

        // Fix the height of the placeholder card
        $(ui.placeholder).height(card.height());

        var container = $(this);
        // bind a keyhandler for esc
        $(document).bind('keydown.dragAndDrop', function (event) {
            if (event.keyCode == 27) {
                WorkDragAndDrop.cancel = true;
                isSortable ? container.sortable('cancel') : container.draggable('cancel');
                WorkDragAndDrop.analytics.dragAndDrop.trigger('cancel');
            }
        });

        // Keep reference to draggable
        WorkDragAndDrop.currentDragElement = card;

        // ensure the issue is selected
        var issueId = GH.WorkSelectionController.getIssueId(card);
        var issueKey = GH.WorkSelectionController.getIssueKey(card);

        // ensure that the dragged issue is selected
        GH.WorkSelectionController.ensureIssueSelected(event, issueKey);

        // used in webdriver tests
        WorkDragAndDrop.droppablesReady = false;
        // destroy all droppables, they may be around from previous drag ops.
        if ($('.ghx-column').data("ui-droppable")) {
            $('.ghx-column').droppable('destroy');
        }

        // fetch selected issues,
        var selectedIssues = GH.WorkSelectionController.getSelectedIssueKeys();

        // kick off possible transitions querying (if we only have a single selected issue)
        if (selectedIssues.length === 1) {
            // initialize the overlay now
            WorkDragAndDrop.initDropZoneOverlay(card);
            if (optimisticTransitionsAvailable() && !WorkTransitionService.hasAnyConditions(issueKey)) {
                WorkDragAndDrop.initDropZones(issueKey);
            } else {
                // kick off transition querying - will add drop zones upon completion
                WorkDragAndDrop.queryPossibleTransitions(issueId, issueKey);
            }
        }

        // handle multidrag
        if (selectedIssues.length > 1) {
            if (card.hasClass("ghx-parent-group")) {
                card.children('.js-issue').addClass('ghx-move-main').find('.ghx-move-count > b').text(selectedIssues.length);
            } else {
                card.addClass('ghx-move-main').find('.ghx-move-count > b').text(selectedIssues.length);
            }
        }

        WorkDragAndDrop.analytics.dragAndDrop.trigger('start', 'issueCount', selectedIssues.length);
    };

    /**
     * Called when an issue drag operation stopped.
     */
    WorkDragAndDrop.issueDragStop = function (event, ui) {
        // unbind the keydown handler
        $(document).unbind('keydown.dragAndDrop');

        // distinguish between sortable (ui.item) and draggable (this)
        var card = ui.item ? $(ui.item) : $(this);

        // remove multi-drag label
        if (card.hasClass('ghx-parent-group')) {
            card.children('.js-issue').removeClass('ghx-move-main');
        } else {
            card.removeClass('ghx-move-main');
        }

        // cleanup
        WorkDragAndDrop.destroyDropZoneOverlay();
        WorkDragAndDrop.cleanTransitionClasses();
        WorkDragAndDrop.currentDragElement = null;

        // if the current board is not rankable, show a warning if the user tried ranking vertically
        // Unfortunately at this stage we don't have much information regarding whether the draggable actually
        // dropped somewhere. let's use fuzzy logic
        if (!WorkDragAndDrop.cancel && !GH.RankingModel.isRankable()) {
            var deltaX = Math.abs(ui.position.left);
            var deltaY = Math.abs(ui.position.top);
            if (deltaX < 35 && deltaY > 50) {
                WorkDragAndDrop.showNotRankableWarning();
            }
        }
        WorkDragAndDrop.droppablesReady = false;
        WorkDragAndDrop.analytics.dragAndDrop.trigger('complete');
    };

    /**
     * Is the given issue element currently being dragged by the user.
     * (Does not include issues in the selection, simply the issue which is hovering over the DOM).
     * @param issueKey
     * @returns {boolean}
     */
    WorkDragAndDrop.isIssueBeingDragged = function (issueKey) {
        const currentDragElement = WorkDragAndDrop.currentDragElement;
        if (currentDragElement) {
            const draggedIssueKey = $(currentDragElement).data('issue-key');
            return issueKey === draggedIssueKey;
        }
    };

    WorkDragAndDrop.showNotRankableWarning = function () {
        var url = GH.RapidBoard.getRapidViewConfigurationUrl(GH.WorkController.rapidViewData.id, 'filter');
        var a = '<a href="' + url + '">';
        var endA = '</a>';

        var message = AJS.I18n.getText('gh.rapid.board.not.rankable.warning', a, endA);
        Notification.showWarning(message);
    };

    /**
     * Called when an issue is dropped.
     * @fires WorkDragAndDrop#before:issueDropRender as soon as the issue is dropped if the drag action was not cancelled.
     */
    WorkDragAndDrop.issueDropped = function (event, ui) {
        // if we're in a cancel, fail fast (strangely, this function still gets called when we call cancel...
        if (WorkDragAndDrop.cancel) {
            return;
        }
        WorkDragAndDrop.trigger('before:issueDropRender');

        var droppable = $(this);

        // fetch the transition id directly from the droppable
        var transitionId = droppable.attr('target-transition-id');
        var transitionStatus = parseInt(droppable.attr('target-transition-status'), 10);
        const selectedIssueKeys = GH.WorkSelectionController.getSelectedIssueKeys();
        const boardId = GH.RapidBoard.State.getRapidViewId();

        const targetColumn = droppable.closest(".ghx-target-option").data('column-id');
        const sourceColumn = ui.draggable.closest(".ghx-column").data("column-id");

        if (WorkDragAndDrop._shouldUseOptimisticTransitions(selectedIssueKeys, transitionId)) {
            GH.WorkController.optimisticallyTransition(selectedIssueKeys, transitionId, transitionStatus, sourceColumn, targetColumn);
        } else if (selectedIssueKeys.length > 1) {
            const request = {
                rapidViewId: boardId,
                issueKeys: selectedIssueKeys,
                targetColumn: targetColumn,
                selectedTransitionId: transitionId,
                mode: GH.RapidBoard.State.getMode()
            };

            GH.WorkController.submitTransitionAndRank(request).always(GH.WorkController.reload);
        } else {
            // fetch the transition id directly from the droppable
            var issueId = GH.WorkSelectionController.getIssueId($(this));

            WorkDragAndDrop.executeWorkflowTransition(issueId, transitionId, transitionStatus);
        }

        // this drop operation means that the user didn't want to rank but made a transition
        WorkDragAndDrop.skipNextRank = true;
    };

    WorkDragAndDrop._shouldUseOptimisticTransitions = function (issueKeys, transitionId) {
        if (!FeatureManager.isFeatureEnabled(WorkDragAndDrop.OPTIMISTIC_TRANSITIONS_FLAG)) {
            sendAnalytics('fallback', {cause: "optimistic-transition-disabled"});
            return false;
        }

        if (!WorkTransitionService.data) {
            sendAnalytics('fallback', {cause: "no-transition-data"});
            return false;
        }

        // Check that there is no column statistic on the board.
        // We can't update the column counts client-side only yet.
        if (GH.WorkController.getStatisticField() !== 'none') {
            sendAnalytics('fallback', {cause: "has-statistic-field"});
            return false;
        }

        // Check that the transition does not have a screen.
        if (WorkTransitionService.hasAnyTransitionHasScreen(issueKeys, transitionId)) {
            sendAnalytics('fallback', {cause: "has-screen"});
            return false;
        }

        // Check that we're not using the following swimlane strategies. With these, the issue may need to move swimlanes
        // when it is transitioned.
        const swimlaneStrategy = GH.GridDataController.getModel().getSwimlaneStrategy();
        if (swimlaneStrategy === 'custom' || swimlaneStrategy === 'assignee' || swimlaneStrategy === 'assigneeUnassignedFirst') {
            sendAnalytics('fallback', {cause: "unsupported-swimlane"});
            return false;
        }

        sendAnalytics('optimistic');
        return true;
    };

    const sendAnalytics = (type, properties) => {
        let analytics = {
            name: `jira-software.rapidboard.issuecard.transition.${type}`
        };

        if (properties) {
            analytics.properties = properties;
        }

        Analytics.send(analytics);
    };
    /**
     * Queries the possible transition targets.
     */
    WorkDragAndDrop.queryPossibleTransitions = function (issueId, issueKey) {
        GH.Ajax.get({
            url: '/xboard/issue/transitions.json',
            data: {
                'issueId': issueId
            }
        }, 'queryPossibleTransitions')
            .done(function (data) {
                WorkDragAndDrop.initDropZones(issueKey, data);
            });
    };

    /**
     * Initializes the transitions overlay.
     */
    WorkDragAndDrop.dropzoneOverlay = null;
    WorkDragAndDrop.initDropZoneOverlay = function (draggedElement) {
        // fetch the columns
        var model = GH.GridDataController.getModel();
        var columns = model.getColumns();
        var sourceColumnId = parseInt(draggedElement.closest('.ghx-column').attr('data-column-id'), 10);

        // render the overlay
        var $overlay = $(GH.tpl.rapid.workdraganddrop.renderDropZoneOverlay({
            columns: columns,
            sourceColumnId: sourceColumnId
        }));
        WorkDragAndDrop.dropzoneOverlay = $overlay;

        // add the zone to the swimlane
        var $swimlane = draggedElement.closest('.ghx-swimlane');
        $swimlane.addClass('ghx-drag-in-progress');
        $swimlane.append($overlay);

        WorkDragAndDrop.positionDropZoneOverlay();
    };

    WorkDragAndDrop.positionDropZoneOverlay = function () {
        if (WorkDragAndDrop.gridLayoutOn()) {
            WorkDragAndDrop.positionDropZoneOverlayGridLayout();
            $('#ghx-pool-column').on('scroll.overlayResizer', WorkDragAndDrop.positionDropZoneOverlayGridLayout);
        } else {
            WorkDragAndDrop.positionDropZoneOverlayLegacyLayout();
            $('#ghx-pool').on('scroll.overlayResizer', WorkDragAndDrop.positionDropZoneOverlayLegacyLayout);
        }
    };

    WorkDragAndDrop.gridLayoutOn = function () {
        const flexibleBoardsEnabled = FeatureManager.isFeatureEnabled("com.atlassian.jira.agile.darkfeature.flexible.boards");
        return flexibleBoardsEnabled && CSS.supports && CSS.supports('display', 'grid');
    };

    WorkDragAndDrop.positionDropZoneOverlayLegacyLayout = function () {
        var $overlay = $('.ghx-zone-overlay');
        var $columns = $overlay.prev('.ghx-columns');
        var $pool = $('#ghx-pool');

        var poolScroll = $pool.scrollTop();

        // current top position of the target swimlane columns
        var colPosTop = $columns.position().top;

        // this is actual position columns is at
        var top = (colPosTop + poolScroll);

        // make sure top stops at the swimlane stalker
        var firstColumnPosition = $pool.children('.ghx-swimlane').first().children('.ghx-columns').position();
        var minTop = firstColumnPosition.top + poolScroll;
        var topDifference = (($.browser.msie && $.browser.versionNumber <= 11) ? colPosTop : colPosTop - minTop);
        if (topDifference < 0) {
            top = top - topDifference;
        }

        // now take care of the height, which by default is the column height
        var columnsHeight = $columns.height();
        var height = columnsHeight;

        // make sure we stop at the window end
        var windowHeight = $(window).height();
        var maxHeight = windowHeight - $pool.offset().top - top + poolScroll;
        if (height > maxHeight) {
            height = maxHeight;
        }

        // also make sure we stop at the swimlane end
        var colEndTop = colPosTop + columnsHeight + poolScroll;
        var topDiff = colEndTop - top;
        if (height > topDiff) {
            height = topDiff;
        }

        // hide the drop zone in case of negative height
        var visible = height < 1 ? 'hidden' : '';

        $overlay.css({
            'top': top,
            'height': height,
            'visibility': visible
        });
    };

    /**
     * Positions the drop zones accordingly
     */
    WorkDragAndDrop.positionDropZoneOverlayGridLayout = function () {
        const $currentElement = WorkDragAndDrop.currentDragElement;
        const $columns = $currentElement.closest('.ghx-columns');
        const $container = $('#ghx-pool-column');

        const columnsTopPosition = $columns.position().top;
        const swimlaneTopPosition = $currentElement.closest('.ghx-swimlane').position().top;
        const scrollPosition = $container.scrollTop();

        let adjustForScrolling = scrollPosition - swimlaneTopPosition;
        adjustForScrolling = adjustForScrolling > 0 ? adjustForScrolling : 0;

        const overlayTopPosition = columnsTopPosition + adjustForScrolling;


        var columnsHeight = $columns.height();

        var containerHeight = $('#ghx-pool-column').height() - $('#ghx-column-header-group').height();
        var heightToContainerEnd = containerHeight - overlayTopPosition + scrollPosition;

        var columnsBottomPosition = columnsTopPosition + columnsHeight;
        var heightToColumnsEnd = columnsBottomPosition - overlayTopPosition;

        const overlayHeight = Math.min(columnsHeight, heightToContainerEnd, heightToColumnsEnd);


        const overlayVisible = overlayHeight < 1 ? 'hidden' : '';

        $('.ghx-zone-overlay').css({
            'top': overlayTopPosition,
            'height': overlayHeight,
            'visibility': overlayVisible
        });
    };

    /**
     * Initializes the target transition zones for a given issue
     */
    WorkDragAndDrop.initDropZones = function (issueKey, data) {
        // ensure the data is valid, thus is for the current draggable
        var dragElement = WorkDragAndDrop.currentDragElement;
        if (!dragElement
            || GH.WorkSelectionController.getIssueKey(dragElement) !== issueKey
            || !WorkDragAndDrop.dropzoneOverlay) {
            return;
        }

        // fetch stuff we need
        var model = GH.GridDataController.getModel();
        var $columnHeaders = $('.ghx-column-headers');
        var $dropZoneColumns = WorkDragAndDrop.dropzoneOverlay.find('.ghx-zone-table');

        // initialize column by column
        $dropZoneColumns.each(function (index, column) {
            var $column = $(column);

            // find the data that corresponds to this column
            var columnId = $column.data('column-id');
            var columnData = model.getColumnById(columnId);
            if (!columnData) {
                return;
            }

            // get transitions for column
            let transitionsPromise;
            if (optimisticTransitionsAvailable() && !WorkTransitionService.hasAnyConditions(issueKey)) {
                transitionsPromise = WorkTransitionService.queryTransitionsForColumn(issueKey, columnData);
            } else {
                sendAnalytics('fallback', {cause: "has-conditions"});
                transitionsPromise = WorkDragAndDrop.getTransitionsForColumn(data, columnData);
            }
            transitionsPromise.done((transitions) => {
                if (_.isEmpty(transitions)) {
                    return;
                }

                // fetch the column and its header
                var $columnHeader = $columnHeaders.children('.ghx-column[data-id=' + columnId + ']');

                // this is a valid target option
                $column.addClass('ghx-target-option');
                $columnHeader.addClass('ghx-target-option');

                WorkDragAndDrop.setupDropZonesForColumn($column, $columnHeader, transitions);

                // force the draggable to reload its droppables
                WorkDragAndDrop.forceDroppablesReload();
            }).fail(() => {
                WorkDragAndDrop.destroyDropZoneOverlay();
                WorkDragAndDrop.cleanTransitionClasses();
            });
        });

    };

    WorkDragAndDrop.getTransitionsForColumn = function (data, columnData) {
        let deferred = $.Deferred();
        // for each column, find the transitions that match
        var matchingTransitions = [];
        for (var x = 0; x < data.transitions.length; x++) {
            for (var i = 0; i < columnData.statusIds.length; i++) {
                if (columnData.statusIds[i] == data.transitions[x].targetStatus) {
                    matchingTransitions.push(data.transitions[x]);
                }
            }
        }
        return deferred.resolve(matchingTransitions);
    };

    /**
     * Sets up the drop zones for a given column.
     */
    WorkDragAndDrop.setupDropZonesForColumn = function ($column, $columnHeader, transitions) {
        // setup the drop zone for each transition
        _.each(transitions, function (transition) {
            var zone = $(GH.tpl.rapid.workdraganddrop.renderDropZone({
                transition: transition,
                singleTransition: transitions.length === 1,
                useOptimisticTransitionModel: optimisticTransitionsAvailable() && notIssueFetchedTransitions(transitions)
            }));
            $column.append(zone);
        });

        // setup the droppables

        // TODO: for single target column transitions we disabled hoverClass, over and out. Should we do this here too?
        // The performance of the hover state handling is not ideal on touch devices
        // so we only apply these options to non-touch devices

        var options = {
            accept: '.js-issue, .ghx-parent-group',
            hoverClass: 'ghx-target-hover',
            over: function () {
                $column.addClass('ghx-target-hover');
                $columnHeader.addClass('ghx-target-hover');
            },
            out: function () {
                // only clear out the class if no other drop zone has the hover state already
                // for some reason when dragging between zones the over even of one zone is triggered before the out event of another
                if ($column.find('.ghx-drop-zone.ghx-target-hover').length < 1) { // @todo Miruflin to look at this
                    $column.removeClass('ghx-target-hover');
                    $columnHeader.removeClass('ghx-target-hover');
                }
            },
            tolerance: 'pointer',
            drop: WorkDragAndDrop.issueDropped
        };
        $column.children().droppable(options);
    };

    /**
     * Removes the transition overlay
     */
    WorkDragAndDrop.destroyDropZoneOverlay = function () {
        if (WorkDragAndDrop.dropzoneOverlay) {
            WorkDragAndDrop.dropzoneOverlay.remove();
            WorkDragAndDrop.dropzoneOverlay = null;
            $('#ghx-pool').off('scroll.overlayResizer');
            $('#ghx-pool-column').off('scroll.overlayResizer');
        }
    };

    /**
     * Clears effects added as part of the transition targets.
     */
    WorkDragAndDrop.cleanTransitionClasses = function () {
        $('.ghx-column')
            .removeClass('ghx-target-option')
            .removeClass('ghx-target-hover');
        $('.ghx-swimlane')
            .removeClass('ghx-drag-in-progress');
    };

    /**
     * Forces the draggable to reload the target droppables on the next mouse move.
     */
    WorkDragAndDrop.forceDroppablesReload = function () {
        // ensure the draggable refreshes the position of all droppables (as they were not around when the drag started)
        var ddmanager = $.ui.ddmanager;
        var draggable = ddmanager.current;
        if (draggable) {
            // this sets the visibility on all dropzones, which otherwise prevents them from being checked on over.
            ddmanager.prepareOffsets(draggable, null);

            // simulate a drag event in case we're already over a dropzone and the user manages to not move the mouse at all,
            // he'll still get the dropzone activated.
            ddmanager.drag(draggable, null);
        }
        WorkDragAndDrop.droppablesReady = true;
    };

    /**
     * Executes a workflow transition.
     */
    WorkDragAndDrop.executeWorkflowTransition = function (issueId, transitionId, transitionStatus) {
        // keep the target status around, to give the dialog success handler a chance to optimistically update the
        // issue column
        WorkDragAndDrop.transitionTargetStatus = transitionStatus;

        var xsrfToken = $('#atlassian-token').attr('content');
        var url = GH.Ajax.CONTEXT_PATH + '/secure/WorkflowUIDispatcher.jspa?id={0}&action='
            + transitionId
            + '&atl_token='
            + xsrfToken;
        var link = '<a href="' + url + '" class="issueaction-workflow-transition" style="display:none"></a>';
        var $link = $(link);
        $('#gh').append($link);
        $link.click();
        $link.remove();
    };

    /** Holds the currently executing transition status. */
    WorkDragAndDrop.transitionTargetStatus = undefined;

    const notIssueFetchedTransitions = transitions => !transitions.some(transition => transition.hasOwnProperty('id'));
    const optimisticTransitionsAvailable = () => {
        return WorkTransitionService.data && FeatureManager.isFeatureEnabled(WorkDragAndDrop.OPTIMISTIC_TRANSITIONS_FLAG)
    };

    return WorkDragAndDrop;
});

AJS.namespace('GH.WorkDragAndDrop', null, require('jira-agile/rapid/ui/work/work-drag-and-drop'));