AJS.test.require(["com.pyxis.greenhopper.jira:gh-rapid", "com.pyxis.greenhopper.jira:gh-test-common"], function () {
    var WorkDataLoader = require('jira-agile/rapid/ui/work/work-data-loader');

    module('WorkDataLoader', {
        setup: function setup() {
            this.server = sinon.fakeServer.create();
            this.clock = sinon.useFakeTimers(new Date().getTime()); // use now
            GH.Ajax.xhrRegistery = {};
            this.oldStallTimeout = GH.Ajax.stallTimeout;
            GH.Ajax.stallTimeout = -1;
            var stub;
            stub = sinon.stub(GH.Ajax, 'handleContextualErrors');
            stub = sinon.stub(GH.Ajax, 'handleGlobalErrors');
            stub = sinon.stub(GH.Ajax, 'matchesCurrentUser');
            stub.returns(true);

            var GlobalEvents = require('jira-agile/rapid/global-events');
            GlobalEvents.trigger('pre-initialization');
        },
        teardown: function teardown() {
            this.server.restore();
            this.clock.restore();
            GH.Ajax.stallTimeout = this.oldStallTimeout;
            GH.Ajax.handleContextualErrors.restore();
            GH.Ajax.handleGlobalErrors.restore();
            GH.Ajax.matchesCurrentUser.restore();
        }
    });

    /*
     * Stub away the quick filters and sprint ids
     */
    var setubStubs = function setubStubs() {
        GH.WorkControls.getActiveQuickFilterIds = function () {
            return [];
        };
        GH.WorkControls.getActiveSprintIds = function () {
            return [];
        };
        GH.WorkController.hasDataChanged = function () {
            return false;
        };
    };

    /** Creates server result data */
    var createResultData = function createResultData(etag) {
        return {
            etagData: {
                etag: etag
            }
        };
    };

    test('get with success', function () {
        setubStubs();
        // ask for the data
        var promise, onSuccess, onError;
        onSuccess = sinon.spy();
        onError = sinon.spy();

        promise = WorkDataLoader.getData(10);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');

        promise.done(onSuccess);
        promise.fail(onError);

        // respond to the request
        var sentData = createResultData('testTag');
        this.server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify(sentData));

        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 1, 'Success callback should be called once');
        deepEqual(onSuccess.getCall(0).args[0], sentData, 'data should be as expected');
    });

    /*
     * Tests that subsequent requests without changed values result in no new ajax request
     */
    test('get with repeated success and dirty', function () {
        setubStubs();

        // ask for the data
        var promise, onSuccess, onError;
        onSuccess = sinon.spy();
        onError = sinon.spy();

        promise = WorkDataLoader.getData(11);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');

        promise.done(onSuccess);
        promise.fail(onError);

        // respond to the request
        var sentData = createResultData('testTag');
        this.server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify(sentData));

        // do it a second time - no request should be issued this time
        promise = WorkDataLoader.getData(11);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'no new request should have been sent');
        promise.done(onSuccess);
        promise.fail(onError);

        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 2, 'Success callback should be called twice');
        deepEqual(onSuccess.getCall(0).args[0], sentData, 'data should be as expected');
        deepEqual(onSuccess.getCall(1).args[0], sentData, 'data should be as expected');

        // now set the data as dirty
        WorkDataLoader.markDirty();

        // do it a third time, this time it should request as the previous data is marked as dirty
        promise = WorkDataLoader.getData(11);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 2, 'a new request should have been sent');
        promise.done(onSuccess);
        promise.fail(onError);

        // respond to the request
        var sentData2 = createResultData('testTag2');
        this.server.requests[1].respond(200, { "Content-Type": "application/json" }, JSON.stringify(sentData2));

        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 3, 'Success callback should be called three times');
        deepEqual(onSuccess.getCall(2).args[0], sentData2, 'data should be as expected');
    });

    test('get with failure', function () {
        setubStubs();

        // ask for the data
        var promise, onSuccess, onError;
        onSuccess = sinon.spy();
        onError = sinon.spy();

        promise = WorkDataLoader.getData(12);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');

        promise.done(onSuccess);
        promise.fail(onError);

        // respond to the request
        this.server.requests[0].respond(404, { "Content-Type": "application/json" }, JSON.stringify({}));

        strictEqual(onError.callCount, 1, 'Error callback should be called once');
        strictEqual(onSuccess.callCount, 0, 'Success callback should never be called');
    });

    /*
     * Tests update checking code
     */
    test('get with repeated success and dirty', function () {
        setubStubs();

        // ask for the data
        var promise, onSuccess, onError;
        onSuccess = sinon.spy();
        onError = sinon.spy();

        // do the initial data load
        promise = WorkDataLoader.getData(13);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 1, 'only one request should have been sent');
        var sentData = createResultData('testTag');
        this.server.requests[0].respond(200, { "Content-Type": "application/json" }, JSON.stringify(sentData));
        promise.done(onSuccess);
        promise.fail(onError);
        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 1, 'Success callback should be called twice');
        deepEqual(onSuccess.getCall(0).args[0], sentData, 'data should be as expected');

        // now tests update checks
        promise = WorkDataLoader.checkForUpdates(13);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 2, 'a new request should have been sent');
        this.server.requests[1].respond(200, { "Content-Type": "application/json" }, JSON.stringify(sentData));

        // no data should have been changed
        promise.done(onSuccess);
        promise.fail(onError);
        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 2, 'Success callback should be called twice');
        deepEqual(onSuccess.getCall(1).args[0], WorkDataLoader.NO_DATA_CHANGE, 'no change expected');

        // now return a different etag, but with equal data (defined by WorkController.hasDataChnaged)
        promise = WorkDataLoader.checkForUpdates(13);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 3, 'a new request should have been sent');

        var sentData2 = createResultData('testTag2');
        this.server.requests[2].respond(200, { "Content-Type": "application/json" }, JSON.stringify(sentData2));

        // data should not have changed, as models are still the same
        promise.done(onSuccess);
        promise.fail(onError);
        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 3, 'Success callback should be called three times');
        deepEqual(onSuccess.getCall(2).args[0], WorkDataLoader.NEW_DATA_NO_CHANGE, 'no change expected');

        // now return a different etag, and also signal the data as changed
        promise = WorkDataLoader.checkForUpdates(13);
        ok(promise, 'promise should be returned');
        strictEqual(this.server.requests.length, 4, 'a new request should have been sent');

        GH.WorkController.hasDataChanged = function () {
            return true;
        };

        var sentData3 = createResultData('testTag3');
        this.server.requests[3].respond(200, { "Content-Type": "application/json" }, JSON.stringify(sentData3));

        // data should have changed
        promise.done(onSuccess);
        promise.fail(onError);
        strictEqual(onError.callCount, 0, 'Error callback should never be called');
        strictEqual(onSuccess.callCount, 4, 'Success callback should be called four times');
        deepEqual(onSuccess.getCall(3).args[0], WorkDataLoader.NEW_DATA_WITH_CHANGES, 'new data expected');
    });
});