define('jira-agile/rapid/ui/work/work-data-loader', [
    "jira-agile/rapid/work/work-controls",
    "jira-agile/rapid/ajax",
    "underscore",
    "jira/jquery/deferred",
    "jira-agile/rapid/global-events",
    "jira-agile/rapid/const/page-mode",
    "jira-agile/rapid/rapid-board-data-loader",
    "require"
], function(WorkControls, Ajax, _, Deferred, GlobalEvents, PageMode, BoardDataLoader, require) {
    "use strict";

    /**
     * Handles work data loading
     */
    const WorkDataLoader = {};

    var WorkController;
    GlobalEvents.on("pre-initialization", function () {
        WorkController = require('jira-agile/rapid/ui/work/work-controller');
    });

    WorkDataLoader.NEW_DATA_WITH_CHANGES = "new data with changes";
    WorkDataLoader.NEW_DATA_NO_CHANGE = "new data no change";
    WorkDataLoader.NO_DATA_CHANGE = "no data change";

    /** Data that has successfully finished loading */
    WorkDataLoader.currentData = undefined;

    /** Data that is in the progress of being loaded */
    WorkDataLoader.loadingData = undefined;

    /**
     * Get the data for the work mode.
     * @return a promise object.
     */
    WorkDataLoader.getData = function(viewId) {
        return WorkDataLoader.getDataForViewId(viewId).promise;
    };

    /**
     * Marks the current as well as currently loading data as dirty. Will force a new data load to happen on the next
     * request for data.
     */
    WorkDataLoader.markDirty = function() {
        if (WorkDataLoader.currentData) {
            WorkDataLoader.currentData.dirty = true;
        }
        if (WorkDataLoader.loadingData) {
            WorkDataLoader.loadingData.dirty = true;
        }
    };

    /**
     * Clears the stored etag, forcing the next data load to compute entirely.
     * Use when attempting to load changes in the work data which may not be distinguished due to the etag.
     */
    WorkDataLoader.clearEtag = function() {
        if (WorkDataLoader.currentData && WorkDataLoader.currentData.data) {
            delete WorkDataLoader.currentData.data.etagData.etag;
        }
    };

    /**
     * Get the data for a given view id.
     * @param forceLoad whether the data should be loaded even if it is currently up to date. Normally the poller
     *        uses this to check for updates
     * @return a promise for the result data
     */
    WorkDataLoader.getDataForViewId = function(viewId, forceLoad) {
        // do we have up to date data
        if (!forceLoad  && WorkDataLoader.isDataUpToDate(WorkDataLoader.currentData, viewId)) {
            return WorkDataLoader.currentData;
        }

        // is the currently loading request up to date
        if (WorkDataLoader.isDataUpToDate(WorkDataLoader.loadingData, viewId)) {
            return WorkDataLoader.loadingData;
        }

        // do we have an etag
        var currData = WorkDataLoader.currentData ? WorkDataLoader.currentData.data : false;
        var etag = currData ? currData.etagData.etag : false;

        // load data, and attach a handler to move the loaded data to currentData upon completion
        var newData = WorkDataLoader.load(viewId, etag);
        var deferred = new Deferred();

        // in case we get new data, set the new data as the current data, and in case where the data hasn't changed
        // copy of the previous data over as all we get from the server is the etag (without an actual data)
        newData.requestPromise.done(function(data) {
            // ensure newData is still loadingData
            if (WorkDataLoader.loadingData === newData) {
                WorkDataLoader.currentData = WorkDataLoader.loadingData;

                // decide whether the data is still up to date
                // in which case we replace the returned data with the current data
                if (data.etagData.etag === etag) {
                    data = currData;
                }
                WorkDataLoader.currentData.data = data;
                WorkDataLoader.loadingData = undefined;

                // now resolve the returned promise
                deferred.resolveWith(this, [ data ]);
            } else {
                // ignore as the request is outdated
            }
        });

        // forward failed requests to the deferred
        newData.requestPromise.fail(deferred.reject);

        newData.promise = deferred.promise();

        WorkDataLoader.loadingData = newData;

        // return the loading data
        return WorkDataLoader.loadingData;
    };

    WorkDataLoader.isLoadingData = function() {
        return WorkDataLoader.loadingData &&
            WorkDataLoader.loadingData.promise.state() === "pending";
    };

    /**
     * Loads a new set of data and returns whether the data is different from
     */
    WorkDataLoader.checkForUpdates = function(viewId) {
        var deferred = Deferred();

        // stop if the current data is not up to date - somewhere we probably already load stuff
        if (WorkDataLoader.isLoadingData()) {
            deferred.resolveWith(this, [ 'ignore' ]);
        }
        else {
            // fetch the etag, then reload and check whether the etag changed
            var currentEtag = WorkDataLoader.currentData.data.etagData.etag;

            // load new data
            var result = WorkDataLoader.getDataForViewId(viewId, true);
            result.promise.done(function(data) {

                // equal etags -> no data change
                if (data.etagData.etag === currentEtag) {
                    // same etag
                    deferred.resolveWith(this, [ WorkDataLoader.NO_DATA_CHANGE ]);
                }

                // data has changed, so lets figure out whether the model actually changed.
                // if not we can still regard the current data as up-to-date, with the added
                // bonus that the etag is now updated to the latest version so subsequent poll requests
                // will return faster
                else if(!WorkController.hasDataChanged(data)) {
                    deferred.resolveWith(this, [ WorkDataLoader.NEW_DATA_NO_CHANGE ]);
                }

                // data has changed
                else {
                    // data contains changes we don't know about yet, signal it
                    deferred.resolveWith(this, [ WorkDataLoader.NEW_DATA_WITH_CHANGES ]);
                }
            });
        }
        return deferred.promise();
    };

    /**
     * Is the passed data object up to date (e.g. exists and is not dirty)
     */
    WorkDataLoader.isDataUpToDate = function(data, viewId) {
        if (!data || data.dirty) {
            return false;
        }

        return WorkDataLoader.areRequestParamsUpToDate(data, viewId);
    };

    /** Are the request parameters still up to date? */
    WorkDataLoader.areRequestParamsUpToDate = function(data, viewId) {
        var activeQuickFilters = WorkControls.getActiveQuickFilterIds();
        var activeSprints = WorkControls.getActiveSprintIds();

        return data.viewId === viewId
            && _.isEqual(data.activeQuickFilters, activeQuickFilters)
            && _.isEqual(data.activeSprints, activeSprints);
    };
    /**
     * Load the pool data
     */
    WorkDataLoader.load = function(viewId, etag) {
        var newData = {};
        newData.viewId = viewId;
        newData.activeQuickFilters = WorkControls.getActiveQuickFilterIds().slice();
        newData.activeSprints = WorkControls.getActiveSprintIds().slice();

        // gather the parts of the screen we want to load in one go
        var params = {
            rapidViewId: viewId,
            selectedProjectKey: GH.RapidBoard.projectKey
        };

        // do we have an etag to send along
        if (etag) {
            params.etag = etag;
        }

        /**
         * Work around for jQuery's messy handling of empty arrays,
         * see http://forum.jquery.com/topic/change-of-behavior-in-1-4-4-because-of-bug-6481
         * fyi: sends empty string instead of no value
         */
        if (newData.activeQuickFilters && newData.activeQuickFilters.length > 0) {
            params.activeQuickFilters = newData.activeQuickFilters;
        }
        if (newData.activeSprints && newData.activeSprints.length > 0) {
            params.activeSprints = newData.activeSprints;
        }

        // load the data
        newData.requestPromise = BoardDataLoader.loadData(PageMode.WORK, () => WorkDataLoader.submitLoadAllDataRequest(newData, params));
        return newData;
    };

    WorkDataLoader.submitLoadAllDataRequest = function(newData, params) {
        return Ajax.get({
            url: '/xboard/work/allData.json',
            data: params,
            deferErrorHandling: true
        }, 'workAllData');
    };

    return WorkDataLoader;
});

AJS.namespace('GH.WorkDataLoader', null, require('jira-agile/rapid/ui/work/work-data-loader'));