AJS.test.require('com.pyxis.greenhopper.jira:gh-rapid', function() {
    module("Work Context Menu Controller tests");
    var _ = require('underscore');
    const WorkContextMenuController = require('jira-agile/rapid/ui/work/work-context-menu-controller');

    test("Context Menu for issue contains top/bottom options", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: false,
            swimlane: false,
            selectedIssues: []
        };
        var allActions = getAllContextActions(ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-top" }), "Send to Top item found in context menu");
        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-send-bottom"}), "Send to Bottom item found in context menu");
    });

    test("Context Menu for issue doesn't show bulk change option when user doesn't have bulk change permission", function () {
        var ctx = {
            type: "kanban",
            selectedIssues: [],
            hasBulkChangePermission: false
        };
        var allActions = getAllContextActions(ctx);

        ok(!_.findWhere(allActions, {id: "ghx-issue-ctx-action-bulk-change"}), "Bulk change option not found in context menu");
    });

    test("Context Menu for issue shows bulk change option when user has bulk change permission", function () {
        var ctx = {
            type: "kanban",
            selectedIssues: [],
            hasBulkChangePermission: true
        };
        var allActions = getAllContextActions(ctx);

        ok(_.findWhere(allActions, {id: "ghx-issue-ctx-action-bulk-change"}), "Bulk change option found in context menu");
    });

    module("Context menu delete", {});

    test("Context Menu for single issue contains archive option", function () {
        var ctx = {
            type: "kanban",
            issueArchive: true,
            selectedIssues: ['ISS-1']
        };
        var allActions = getAllContextActions(ctx);

        ok(_.findWhere(allActions, {id: "ghx-issue-ctx-action-archive"}), "Archive option found in the context menu")
    });

    test("Archiving option in the context menu not found", function () {
        var ctx = {
            type: "kanban",
            issueArchive: false,
            selectedIssues: ["ISS-1"]
        };
        var allActions = getAllContextActions(ctx);

        ok(!_.findWhere(allActions, {id: "ghx-issue-ctx-action-archive"}), "Archive option not found in the context menu")
    });

    test("Context Menu for single issue contains delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: false,
            swimlane: false,
            selectedIssues: ['ISS-1']
        };
        var allActions = getAllContextActions(ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item found in context menu");
    });

    test("Context Menu for no issues does not contain delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: false,
            swimlane: false,
            selectedIssues: []
        };
        var allActions = getAllContextActions(ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item not found in context menu");
    });

    test("Context Menu for multiple issues does not contain delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: false,
            swimlane: false,
            selectedIssues: ['ISS-1', 'ISS-2']
        };
        var allActions = getAllContextActions(ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item not found in context menu");
    });

    test("Context Menu for issue with subtasks contains delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: true,
            swimlane: false,
            selectedIssues: ['ISS-1']
        };
        var allActions = getAllContextActions(ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item found in context menu");
    });

    test("Context Menu for swimlane contains delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: false,
            swimlane: true,
            selectedIssues: ['ISS-1']
        };
        var allActions = getAllContextActions(ctx);

        ok(_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item found in context menu");
    });

    test("Context Menu for multiple issues where atleast one is a swimlane does not contain delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: false,
            swimlane: true,
            selectedIssues: ['ISS-1', 'ISS-2']
        };
        var allActions = getAllContextActions(ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item not found in context menu");
    });

    test("Context Menu for multiple issues where atleast one has subtasks does not contain delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: true,
            swimlane: false,
            selectedIssues: ['ISS-1', 'ISS-2']
        };
        var allActions = getAllContextActions(ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item not found in context menu");
    });

    test("Context Menu for multiple issues with both subtasks and swimlanes does not contain delete option", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: true,
            swimlane: true,
            selectedIssues: ['ISS-1', 'ISS-2']
        };
        var allActions = getAllContextActions(ctx);

        ok(!_.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" }), "Delete item not found in context menu");
    });

    test("Calling actionFn for delete function calls delete shortcut method and fires analytics", function () {
        var ctx = {
            type: "kanban",
            hasSubtasks: false,
            swimlane: false,
            selectedIssues: ['ISS-1']
        };
        var expectedAnalytics = { name: "gh.issue.ctx.menu.action.delete" };

        sinon.stub(GH.IssueOperationShortcuts, "deleteSelectedIssue");
        sinon.stub(AJS, "trigger");

        var allActions = getAllContextActions(ctx);
        var deleteAction = _.findWhere(allActions, { id: "ghx-issue-ctx-action-delete" });
        ok(deleteAction, "Delete item found in context menu");

        deleteAction.actionFn();
        ok(GH.IssueOperationShortcuts.deleteSelectedIssue.called, "Called delete shortcut method");
        ok(AJS.trigger.calledWith("analytics", sinon.match(expectedAnalytics)), "Analytics triggered");
    });

    function getAllContextActions(contextData) {
        var optionsArray = WorkContextMenuController.resolveActionsForContext(contextData);
        return _.flatten(_.pluck(optionsArray, "actions"));
    }

    test('Context Menu for issue contains Print cards item in scrum board', function () {
        executeTestPrintCardsMenuItemInBoard('scrum');
    });
    test('Context Menu for issue contains Print cards item in kanban board', function () {
        executeTestPrintCardsMenuItemInBoard('kanban');
    });

    function executeTestPrintCardsMenuItemInBoard(board) {
        var ctx = {
            type: board,
            selectedIssues: ['ISS-1']
        };

        var allActions = getAllContextActions(ctx);
        ok(_.findWhere(allActions, {id: 'ghx-issue-ctx-action-print-cards'}), 'Print cards item was found in context menu');
    }
});

