define('jira-agile/rapid/ui/work/work-context-menu-controller', [
    "jira-agile/rapid/ui/work/grid-data-controller",
    "jira-agile/rapid/ui/plan/issue-actions",
    "jira/util/formatter",
    "underscore",
    "jquery",
    'jira-agile/rapid/analytics-tracker',
    'jira-agile/rapid/ui/print/card-printing-controller'
], function(GridDataController, IssueActions, formatter, _, $, AnalyticsTracker, CardPrintingController) {
    "use strict";

    const WorkContextMenuController = {};
    WorkContextMenuController.ctxMenu = null;
    WorkContextMenuController.Analytics = {};

    /**
     * @type module:jira-agile/rapid/analytics-tracker
     */
    WorkContextMenuController.Analytics.show = new AnalyticsTracker("gh.issue.ctx.menu.show");
    WorkContextMenuController.Analytics.action = new AnalyticsTracker("gh.issue.ctx.menu.action");

    /**
     * Initializes a context menu for use as an issue context menu and shows it at the given event's coordinates
     * @param event
     */
    WorkContextMenuController.showContextMenuAt = function (event, context) {
        WorkContextMenuController.hideContextMenu();

        // don't open the context menu when the user clicks on a link.
        if ($(event.target).closest('a').length > 0) {
            return;
        }

        var ctx = WorkContextMenuController.resolveContext(context);
        var menuOptions = {
            actionGroups: WorkContextMenuController.resolveActionsForContext(ctx)
        };
        WorkContextMenuController.ctxMenu = new GH.ContextMenu(menuOptions);
        WorkContextMenuController.ctxMenu.showAt(event);

        // prevent native context menu from showing
        event.preventDefault();
        // bring focus to the window, the preventDefault action might not have brought focus to the window
        $(GH.ContextMenu.menuSelector).focus();

        WorkContextMenuController.Analytics.show.trigger(ctx.type, "selectedIssuesCount", ctx.selectedIssues.length);
    };

    /**
     * Resolves the current issue context based on the selected issues in the WorkSelectionController.
     */
    WorkContextMenuController.resolveContext = function (context) {
        var selectedIssueKeys = GH.WorkSelectionController.getSelectedIssuesInOrder();

        var ctx = _.extend({
            selectedIssues: selectedIssueKeys
        }, context);

        ctx.type = GH.WorkControls.getBoardType();

        var hasSubstasks = false;
        var issues = GridDataController.getModel().getIssueDataByKeys(selectedIssueKeys);
        ctx.selectedIssuesData = issues;

        ctx.flaggedIssuesCount = _.where(issues, {flagged: true}).length;

        _.every(issues, function (issue) {
            if (issue.parentId) {
                hasSubstasks = true;
                return false;
            }
            return true;
        });
        ctx.hasSubtasks = hasSubstasks;
        ctx.hasBulkChangePermission = GridDataController.hasBulkChangePermission;
        ctx.issueArchive = GridDataController.issueArchivingEnabled;

        return ctx;
    };

    /**
     * Returns an array of action group objects relevant to the given context
     * @param ctx
     */
    WorkContextMenuController.resolveActionsForContext = function (ctx) {
        function updateIssueFlags(selectedIssues, hasSwimlane, flag) {
            return function () {
                if (!hasSwimlane) {
                    GH.WorkController.reload();
                } else {
                    var selectedIssues = GridDataController.getModel().getIssueDataByKeys(selectedIssues);
                    _.each(selectedIssues, function (issue) {
                        issue.flagged = flag;

                        if (flag) {
                            $("#ghx-pool").find(".ghx-swimlane-header[data-issue-key='" + issue.key + "']").addClass("ghx-flagged");
                        } else {
                            $("#ghx-pool").find(".ghx-swimlane-header[data-issue-key='" + issue.key + "']").removeClass("ghx-flagged");
                        }
                    });
                }
            };
        }

        var actionGroups = [];

        var swimlane = $(".ghx-swimlane-header").length > 1;

        var sendToTopAction = WorkContextMenuController.getSendToTopAction(ctx,
            swimlane ? formatter.I18n.getText("gh.rapid.operations.send.top.of.swimlane") : formatter.I18n.getText("gh.rapid.operations.send.top.of.column"));

        var sendToBottomAction = WorkContextMenuController.getSendToBottomAction(ctx,
            swimlane ? formatter.I18n.getText("gh.rapid.operations.send.bottom.of.swimlane") : formatter.I18n.getText("gh.rapid.operations.send.bottom.of.column"));

        actionGroups.push({
            label: formatter.I18n.getText("gh.rapid.operations.send.to"),
            actions: [sendToTopAction, sendToBottomAction]
        });

        if (ctx.type === "scrum" && !ctx.hasSubtasks && !ctx.swimlane) {
            var removeFromSprintAction = {
                id: "ghx-issue-ctx-action-remove-from-sprint",
                name: "remove from sprint",
                label: formatter.I18n.getText("gh.rapid.operations.remove.from.sprint"),
                actionFn: function () {
                    IssueActions.removeIssuesFromSprint(GH.WorkController.rapidViewData.id, ctx.selectedIssues);
                    WorkContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
                }
            };

            actionGroups.push({
                actions: [removeFromSprintAction]
            });

        }

        if (ctx.selectedIssues.length === 1 && ctx.issueArchive === true) {
            var archiveAction = {
                id: "ghx-issue-ctx-action-archive",
                name: "archive",
                label: formatter.I18n.getText("gh.rapid.operations.archive"),
                actionFn: function () {
                    // spawn archive dialog
                    GH.IssueOperationShortcuts.archiveSelectedIssue();
                }
            };
            actionGroups.push({
                actions: [archiveAction]
            });
        }

        if (ctx.selectedIssues.length === 1) {
            var deleteAction = {
                id: "ghx-issue-ctx-action-delete",
                name: "delete",
                label: formatter.I18n.getText("gh.rapid.operations.delete"),
                actionFn: function () {
                    // spawn delete dialog
                    GH.IssueOperationShortcuts.deleteSelectedIssue();

                    // trigger analytics
                    AJS.trigger("analytics", {name: "gh.issue.ctx.menu.action.delete"});
                }
            };
            actionGroups.push({
                actions: [deleteAction]
            });
        }

        var flagAction;
        var flagAndCommentAction;
        if (ctx.flaggedIssuesCount !== ctx.selectedIssues.length) {

            // propose flag
            flagAction = {
                id: "ghx-issue-ctx-action-flag-toggle",
                name: "flag",
                label: formatter.I18n.getText("gh.rapid.operations.flag"),
                actionFn: function () {
                    IssueActions.flag(ctx.selectedIssues, true).done(updateIssueFlags(ctx.selectedIssues, ctx.swimlane, true));
                    AJS.trigger("analytics", {
                        name: "gh.issue.ctx.menu.action.flag",
                        data: {selectedIssueCount: ctx.selectedIssues.length}
                    });
                }
            };

            flagAndCommentAction = {
                id: "ghx-issue-ctx-action-flag-and-comment",
                name: "flag-and-comment",
                label: formatter.I18n.getText("gh.rapid.operations.flag.and.comment"),
                actionFn: function () {
                    GH.IssueFlagAndCommentAction.execute(ctx.selectedIssues, true, updateIssueFlags(ctx.selectedIssues, ctx.swimlane, true));
                    AJS.trigger("analytics", {
                        name: "gh.issue.ctx.menu.action.flag.with.comment",
                        data: {selectedIssueCount: ctx.selectedIssues.length}
                    });
                }
            };
        } else {
            // propose unflag only if all the selected issues are already flagged
            flagAction = {
                id: "ghx-issue-ctx-action-unflag",
                name: "unflag",
                label: formatter.I18n.getText("gh.rapid.operations.unflag"),
                actionFn: function () {
                    IssueActions.flag(ctx.selectedIssues, false).done(updateIssueFlags(ctx.selectedIssues, ctx.swimlane, false));
                    AJS.trigger("analytics", {
                        name: "gh.issue.ctx.menu.action.unflag",
                        data: {selectedIssueCount: ctx.selectedIssues.length}
                    });
                }
            };

            flagAndCommentAction = {
                id: "ghx-issue-ctx-action-unflag-and-comment",
                name: "unflag-and-comment",
                label: formatter.I18n.getText("gh.rapid.operations.unflag.and.comment"),
                actionFn: function () {
                    GH.IssueFlagAndCommentAction.execute(ctx.selectedIssues, false, updateIssueFlags(ctx.selectedIssues, ctx.swimlane, false));
                    AJS.trigger("analytics", {
                        name: "gh.issue.ctx.menu.action.unflag.with.comment",
                        data: {selectedIssueCount: ctx.selectedIssues.length}
                    });
                }
            };
        }

        actionGroups.push({
            actions: [flagAction, flagAndCommentAction]
        });

        if (!ctx.swimlane) {
            var viewInIssueNavigatorAction = {
                id: "ghx-issue-ctx-action-view-in-navigator",
                name: "view in navigator",
                label: formatter.I18n.getText("gh.rapid.operations.view.in.navigator"),
                actionFn: function () {
                    IssueActions.viewInIssueNavigator(ctx.selectedIssues);
                    WorkContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
                }
            };

            actionGroups.push({
                actions: [viewInIssueNavigatorAction]
            });

            var bulkChangeAction = {
                id: "ghx-issue-ctx-action-bulk-change",
                name: "bulk change",
                label: formatter.I18n.getText("gh.rapid.operations.bulk.change"),
                actionFn: function () {
                    IssueActions.bulkChange(ctx.selectedIssues);
                    WorkContextMenuController.Analytics.action.trigger(this.name, "selectedIssuesCount", ctx.selectedIssues.length);
                }

            };

            if (ctx.hasBulkChangePermission) {
                actionGroups.push({
                    actions: [bulkChangeAction]
                });
            }
        }

        var prepareDataToPrintWorkSelectedIssues = function (issueKeys) {
            // map keys to issue objects
            var issuesToPrint = _.map(issueKeys, function (issueKey) {
                return GridDataController.getModel().getIssues()[issueKey];
            });

            return {
                issues: issuesToPrint,
                viewMode: 'work',
                boardType: GH.WorkControls.getBoardType()
            };
        };

        if (ctx.selectedIssues.length > 0) {
            var printCardsAction = {
                id: "ghx-issue-ctx-action-print-cards",
                name: "print cards",
                label: formatter.I18n.getText("gh.rapid.operations.print.selected.cards", ctx.selectedIssues.length),
                actionFn: function () {
                    CardPrintingController.printSelectedIssues(prepareDataToPrintWorkSelectedIssues(ctx.selectedIssues));
                }
            };

            actionGroups.push({
                actions: [printCardsAction]
            });
        }

        return actionGroups;
    };

    WorkContextMenuController.getSendToTopAction = function (ctx, label) {
        return {
            id: "ghx-issue-ctx-action-send-top",
            name: "send to top of swimlane",
            label: label,
            actionFn: function () {
                // send issues to top
                GH.WorkRanking.sendToTop("WorkContextMenu");
            }
        };
    };

    WorkContextMenuController.getSendToBottomAction = function (ctx, label) {
        return {
            id: "ghx-issue-ctx-action-send-bottom",
            name: "send to bottom of swimlane",
            label: label,
            actionFn: function () {
                // send issues to bottom
                GH.WorkRanking.sendToBottom("WorkContextMenu");
            }
        };
    };

    WorkContextMenuController.hideContextMenu = function () {
        if (WorkContextMenuController.ctxMenu != null) {
            WorkContextMenuController.ctxMenu.hide();
            WorkContextMenuController.ctxMenu = null;
        }
    };

    return WorkContextMenuController;
});

AJS.namespace('GH.WorkContextMenuController', null, require('jira-agile/rapid/ui/work/work-context-menu-controller'));
