define('jira-agile/rapid/ui/work/ranking-model', ['underscore', 'require', 'jira-agile/rapid/global-events'], function (_, require, GlobalEvents) {
    "use strict";

    /**
     * Controller for the grid data model
     */

    var RankingModel = {};

    // GLOBALS... FIX ME
    var GridDataController;
    // Resolve timing issue
    GlobalEvents.on("pre-initialization", function () {
        GridDataController = require('jira-agile/rapid/ui/work/grid-data-controller');
    });

    /**
     * Data stored in this model
     */
    RankingModel.orderData = {
        rankable: false,
        rankCustomFieldId: undefined
    };

    /**
     * Sets the order data
     */
    RankingModel.setOrderData = function (orderData) {
        RankingModel.orderData = orderData;
    };

    /**
     * Tell if this rapid view supports ranking (meaning is ordered by a rank field)
     */
    RankingModel.isRankable = function () {
        return RankingModel.orderData.rankable;
    };

    /**
     * Is the user allowed to rank on the board?
     */
    RankingModel.canUserRank = function () {
        if (!RankingModel.orderData.rankable) {
            return false;
        }

        var canRank = false;
        _.each(RankingModel.orderData.canRankPerProject, function (projData) {
            canRank = canRank || projData.canRank;
        });
        return canRank;
    };

    /**
     * Is the user allowed to rank a specific issue on the board?
     * Note: this only checks whether the user has rank permission on the project the issue is part
     */
    RankingModel.canUserRankIssue = function (issueKey) {
        if (!RankingModel.isRankable()) {
            return false;
        }

        var projKey = issueKey.substr(0, issueKey.indexOf('-'));
        var canRank = false;
        _.each(RankingModel.orderData.canRankPerProject, function (projData) {
            if (projData.projectKey == projKey) {
                canRank = projData.canRank;
            }
        });
        return canRank;
    };

    RankingModel.getRankCustomFieldId = function () {
        return RankingModel.isRankable() ? RankingModel.orderData.rankCustomFieldId : null;
    };

    RankingModel.isRankable = function () {
        return RankingModel.orderData.rankable;
    };

    /**
     * Get the first issue in the specified issue's swimlane-column.
     * If the specified issue is a sub-task, it will retrieve the first sub-task in it's group.
     */
    RankingModel.getFirstRankableIssueKeyInColumn = function (sourceIssueKey) {
        var model = GridDataController.getModel();

        // retrieve position information about the source issue
        var position = model._getIssuePositionAndDataByKey(sourceIssueKey);
        if (position.isSwimlane) {
            return false;
        }

        var issues = model.getIssuesDataForCell(position.swimlaneId, position.columnId);
        var isSubTask = !!position.issue.parentKey;

        var issue = _.find(issues, function (issue) {
            // subtasks can only be ranked against each other (inside the same parent issue). canIssuesBeRankedTogether properly checks this
            return !isSubTask || RankingModel.canIssuesBeRankedTogether(issue.key, sourceIssueKey);
        });

        if (issue) {
            if (!isSubTask) {
                // for subtasks we have to use the parentKey though
                return issue.parentKey ? issue.parentKey : issue.key;
            } else {
                return issue.key;
            }
        }

        return false;
    };

    /**
     * Get the last issue in the specified issue's swimlane-column.
     * If the specified issue is a sub-task, it will retrieve the last sub-task in it's group.
     */
    RankingModel.getLastRankableIssueKeyInColumn = function (sourceIssueKey) {
        var model = GridDataController.getModel();

        // retrieve position information about the source issue
        var position = model._getIssuePositionAndDataByKey(sourceIssueKey);
        if (position.isSwimlane) {
            return false;
        }

        var issues = model.getIssuesDataForCell(position.swimlaneId, position.columnId);
        var isSubTask = !!position.issue.parentKey;

        // get the key of the last satisfactory issue.
        var key = false;
        _.each(issues, function (issue) {
            if (!isSubTask) {
                // for subtasks we have to use the parentKey though
                key = issue.parentKey ? issue.parentKey : issue.key;
            } else if (RankingModel.canIssuesBeRankedTogether(issue.key, sourceIssueKey)) {
                key = issue.key;
            }
        });

        return key;
    };

    /**
     * Are the two passed issues be ranked at the same time.
     */
    RankingModel.canIssuesBeRankedTogether = function (keyOne, keyTwo) {
        var model = GridDataController.getModel();

        // fetch data for both issues
        var dataOne = model._getIssuePositionAndDataByKey(keyOne);
        var dataTwo = model._getIssuePositionAndDataByKey(keyTwo);
        if (!dataOne || !dataTwo) {
            return false;
        }
        if (dataOne.isSwimlane || dataTwo.isSwimlane) {
            return false;
        }

        // ensure both are in the same swimlane
        if (dataOne.swimlaneId != dataTwo.swimlaneId) {
            return false;
        }

        // ensure both are in the same column
        if (dataOne.columnId != dataTwo.columnId) {
            return false;
        }

        // if both are top level issues they can be ranked together
        if (!dataOne.issue.parentKey && !dataTwo.issue.parentKey) {
            return true;
        }

        // if both are child issues of the same parent they can be ranked together
        if (dataOne.issue.parentKey == dataTwo.issue.parentKey) {
            return true;
        }

        // not rankable together (different parent or parent and child issue)
        return false;
    };

    /**
     * Get all issues between two issues (including the limits, in order fromKey to toKey).
     * If the two issues are incompatible to each other an empty array is returned.
     */
    RankingModel.getRankableIssuesRange = function (fromKey, toKey) {
        // return an empty array if the two issue keys are incompatible
        if (!RankingModel.canIssuesBeRankedTogether(fromKey, toKey)) {
            return [];
        }

        var model = GridDataController.getModel();

        // ensure the range is valid, this also ensures we are in a single column of a single swimlane
        if (fromKey == toKey) {
            return [fromKey];
        }

        // fetch data for both issues
        var dataOne = model._getIssuePositionAndDataByKey(fromKey);
        var dataTwo = model._getIssuePositionAndDataByKey(toKey);
        if (dataOne.isSwimlane || dataTwo.isSwimlane) {
            return [];
        }

        // initialize result array and add the first key
        var issueRange = [];
        issueRange.push(fromKey);

        var issues = model.getIssuesDataForCell(dataOne.swimlaneId, dataOne.columnId);
        var fromIndex = dataOne.issueIndex;
        var toIndex = dataTwo.issueIndex;
        var i;
        var issue;
        if (toIndex > fromIndex) {
            // check top to bottom, skip fromKey
            for (i = fromIndex + 1; i <= toIndex; i++) {
                issue = issues[i];
                if (RankingModel.canIssuesBeRankedTogether(fromKey, issue.key)) {
                    issueRange.push(issue.key);
                }
            }
        } else {
            // check bottom to to, skip fromKey
            for (i = fromIndex - 1; i >= toIndex; i--) {
                issue = issues[i];
                if (RankingModel.canIssuesBeRankedTogether(fromKey, issue.key)) {
                    issueRange.push(issue.key);
                }
            }
        }

        // return the issues
        return issueRange;
    };

    /**
     * Finds the next issue to select given a set of to-be-ranked issues
     */
    RankingModel.getBestFitSelectionAfterRank = function (toBeRankedIssueKeys) {
        var bestFit = RankingModel.getNextIssueAfterSendToBottom(toBeRankedIssueKeys);
        if (!bestFit) {
            bestFit = RankingModel.getPreviousIssueAfterSendToBottom(toBeRankedIssueKeys);
        }
        return bestFit;
    };

    /**
     * Returns the id of the issue to go to after sending the specified issue to the bottom
     * We have to do some tricky handling to ensure that we don't return an issue which is a child of the current issue.
     */
    RankingModel.getNextIssueAfterSendToBottom = function (toBeRankedIssueKeys) {
        // the issue keys are in order, so we really only care about the last one
        var issueKey = _.last(toBeRankedIssueKeys);

        var model = GridDataController.getModel();

        // get the id of the next issue
        var nextIssueKey = model.getNextIssueKey(issueKey);
        if (!nextIssueKey) {
            return false;
        }

        // get the position of the current issue and the next issue
        var currentIssue = model.getIssueDataByKey(issueKey);
        var nextIssue = model.getIssueDataByKey(nextIssueKey);

        // if the current issue is a sub-task, return the next issue key regardless
        if (currentIssue.parentKey) {
            return nextIssueKey;
        }

        // search for the next issue which meets these criteria:
        //   is NOT a sub-task; OR
        //   the parent is NOT the current issue
        while (nextIssue.parentKey && nextIssue.parentKey == issueKey) {
            nextIssueKey = model.getNextIssueKey(nextIssueKey);
            if (!nextIssueKey) {
                // if we can't find any more issues, stop here
                return false;
            }
            nextIssue = model.getIssueDataByKey(nextIssueKey);
        }

        return nextIssueKey;
    };

    RankingModel.getPreviousIssueAfterSendToBottom = function (toBeRankedIssueKeys) {
        // the issue keys are in order, so we really only care about the last one
        var issueKey = _.first(toBeRankedIssueKeys);

        var model = GridDataController.getModel();

        var previousIssueKey = model.getPreviousIssueKey(issueKey);
        if (!previousIssueKey) {
            return false;
        }

        // get the position of the current issue and the next issue
        var currentIssue = model.getIssueDataByKey(issueKey);
        var previousIssue = model.getIssueDataByKey(previousIssueKey);

        // if the current issue is a sub-task, return the previous issue key regardless
        if (currentIssue.parentKey) {
            return previousIssueKey;
        }

        // search for the previous issue which meets these criteria:
        //   is NOT a sub-task
        while (previousIssue.parentKey) {
            previousIssueKey = model.getPreviousIssueKey(previousIssueKey);
            if (!previousIssueKey) {
                // if we can't find any more issues, stop here
                return false;
            }
            previousIssue = model.getIssueDataByKey(previousIssueKey);
        }

        return previousIssueKey;
    };

    /**
     * Reorders a set of issues to either before or after another issue. afterKey/beforeKey can be part of issueKeys,
     * in which case the result for afterKey/beforeKey is the same.
     */
    RankingModel.reorderIssues = function (issueKeys, afterKey, beforeKey) {
        var model = GridDataController.getModel();

        // because subtasks are ranked together with their parents, ensure that we move them as
        // well when the parent is reranked.
        issueKeys = RankingModel.getParentAndChildrenKeys(issueKeys);

        var insertPosition;
        var orderKeys = model.getOrder();
        if (beforeKey) {
            // find position to insert before
            insertPosition = _.indexOf(orderKeys, beforeKey);
        } else {
            // find the position to insert after
            insertPosition = _.indexOf(orderKeys, afterKey);
            insertPosition++;

            // make sure that we insert after a complete parent/children group in case ranking is enabled
            if (RankingModel.isRankable()) {
                while (insertPosition < orderKeys.length) {
                    var issueDataAtPosition = model.getIssueDataByKey(orderKeys[insertPosition]);
                    if (issueDataAtPosition && issueDataAtPosition.parentKey === afterKey) {
                        insertPosition++;
                    } else {
                        break;
                    }
                }
            }
        }

        // split lists, then remove keys, then put together (avoids having to fiddle with all weird edge cases
        var beforePosition = orderKeys.slice(0, insertPosition);
        var afterPosition = orderKeys.slice(insertPosition);
        beforePosition = RankingModel.withoutArray(beforePosition, issueKeys);
        afterPosition = RankingModel.withoutArray(afterPosition, issueKeys);
        var newOrder = _.flatten([beforePosition, issueKeys, afterPosition]);
        model.updateOrder(newOrder);
    };

    /**
     * Gets the keys for the parents and associated children. The children are expected to directly follow the parent
     */
    RankingModel.getParentAndChildrenKeys = function (issueKeys) {
        var model = GridDataController.getModel();
        var order = model.getOrder();
        var newKeys = [];

        _.each(issueKeys, function (key) {
            // add key itself
            newKeys.push(key);

            // get the data for this key
            var issueData = model.getIssueDataByKey(key);
            if (!issueData) {
                return;
            }

            // nothing to do if this issue is a subtask
            if (issueData.parentKey) {
                return;
            }

            // find the position, then check following issues for subtasks belonging to that parent
            var position = _.indexOf(order, key);
            if (position < 0) {
                return;
            }

            for (var i = position + 1; i < order.length; i++) {
                var possibleChildData = model.getIssueDataByKey(order[i]);
                if (possibleChildData.parentKey == issueData.key) {
                    // we found a child, add it to the list
                    newKeys.push(possibleChildData.key);
                } else {
                    // stop here
                    break;
                }
            }
        });
        return newKeys;
    };

    // not available in underscore
    RankingModel.withoutArray = function (a, b) {
        var res = a;
        _.each(b, function (elem) {
            res = _.without(res, elem);
        });
        return res;
    };

    return RankingModel;
});

AJS.namespace('GH.RankingModel', null, require('jira-agile/rapid/ui/work/ranking-model'));