define('jira-agile/rapid/ui/work/grid-swimlanes-data', ['underscore'], function (_) {
    "use strict";

    /* global _ */
    /**
     * Contains the logic for preprocessing swimlane information
     */

    var GridSwimlanesData = {};

    /**
     * Fetches the swimlanes for the model
     */
    GridSwimlanesData.calculateSwimlanesData = function (issuesData, swimlanesData, entityData) {
        var swimlanes = [];

        switch (swimlanesData.swimlaneStrategy) {
            case 'parentChild':
                swimlanes = GridSwimlanesData.calculateParentChildSwimlanes(issuesData, swimlanesData.parentSwimlanesData, entityData);
                break;

            case 'assignee':
                swimlanes = GridSwimlanesData.calculateAssigneeSwimlanes(issuesData, false);
                break;

            case 'assigneeUnassignedFirst':
                swimlanes = GridSwimlanesData.calculateAssigneeSwimlanes(issuesData, true);
                break;

            case 'custom':
                // use custom swimlanes as returned by the server
                swimlanes = GridSwimlanesData.calculateCustomSwimlanes(issuesData, swimlanesData.customSwimlanesData);
                break;

            case 'epic':
                swimlanes = GridSwimlanesData.calculateEpicSwimlanes(issuesData, swimlanesData.epicSwimlanesData);
                break;

            case 'project':
                swimlanes = GridSwimlanesData.calculateProjectSwimlanes(issuesData);
                break;

            case 'none':
            /* falls through */
            default:
                swimlanes = GridSwimlanesData.calculateNoneSwimlanes(issuesData);
                break;
        }

        // return config
        return {
            type: swimlanesData.swimlaneStrategy,
            swimlanes: swimlanes,
            rapidViewId: swimlanesData.rapidViewId
        };
    };

    /**
     * Generates the custom swimlanes
     */
    GridSwimlanesData.calculateCustomSwimlanes = function (issuesData, swimlanesData) {
        var issues = issuesData.issues;
        var swimlanes = swimlanesData.swimlanes;

        // calculate issueKeys contained in each swimlane (we transfer ids from the server to reduce transfer space)
        GridSwimlanesData.calculateSwimlaneIssueKeys(issues, swimlanes);

        // everything else is already in ok state
        return swimlanes;
    };

    /**
     * Calculates the swimlane issueKeys from the input issueIds
     * To reduce the transferred data we return ids instead of keys.
     */
    GridSwimlanesData.calculateSwimlaneIssueKeys = function (issues, swimlanes) {
        var issueIdsToKeys;

        _.each(swimlanes, function (swimlane) {
            swimlane.issueKeys = [];
            if (!_.isEmpty(swimlane.issueIds)) {
                // lazily calculate the mapping as some swimlane mappings are calculated on the client
                if (!issueIdsToKeys) {
                    issueIdsToKeys = {};
                    _.each(issues, function (issue) {
                        issueIdsToKeys[issue.id] = issue.key;
                    });
                }
                _.each(swimlane.issueIds, function (issueId) {
                    swimlane.issueKeys.push(issueIdsToKeys[issueId]);
                });
            }
        });
    };

    /**
     * Generates parent/child swimlanes
     */
    GridSwimlanesData.calculateParentChildSwimlanes = function (issuesData, swimlanesData, entityData) {
        // start filling the parent data lookup. add the missing parents first
        // key: parentId value: issueData
        var parentDataLookup = {};
        _.each(issuesData.missingParents, function (parent) {
            parentDataLookup[parent.id] = parent;
        });

        // following are the real parent issues
        var realParentIds = swimlanesData.parentIssueIds;

        // we'll add all parent ids into this array
        var parentIdsInOrder = [];

        // holds the children of each parent, with key=parentId, value=[childIssueKey]
        var childrenOfParents = {};

        // go through all issues, we either encounter the parent or at least one of its children
        // add the id to parentIdsInOrder the first time it is encountered (this ensures we keep a static order even in
        // cases where the parent issue has been filtered out
        _.each(issuesData.issues, function (issue) {
            if (issue.parentId) {
                parentIdsInOrder.push(issue.parentId);

                // add to children of parent
                if (!childrenOfParents[issue.parentId]) {
                    childrenOfParents[issue.parentId] = [];
                }
                childrenOfParents[issue.parentId].push(issue.key);
            } else {
                // add to parents in order
                parentIdsInOrder.push(issue.id);

                // also add to lookup
                parentDataLookup[issue.id] = issue;
            }
        });

        // remove duplicate parent ids (we know the same ids come in a row (as parents are ordered right before the children), so in a
        // way the array is already sorted
        parentIdsInOrder = _.uniq(parentIdsInOrder, true);

        // real parents in order
        var realParentsInOrder = _.intersection(parentIdsInOrder, realParentIds);

        // given the parents, now put together the swimlanes
        var swimlaneId = 1;
        var swimlanes = [];
        _.each(realParentsInOrder, function (parentId) {
            // fetch the children, could be empty
            var children = childrenOfParents[parentId] || [];

            // fetch the parent issue
            var parentIssue = parentDataLookup[parentId];

            var outOfSync = _.contains(swimlanesData.doneCandidates, parentId);

            // create a swimlane
            var swimlane = {
                id: swimlaneId++,
                name: parentIssue.key,
                description: parentIssue.summary,
                defaultSwimlane: false,
                parentKey: parentIssue.key,
                query: '',
                issueKeys: children,
                avatarUrl: entityData.types[parentIssue.typeId] ? entityData.types[parentIssue.typeId].typeUrl : parentIssue.typeUrl,
                parentOutOfSync: outOfSync
            };

            if (outOfSync) {
                // track that a move to done button is shown
                GH.WorkController.analytics.trigger('movetodonebutton.shown');
            }

            swimlanes.push(swimlane);
        });

        // find the parent issues that are leaves
        // TODO: use _.difference once available
        var leafIssues = _.filter(parentIdsInOrder, function (value) {
            return !_.include(realParentIds, value);
        });

        // put together the issue keys for the default swimlane
        var defaultLaneIssues = [];
        _.each(leafIssues, function (issueId) {
            var issue = parentDataLookup[issueId];
            defaultLaneIssues.push(issue.key);
        });
        var swimlane = {
            id: swimlaneId++,
            name: AJS.I18n.getText('gh.rapid.swimlane.parentchild.default.name'),
            description: '',
            defaultSwimlane: true,
            query: '',
            issueKeys: defaultLaneIssues
        };
        swimlanes.push(swimlane);

        return swimlanes;
    };

    /**
     * Generates assignee swimlanes
     */
    GridSwimlanesData.calculateAssigneeSwimlanes = function (issuesData, unscheduledFirst) {
        var issuesPerAssigneeId = {};
        var unassignedIssueKeys = [];

        // separate into groups
        _.each(issuesData.issues, function (issue) {
            if (issue.assignee) {
                issuesPerAssigneeId[issue.assignee] = issuesPerAssigneeId[issue.assignee] || [];
                issuesPerAssigneeId[issue.assignee].push(issue);
            } else {
                unassignedIssueKeys.push(issue.key);
            }
        });

        // put together the swimlanes for the assignees
        var swimlaneId = 1;
        var swimlanes = [];
        _.each(issuesPerAssigneeId, function (issues) {
            var assigneeName = issues[0].assigneeName;
            var swimlane = {
                id: swimlaneId++,
                name: assigneeName,
                lowercaseName: assigneeName.toLowerCase(), // used for sorting the lanes
                description: assigneeName,
                defaultSwimlane: false,
                query: '',
                issueKeys: _.map(issues, function (issue) {
                    return issue.key;
                })
            };
            swimlanes.push(swimlane);
        });

        // order the swimlanes according to the name
        swimlanes.sort(GridSwimlanesData.swimlaneNameComparator);

        // create default for leaves
        if (!_.isEmpty(unassignedIssueKeys)) {
            // default swimlane
            var swimlane = {
                id: swimlaneId++,
                name: AJS.I18n.getText('gh.issue.noassignee'),
                description: '',
                defaultSwimlane: false, // don't mark unassigned as defaultSwimlane, as we we want to show the swimlane label even if all displayed issues are unassigned
                query: '',
                issueKeys: unassignedIssueKeys
            };

            if (unscheduledFirst) {
                swimlanes.splice(0, 0, swimlane);
            } else {
                swimlanes.push(swimlane);
            }
        }
        return swimlanes;
    };

    /**
     * Generates epic swimlanes
     */
    GridSwimlanesData.calculateEpicSwimlanes = function (issuesData, swimlanesData) {
        var issuesPerEpic = {};
        var issuesWithoutEpic = [];
        var issuesWithEpic = {};
        var subtasks = [];

        // separate into groups
        _.each(issuesData.issues, function (issue) {
            if (issue.epic) {
                issuesPerEpic[issue.epic] = issuesPerEpic[issue.epic] || [];
                issuesPerEpic[issue.epic].push(issue.key);
                issuesWithEpic[issue.key] = issue.epic;
            } else if (issue.parentKey) {
                subtasks.push(issue);
            } else {
                // For Epic swimlanes, exclude issues of type Epic from appearing on the board
                if (issue.typeId !== GH.EpicConfig.getEpicIssueTypeId()) {
                    issuesWithoutEpic.push(issue.key);
                }
            }
        });

        // add subtasks to an epic group if their parent issue belongs to an epic
        _.each(subtasks, function (subtask) {
            var epicKey = issuesWithEpic[subtask.parentKey];
            if (epicKey) {
                issuesPerEpic[epicKey].push(subtask.key);
            } else {
                issuesWithoutEpic.push(subtask.key);
            }
        });

        var swimlanes = [];
        var swimlaneId = 1;
        _.each(swimlanesData.epics, function (epicKey) {
            var epicName = GH.EpicConfig.getEpicNameString(swimlanesData.epicNames[epicKey], epicKey);
            var swimlane = {
                id: swimlaneId++,
                name: epicName,
                lowercaseName: epicName.toLowerCase(), // used for sorting the lanes
                description: epicName,
                defaultSwimlane: false,
                query: '',
                issueKeys: issuesPerEpic[epicKey] || [] // note: should always have some issues
            };
            swimlanes.push(swimlane);
        });

        // create default for issues without an epic
        if (!_.isEmpty(issuesWithoutEpic)) {
            var swimlane = {
                id: swimlaneId++,
                name: AJS.I18n.getText('gh.epic.pending.title'),
                description: '',
                defaultSwimlane: true,
                query: '',
                issueKeys: issuesWithoutEpic
            };
            swimlanes.push(swimlane);
        }
        return swimlanes;
    };

    GridSwimlanesData.calculateProjectSwimlanes = function (issuesData) {
        var issuesPerProject = [];
        // separate into groups
        _.each(issuesData.issues, function (issue) {
            issuesPerProject[issue.projectId] = issuesPerProject[issue.projectId] || [];
            issuesPerProject[issue.projectId].push(issue.key);
        });

        var swimlanes = [];
        var swimlaneId = 1;

        _.each(issuesData.projects, function (project) {
            var swimlane = {
                id: swimlaneId++,
                name: project.key,
                lowercaseName: project.key, // project key is all uppercase so they will sort the same
                description: project.name,
                defaultSwimlane: false,
                query: '',
                avatarUrl: project.avatarUrl,
                issueKeys: issuesPerProject[project.id] || []
            };
            if (swimlane.issueKeys.length > 0) {
                swimlanes.push(swimlane);
            }
        });

        swimlanes.sort(GridSwimlanesData.swimlaneNameComparator);

        return swimlanes;
    };

    /**
     * Generates a single 'none' swimlane
     */
    GridSwimlanesData.calculateNoneSwimlanes = function (issuesData) {
        var noneSwimlane = {
            id: 1,
            name: 'None',
            description: 'None',
            defaultSwimlane: true,
            query: '',
            issueKeys: _.map(issuesData.issues, function (issue) {
                return issue.key;
            })
        };

        return [noneSwimlane];
    };

    /**
     * Sorts swimlanes alphabetically by lowercase name
     */
    GridSwimlanesData.swimlaneNameComparator = function (a, b) {
        return a.lowercaseName.localeCompare(b.lowercaseName);
    };

    return GridSwimlanesData;
});

AJS.namespace('GH.GridSwimlanesData', null, require('jira-agile/rapid/ui/work/grid-swimlanes-data'));