define('jira-agile/rapid/ui/version/version-view', ['require'], function (require) {
const VersionController = require('jira-agile/rapid/ui/version/version-controller');
const UiUtils = require('jira-agile/ui-utils');
const _ = require('underscore');

const VersionView = {};

VersionView.dropdowns = [];


/** @type {string} */
VersionView.datePickerFormat = undefined;

/** @param {string} datePickerFormat */
VersionView.setDatePickerFormat = function(datePickerFormat) {
    VersionView.datePickerFormat = datePickerFormat;
};

VersionView.init = function () {
    VersionView.initializeInlineEdits();
};

/**
 * @param {boolean} doNotUseCache
 * @return {AJS.$}
 */
VersionView.getContainer = function (doNotUseCache) {
    if (doNotUseCache || _.isUndefined(VersionView.container)) {
        VersionView.container = AJS.$("#ghx-version-column");
    }
    return VersionView.container;
};

VersionView.draw = function () {
    // fetch the version model
    const model = VersionController.getVersionModel();
    const $container = VersionView.getContainer(true);
    $container.html(GH.tpl.versionview.renderSkeleton({canCreateVersion: model.canUserCreateVersion()}));
    // bind the click listener one since we use event delegation
    var $scrollView = $container.find('.ghx-classification-scrollview');
    $scrollView.delegate('.ghx-classification-item', 'click', function (e) {
        // elements to be ignored
        if (AJS.$(e.target)
                .closest('.js-version-details-link, .js-version-actions, .js-editable-field, .js-editing-field, .aui-iconfont-calendar, input.text')
                .not('.js-edit-versionName-trigger')
                .length > 0) {
            return;
        }
        VersionView.toggleFiltering(AJS.$(this));
    });
    $scrollView.delegate('.js-version-details-toggle', 'click', function(event) {
        VersionView.toggleVersionDisplay(AJS.$(this));
        // prevent twixie toggle from also selecting version for filtering
        return false;
    });
    // ignore these - name uses anchor to inherit link colour, but is not a link
    $scrollView.delegate('.js-version-name', 'click', function (event) {
        event.preventDefault();
    });
    $scrollView.delegate('.js-try-again', 'click', function (event) {
        GH.BacklogController.updateVersionData();
    });

    GH.VersionQuickCreate.setAccessibleProjects(model.projects);

    // register the create epic dialog
    GH.VersionQuickCreate.registerCreateVersionDialog();
};

VersionView.renderVersionExpanded = function (versionId) {
    const versionModel = VersionController.getVersionModel();
    const versionsList = versionModel.getUnreleasedVersionList();
    const version = versionsList.find((v) => v.id == versionId);

    if (!version) {
        return;
    }

    const $container = VersionView.getContainer(true);
    const showEstimates = GH.RapidBoard.State.isScrumBoard();
    const $version = $container.find(`.ghx-classification-item[data-version-id=${versionId}]`);
    const $versionHeader = $version.find('.ghx-header');

    if (showEstimates) {
        VersionView.prepareEstimates(version);
    }

    const renderVersionsExpandedContext = {
        version,
        showEstimates,
        isLinkToDevStatusVersionAvailable: versionModel.isLinkToDevStatusVersionAvailable(),
        canEditVersions: versionModel.canUserCreateVersion(),
    };

    const $html = AJS.$(GH.tpl.versionview.renderVersionExpanded(renderVersionsExpandedContext));

    $html.insertAfter($versionHeader);
};

/**
 * Render the versions into view and initialise version Drag and Drop functionality.
 */
VersionView.updateView = function () {
    // render
    VersionView.renderVersions();

    // fix up dnd
    GH.PlanDragAndDrop.initializeClassificationDnDVersions();
};

VersionView.renderVersions = function () {

    var versionModel = VersionController.getVersionModel();

    // make sure the model is set
    if (_.isUndefined(versionModel) || !versionModel.isValid()) {
        return;
    }

    let isVersionExpandedById = {};
    let selectedVersionId = VersionController.getFilteredVersionId();
    let versionsList = versionModel.getUnreleasedVersionList();
    let showEstimates = GH.RapidBoard.State.isScrumBoard();

    _.each(versionsList, function (version) {
        isVersionExpandedById[version.id] = VersionView.getTwixieState(version.id) === 'open';
        if (showEstimates) {
            VersionView.prepareEstimates(version);
        }
    });

    VersionView.getContainer().find(".ghx-classification-scrollview").html(GH.tpl.versionview.renderVersions({
        versions: versionsList,
        isNoneSelected: GH.PlanIssueListFiltering.isNoneVersionFilterId(selectedVersionId),
        isAllSelected: !selectedVersionId,
        selectedVersionId: selectedVersionId,
        isVersionExpandedById: isVersionExpandedById,
        // the permission to edit is the same as the permission to create
        canEditVersions: versionModel.canUserCreateVersion(),
        showProjects: versionModel.hasMultiProjects(),
        isLinkToDevStatusVersionAvailable: versionModel.isLinkToDevStatusVersionAvailable(),
        showEstimates,
    }));

    // dropdowns
    AJS.$(".js-version-actions").each(function () {
        var trigger = this;
        var contentId = "#" + AJS.$(trigger).attr('id').replace(/^dd-trigger-/, "dd-content-");
        var content = AJS.$(contentId);
        var dropdown = AJS.Dropdown.create({
            trigger: trigger,
            content: content,
            alignment: AJS.LEFT
        });
        AJS.$(dropdown).bind("showLayer", function () {
            GH.RapidBoard.Util.InlineEditable.submitActiveEdits();
        });

        VersionView.dropdowns.push(dropdown[0]);
    });
};

VersionView.prepareEstimates = function(version) {
    // render the estimate vs done values
    version.versionStats.estimateDone = GH.BacklogStatistics.formatStatisticForRendering(GH.BacklogModel.estimationStatistic, version.versionStats.doneEstimate);
    version.versionStats.estimateTotal = GH.BacklogStatistics.formatStatisticForRendering(GH.BacklogModel.estimationStatistic, version.versionStats.totalEstimate);
    version.versionStats.isIssueCountStatistic = GH.BacklogModel.estimationStatistic.typeId === 'issueCount';
    version.versionStats.estimateStatisticName = GH.BacklogModel.estimationStatistic.name;
};

VersionView.renderError = function () {
    let selectedVersionId = VersionController.getFilteredVersionId();

    VersionView.getContainer()
        .find(".ghx-classification-scrollview")
        .html(GH.tpl.versionview.renderError({
            isNoneSelected: GH.PlanIssueListFiltering.isNoneVersionFilterId(selectedVersionId),
            isAllSelected: !selectedVersionId,
        }));
};

VersionView.toggleFiltering = function ($clickedVersion) {

    // finish ongoing edits
    GH.RapidBoard.Util.InlineEditable.submitActiveEdits();

    VersionView.deselectAllVersions();

    var versionId = $clickedVersion.attr('data-version-id');
    var selectedVersionId;

    if (VersionController.getFilteredVersionId() === versionId) {
        selectedVersionId = null;
    } else {
        selectedVersionId = versionId;
        AJS.$('#ghx-version-column').find('.ghx-classification-all').removeClass('ghx-selected');
        $clickedVersion.addClass('ghx-selected');
    }

    // update version for filtering
    VersionController.setFilteredVersionId(selectedVersionId);

    // update state
    GH.RapidBoard.State.pushState();

    // apply version filter
    GH.BacklogController.updateVersionFiltering(false);

};

VersionView.deselectAllVersions = function () {
    // deselect all version
    var $versionsColumn = AJS.$('#ghx-version-column');
    var $versions = $versionsColumn.find('.ghx-classification-item');
    $versions.removeClass('ghx-selected');
    $versionsColumn.find('.ghx-classification-all').addClass('ghx-selected');
};

VersionView.setTwixieState = function (versionId, value) {
    var key = 'versionTwix-' + versionId;
    GH.BoardState.setPerViewValue(key, value);
};

VersionView.getTwixieState = function (versionId) {
    var key = 'versionTwix-' + versionId;
    return GH.BoardState.getPerViewValue(key, 'closed');
};

// Version toggle state used for Expand Collapse
VersionView.toggleVersionDisplay = function ($clickedVersion) {
    var $version = $clickedVersion.parents('.ghx-classification-item');
    VersionView.toggleVersionDisplayByElementWithAnimation($version);
};

VersionView.toggleVersionDisplayByElement = function($version) {
    const versionId = $version.attr('data-version-id');
    if ($version.hasClass('ghx-open')){
        $version.removeClass('ghx-open').addClass('ghx-closed');
        VersionView.setTwixieState(versionId, 'closed');
        $version.find('.ghx-expanded').remove();
    } else if ($version.hasClass('ghx-closed')){
        VersionView.renderVersionExpanded(versionId);
        $version.removeClass('ghx-closed').addClass('ghx-open');
        VersionView.setTwixieState(versionId, 'open');
    }
    UiUtils.updateExpanderA11yAttrs($version);
};

VersionView.toggleVersionDisplayByElementWithAnimation = function($version) {
    var versionId = $version.attr('data-version-id');
    var isOpen = $version.hasClass('ghx-open');
    if (isOpen){
        const expandedElement = $version.find(".ghx-expanded");
        GH.RapidBoard.Animation.animateHeight(expandedElement, 195, 0).done(function() {
            $version.removeClass('ghx-open').addClass('ghx-closed');
            VersionView.setTwixieState(versionId, 'closed');
            expandedElement.remove();
            UiUtils.updateExpanderA11yAttrs($version);
        });
        UiUtils.updateExpanderA11yAttrs($version, UiUtils.Aria.COLLAPSED);
    } else if ($version.hasClass('ghx-closed')){
        VersionView.renderVersionExpanded(versionId);
        const expandedElement = $version.find(".ghx-expanded");
        $version.removeClass('ghx-closed').addClass('ghx-open');
        GH.RapidBoard.Animation.animateHeight(expandedElement, 0, 195).done(function() {
            VersionView.setTwixieState(versionId, 'open');
            UiUtils.updateExpanderA11yAttrs($version);
        });
        UiUtils.updateExpanderA11yAttrs($version, UiUtils.Aria.EXPANDED);
    }
    VersionController.sendAnalytics('versionToggled', {
        isOpen: !isOpen
    });
};

VersionView.hideDropdown = function () {
    _.each(VersionView.dropdowns, function (dropdown) {
        dropdown.hide();
    });
};

VersionView.initializeInlineEdits = function () {

    // Validation for name, start and end date. These function also extract the value from the dom
    // edit version name
    GH.RapidBoard.Util.InlineEditable.register('.js-edit-version-name', {
        renderView: GH.RapidBoard.Util.InlineEditable.remoteTriggeredTextRenderView,
        preEdit: GH.RapidBoard.Util.InlineEditable.cancelActiveEdits,
        getViewElement: function (event) {
            var versionId = AJS.$(this).parents('.js-version-actions-list').attr('data-version-id');
            return AJS.$('.ghx-classification-item[data-version-id="' + versionId + '"]').find('.js-edit-versionName-trigger');
        },
        getData: function (viewElement) {
            var editData = GH.RapidBoard.Util.InlineEditable.defaultFieldData(viewElement);
            // add the version id to the data structure to be able to recognize it on save
            editData.versionId = viewElement.parents('.ghx-classification-item').attr('data-version-id');

            // force other edits to close
            GH.RapidBoard.Util.InlineEditable.submitActiveEdits();

            // set a class that we're editing the name
            AJS.$(viewElement).closest('.ghx-header').addClass('ghx-editing-name');

            editData.maxLength = 30;
            return editData;
        },
        validate: function (editData) {
            var newValue = editData.editElement.val();
            if (!GH.Validation.notBlank(editData.editElement, AJS.I18n.getText('gh.version.name.error.required'))) {
                return false;
            }
            editData.newValue = newValue;
            return true;
        },
        save: function (editData) {
            VersionView.updateVersion(editData.versionId, {name: editData.newValue}).done(function() {
                GH.RapidBoard.Util.InlineEditable.updateView(editData);
            });
        },
        postEdit: function (editData) {
            // remove class that we're editing the name
            AJS.$(editData.viewElement).closest('.ghx-header').removeClass('ghx-editing-name');
        }
    });

    var inlineEditConfig = {
        getData: function (viewElement) {
            // force other edits to close
            GH.RapidBoard.Util.InlineEditable.cancelActiveEdits();

            var editData = GH.RapidBoard.Util.InlineEditable.defaultFieldData(viewElement);
            // add the version id to the data structure to be able to recognize it on save
            editData.versionId = viewElement.parents('.ghx-classification-item').attr('data-version-id');

            return editData;
        }
    };

    GH.RapidBoard.Util.InlineEditable.register('#ghx-version-column .js-edit-description-trigger',
        _.extend({}, inlineEditConfig, {
            save: function (editData) {
                VersionView.updateVersion(editData.versionId, {description: editData.newValue});
                GH.RapidBoard.Util.InlineEditable.updateView(editData);
            }
        }));

    var cal = new Calendar(); // GHS-5829 Need to call new calendar to set up short month names and day names when they are not defined
    var releaseDateInlineFieldConfig = _.extend({}, inlineEditConfig, {
        renderEdit: GH.RapidBoard.Util.InlineEditable.renderDatePickerInlineEdit,
        activate: function (editData) {
            GH.RapidBoard.Util.InlineEditable.datePickerActivate(editData, {
                ifFormat: VersionView.datePickerFormat,
                showsTime: false
            });
        },
        validate: function (editData) {
            // extract the date from the input and fill the data structure to fake regular behaviour
            editData.newValue = editData.editElement.find('input').val();
            return true;
        },
        postEdit: function(editData) {
            // remove errors from both date controls
            AJS.$(editData.viewElement).closest('.js-version-date').siblings('.ghx-error').remove();
        }
    });

    GH.RapidBoard.Util.InlineEditable.register('#ghx-version-column .js-edit-startDateFormatted-trigger',
        _.extend({}, releaseDateInlineFieldConfig, {
            save: function (editData) {
                VersionView.updateVersion(editData.versionId, {startDateFormatted: editData.newValue}).done(function() {
                    GH.RapidBoard.Util.InlineEditable.updateView(editData);
                });
            }
        }));
    GH.RapidBoard.Util.InlineEditable.register('#ghx-version-column .js-edit-releaseDateFormatted-trigger',
        _.extend({}, releaseDateInlineFieldConfig, {
            save: function (editData) {
                VersionView.updateVersion(editData.versionId, {releaseDateFormatted: editData.newValue}).done(function() {
                    GH.RapidBoard.Util.InlineEditable.updateView(editData);
                });
            }
        }));
};

/**
 * @param {number} versionId
 * @param {Object} newProperties
 * @return {AJS.$.Deferred.promise}
 */
VersionView.updateVersion = function(versionId, newProperties) {
    var parentSelector = '.ghx-classification-item[data-version-id="' + versionId + '"]';

    return VersionController.updateVersion(versionId, newProperties, {
        'name': parentSelector + ' .js-version-name',
        'description': parentSelector + ' .js-version-description',
        'startDate': parentSelector + ' .js-version-start-date',
        'releaseDate': parentSelector + ' .js-version-release-date'
    }, function () {
        AJS.$(parentSelector + " .ghx-error").empty();
    }).done(function () {
        var updatedProperties = _.keys(newProperties);
        VersionController.sendAnalytics('versionUpdated', {
            updatedProperty: _.first(updatedProperties),
            propertiesCount: updatedProperties.length
        });
    });
};

VersionView.renderNewVersions = function(versions) {
    var versionColumn = AJS.$("#ghx-version-column");
    _.each(versions, function(version) {
        var $version = versionColumn.find('.ghx-classification-item[data-version-id="' + version.id + '"]');
        if (! $version.hasClass('ghx-open')) {
            VersionView.toggleVersionDisplayByElement($version);
        }

        GH.RapidBoard.Animation.animateHeight($version, 0, 251, GH.RapidBoard.Animation.LONG_DURATION);
    });
};

return VersionView;

});
