define('jira-agile/rapid/ui/version/version-controller', ['require'], function (require) {

let _ = require('underscore');
let logger = require('jira/util/logger');
let AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');

const VersionController = {};

/** @type {{versionList: GH.VersionsModel, projects: Array.<{id: number, name: string}>, canCreateVersion: boolean}} */
VersionController.model = null;


/**
 * @type module:jira-agile/rapid/analytics-tracker
 */
VersionController.analytics = new AnalyticsTracker("gh.rapidboard.versions", "versionCount", "");

var analytics = require('jira/analytics');
var AnalyticsHelper = require('jira-agile/rapid/ui/plan/analytics-helper');
VersionController.sendAnalytics = function (eventKey, data, skipDefaults) {
    if(!eventKey) {
        throw new Error("You must specify a category");
    }
    var eventName = 'jira-software.' + GH.RapidBoard.State.getBoardType() + '.plan.versions.' + eventKey;
    var defaultData = AnalyticsHelper.baseEventData();

    analytics.send({
        name: eventName,
        properties: skipDefaults ? data : _.extend(defaultData, data)
    });
};


/**
 * Initializes the version controller object
 */
VersionController.init = function() {
    VersionController.model = new GH.VersionsModel([]);

    // handle a version created event
    var $GH = AJS.$(GH);
    $GH.bind(GH.VersionQuickCreate.EVENT_VERSION_CREATED, function(evt, data) {
        VersionController.handleVersionCreated(data.createdVersion);
    });

    GH.VersionView.init();
};

/**
 * Set the projects data
 */
VersionController.setProjects = function(projects) {
    VersionController.model.setProjects(projects);
};

/**
 * Set the versions data
 */
VersionController.setVersions = function(versions) {
    VersionController.model.updateRawData(versions);
};

/**
 * Returns the model managed by this controller
 * @return {{versionList: GH.VersionsModel, projects: Array.<{id: number, name: string}>, canCreateVersion: boolean}} model
 */
VersionController.getVersionModel = function() {
    return VersionController.model;
};

/**
 * Fires off an analytics event containing the number of versions on the board
 * Happens once per page load
 */
// TODO: test that this still works!
VersionController.registerVersionsCount = function() {
    // track number of epics on board
    var model = VersionController.model;
    var versions = model.getUnreleasedVersionList().length;
    VersionController._triggerVersionCount(versions);
};

/** We only want to track this once per page load */
VersionController._triggerVersionCount = _.once(function(versionCount) {
    VersionController.analytics.trigger(versionCount);
});

VersionController.getFilteredVersionId = function() {
    return GH.BoardState.getPerViewValue('planSelectedVersion', null);
};

VersionController.setFilteredVersionId = function(versionId, skipAnalytics) {
    GH.BoardState.setPerViewValue('planSelectedVersion', versionId);

    if(!skipAnalytics) {
        VersionController.sendAnalytics(versionId ? "versionFilterActive" : "versionFilterInactive");
    }
};

VersionController.hasFilteredVersion = function() {
    return VersionController.getFilteredVersionId() != null;
};

/**
 * Validates the currently selected version, updates the URL if the current selection is invalid
 */
VersionController.validateSelectedVersion = function() {
    // fetch the current state
    var versionList = VersionController.getVersionModel();
    var filteredVersionId = VersionController.getFilteredVersionId();

    // is the currently selected version valid?
    var versionIdValid = _.isNull(filteredVersionId) || GH.PlanIssueListFiltering.isNoneVersionFilterId(filteredVersionId) && !versionList.isEmpty()
        || versionList.containsId(filteredVersionId);

    // clear the filtered version if invalid
    if (!versionIdValid) {
        // remove the version filtering id from the board settings if it doesn't exist
        VersionController.setFilteredVersionId(null);
        GH.RapidBoard.UrlState.replaceState();
    }
};


/**
 * Sets issues fix versions for each supplied issues with the given one.
 *
 * @param {number} versionId
 * @param {Array.<{key: string}>} issues
 */
VersionController.setIssuesVersion = function(versionId, issues, filteredIssuesCount) {
    if (!_.isEmpty(issues)) {
        _.each(issues, function(issue) {
            issue.fixVersions = [versionId];
        });
        GH.BacklogController.updateIssues(issues);

    }

    var issuesKeys = _.pluck(issues, 'key');

    // then update on the server
    VersionController.setIssuesVersionOnServer(versionId, issuesKeys).done(function () {
        // put up a success message
        GH.Notification.showSuccess(getProgressiveMessageFor('gh.version.issues.added', issues, filteredIssuesCount));

        if (GH.PlanController.isActive()) {
            // update the versions data (earlier the data on the server wouldn't be correct)
            GH.BacklogController.updateVersionData();

            // Refresh the detail view
            VersionController._reloadDetailsView(GH.PlanController);
        }

        VersionController.sendAnalytics("setIssuesVersion", {issueCount: issues.length});

    }).fail(function() {
            if (GH.PlanController.isActive()) {
                // something didn't quite work out. reload to get back into a good state
                GH.BacklogController.loadData();
            }
        });
};

/**
 * Removes all fix versions form the given issues.
 *
 * @param {Array.<{key: string}>} issues
 */
VersionController.clearIssuesVersion = function(issues) {

    if (_.isEmpty(issues)) {
        return;
    }
    var issuesToBeUpdated = [];
    _.each(issues, function(issue) {
        if(issue.fixVersions && issue.fixVersions.length > 0){
            issue.fixVersions = [];
            issuesToBeUpdated.push(issue);
        }
    });

    if(_.isEmpty(issuesToBeUpdated)){
        return;
    }

    GH.BacklogController.updateIssues(issuesToBeUpdated);

    var issuesKeys = _.pluck(issues, 'key');

    VersionController.clearIssuesVersionOnServer(issuesKeys).done(function () {
            // put up a success message
            GH.Notification.showSuccess(getProgressiveMessageFor('gh.version.issues.removed', issues));

            if (GH.PlanController.isActive()) {
                // reload the epics data
                GH.BacklogController.updateVersionData();

                // reload the detail view
                VersionController._reloadDetailsView(GH.PlanController);

            } else if (GH.WorkController.isActive()) {
                // just reload the detail view
                VersionController._reloadDetailsView(GH.WorkController);
            }
        VersionController.sendAnalytics("clearIssuesVersion", {issueCount: issues.length});
    }).fail(function() {
            if (GH.PlanController.isActive()) {
                // something didn't quite work out. reload to get back into a good state
                GH.BacklogController.loadData();
            }
        });
};

/**
 * @param {number} versionId
 * @param {Object} newProperties
 * @param {Object.<string, string>} errorContextMap css selector indexed by REST resource name
 * @param {function} clearErrorFn executed to remove all the error messages
 * @return {AJS.$.Deferred.promise}
 */
VersionController.updateVersion = function (versionId, newProperties, errorContextMap, clearErrorFn) {
    var versionList = VersionController.getVersionModel();
    var version = versionList.getVersion(versionId);
    var updated = _.extend({}, version, newProperties);

    clearErrorFn();
    return VersionController.updateVersionOnServer(updated, errorContextMap)
        .done((resp) => {
            // Now we have to update backlog and then version data, since it is separated
            GH.BacklogController.loadData(resp).done(GH.BacklogController.updateVersionData);
        });

};

/**
 * @param {{id: number, project: {id: number}, name: string, description: string, releaseDateFormatted: string, startDateFormatted: string}} version
 * @param {Object.<string, string>} errorContextMap css selector indexed by REST resource name
 * @return {AJS.$.Deferred.promise} resolved when the request returns
 */
VersionController.updateVersionOnServer = function (version, errorContextMap) {
    return GH.Ajax.post({
        url: '/versions/' + version.id,
        data: {
            id: version.id,
            projectId: version.project.id,
            name: version.name,
            description: version.description,
            releaseDate: version.releaseDateFormatted,
            startDate: version.startDateFormatted
        },
        errorContextMap: errorContextMap
    });
};

/**
 * Sets a fix version for the supplied issues on the server side.
 *
 * @param {number} versionId
 * @param {Array.<string>} issueKeys
 * @return {AJS.$.Deferred.promise} that will be resolved when the request returns
 */
VersionController.setIssuesVersionOnServer = function (versionId, issueKeys) {
    return GH.Ajax.put({
        url: '/xboard/issue/fixversion/set',
        data: {
            issueKeys: issueKeys,
            versionId: versionId
        }
    });
};

/**
 * Clears fix version for the supplied issues on the server side.
 *
 * @param {Array.<string>} issueKeys
 * @return {AJS.$.Deferred.promise} that will be resolved when the request returns
 */
VersionController.clearIssuesVersionOnServer = function(issueKeys) {
    return GH.Ajax.put({
        url: '/xboard/issue/fixversion/clear',
        data: {
            issueKeys: issueKeys
        }
    });
};


VersionController.handleVersionCreated = function(createdVersion) {
    if (!GH.BacklogController.visible) {
        return;
    }

    // Ajax requests on Firefox fail if issued through an (escape) key event, IF the key event propagation is not stopped.
    // This event could come from quick create (create an issue with "create another" selected, then hit escape)
    // Workaround: do the actual action after a timeout, which will happen outside the keyboard event.

    setTimeout(function(){
        VersionController.handleVersionCreatedImpl(createdVersion);
    }, 0);
};


VersionController.handleVersionCreatedImpl = function(createdVersion) {
    GH.BacklogController.updateVersionData().done(function() {
        GH.VersionView.renderNewVersions([createdVersion]);
    });
    VersionController.sendAnalytics('versionCreated');
};

VersionController._reloadDetailsView = function(controller) {
    var EditableDetailsViewReloadReason = require('jira-agile/rapid/ui/detail/inlineedit/details-view-reload-reason');
    controller.reloadDetailView(EditableDetailsViewReloadReason.VERSION_PANEL_CHANGED);
};


/**
 * Returns a message which depends on how many issues were altered.
 * It uses given i18n key and passes to getText number of issues and up to 2 issue keys:
 *  AJS.I18n.getText(i18nKey, issues.length, issue1Key, issue2Key);
 *
 * Take a look into impl which i18n keys are supported.
 *
 * @param i18nKey {string}
 * @param issues {IssueData[]} list of issue objects
 * @returns {string} progressive message
 */
function getProgressiveMessageFor(i18nKey, issues, issuesRejected) {
    issues = issues || [];
    let i18n = [];

    let length = issues.length;
    let issue1Key = (length >= 1) ? issues[0].key : undefined;
    let issue2Key = (length >= 2) ? issues[1].key : undefined;

    // because of JIRA's i18n build-time transformer i18n keys have to be explicitly written
    switch (i18nKey) {
        case 'gh.version.issues.removed':
            logger.trace('gh.version.issues.removed', issues.length, issue1Key, issue2Key);
            i18n.push(AJS.I18n.getText('gh.version.issues.removed', issues.length, issue1Key, issue2Key));
            break;
        case 'gh.version.issues.added':
            logger.trace('gh.version.issues.added', issues.length, issue1Key, issue2Key);
            i18n.push(AJS.I18n.getText('gh.version.issues.added', issues.length, issue1Key, issue2Key));
            break;
    }

    if(issuesRejected) {
        i18n.push(AJS.I18n.getText('gh.version.issues.skipped'))
    }

    return i18n.join("<br /><br />");
}


return VersionController;

});
