/**
 * globals
 *
 * JIRA.Projects.Subnavigator, AJS.I18n, GH.RapidBoard.State
 */

/**
 * This module is repsible for building UI for the the card printing screen. It handles all DOM manipulations.
 *
 * @module jira-agile/rapid/ui/print/card-printing-view
 * @see module:underscore
 * @see module:jquery
 * @see module:ira-agile/rapid/ui/print/card-printing-analytic
 */
define('jira-agile/rapid/ui/print/card-printing-view', [
    'underscore',
    'jquery',
    'jira-agile/rapid/ui/component/dynamic-colors',
    'jira-agile/rapid/ui/print/card-printing-analytic'
], function(
    _,
    $,
    DynamicColors,
    CardPrintingAnalytic
) {
    'use strict';

    // "constants"
    // 8 issues / A4 paper
    // lets set threshold 50 papers ==> 400 issues
    var ISSUE_COUNT_WARNING_THRESHOLD = 400;

    var RapidBoardState = GH.RapidBoard.State;

    var CardPrintingView = {
        /**
         * Displays the card printing screen for the given list of issues. To be called when display the print screen
         * for the first time.
         *
         * @param {Object} params
         * @param {Object[]} params.issues list of issues to be displayed on the initial page
         * @param {Object[]} params.filterItems items to build filter dropdown. If not provided (or empty),
         *  there will be no filter built.
         * @param {string} params.selectedItem ID of the default value to be displayed on the filter dropdown.
         * @param {Object} params.epics epic data object which *maps* epic key to epic issue.
         *
         * @param {Object} params.actionContext
         * @param {string} params.actionContext.triggeredFrom for example: 'context-menu', 'board-dropdown-menu'.
         * @param {string} params.actionContext.viewMode 'plan', 'work'
         * @param {string} params.actionContext.boardType 'scrum', 'kanban'
         */
        showPrintScreen: function(params) {
            buildAndDisplayPrintScreen(params);
        },

        /**
         * Performs initialization after the new window loaded.
         */
        initPrintScreen: function() {
            var printContext = window.opener.cardPrintingContext;
            // unneeded anymore, so remove it
            // using delete keyword on window's properties will create the error "TypeError: Object doesn't support this property or method"
            window.opener.cardPrintingContext = undefined;

            showCardSizeDropdown();
            showFilterBox(printContext.filterItems, printContext.selectedItem);
            registerEventHandlers(printContext);
        }
    };

    /**
     * Constructs html of the screen and displays it on a new browser tab.
     *
     * @param {Object} params see showPrintScreen method
     */
    function buildAndDisplayPrintScreen(params) {
        // call soy to generate the print layout
        var html = GH.tpl.rapid.print.printCardScreen($.extend({}, params, {
            showEpic : RapidBoardState.getEpicShownOnRapidBoard(),
            cardSize: RapidBoardState.getLastCardPrintingSize(),
            warnIssueCount: shouldShowIssueCountWarning(_.size(params.issues))
        }));

        // what are we going to do?
        // - open a new (blank) tab - yay for better printing UX!
        // - copy the whole HEAD tag of the current page and inject it into the new tab
        //   that is OK because in production, all web resources (css, js) of the current page have already been loaded
        //   and cached on the browser, thus, there will be no server call on the new tab
        //
        //      why need to inject the HEAD tag of the current page? because we want to apply css and run JS on that
        //      window. But why don't we make a simple action that returns an empty page which loads only web resource
        //      needed for printing then open it in new the tab and write the page body into it? Well:
        //          + first from the parent tab, knowing when the child tab finishes loading the page in order to inject
        //              the content is not straightforward. Can use jquery.ready() from the parent. One possible way is
        //              to have a js on the child tab calling back to its window.parent ...
        //          + second, what to show if the user bookmarks the url of the print tab?
        //
        // - write the print screen html into the BODY tag of the new tab
        // - bootstrap the event handler on the new tab.

        var newPageHead = buildHtmlHeadContentOfCurrentPage();

        // pass these values to the new window so that they can be used by the showFilterBox method.
        // hmm, we go global here! But I don't see any better way
        // Why don't put those variables to the new window? Because it doesn't work on IE! In IE, it seems that we need
        // to wait until the document is ready in order to assign global variables to it.
        // So, we just put these variable in the parent window and have them refered from the child window.
        window.cardPrintingContext = {
            filterItems: params.filterItems,
            selectedItem: params.selectedItem,
            actionContext: params.actionContext,
            AJS: AJS
        };

        var newWindow = window.open();

        // $(newWindow.document).append won't work here. It will create a javascript stack overflow (recursion) error.
        /* jshint -W015 */
        // ignore all indentation warnings of the following construct, that is what we want to see, e.g. html code
        newWindow.document.write([
            getDoctypeOfCurrentPage(),
            '<html>',
            '<head>',
                // This will NOT work in IE9, works in IE10
                newPageHead,
            '</head>',
            '<body id="jira" class="ghx-print-card-body">',
                html,
                '<script type="text/javascript">',
                '   var CardPrintingView = require("jira-agile/rapid/ui/print/card-printing-view");',
                '   CardPrintingView.initPrintScreen();',

                // what's that stop()? this is to fix an issue with firefox: in FF, if we open a new (blank) tab and write
                // the content (any content, you can try with a simple <html><head></head><body></body></html>) into it,
                // it will display the url of the current page (the trigger page) on the address bar then start loading
                // (the loading indicator spins) indefinitely. The worse thing is that, when we inject JIRA HEAD, that
                // spinning loading will prevent user from clicking on "print": clicking on print, nothing will happen, then
                // you hit the stop button on the browser, the print dialog will appear.
                // So, we make a window.stop() here will do the trick. That is OK because this line of code will be executed
                // at the latest, after all needed resources are loaded.
                // why this if()? because IE does not have this function. Luckily, IE doesn't have problem like FF
                '   if (window.stop) {',
                '       window.stop();',
            '       }',
                '</script>',
            '</body></html>'
        ].join(''));

        if (params.cardColorStrategy === "custom") {
            newWindow.document.onreadystatechange = () => {
                DynamicColors.loadColors($(newWindow.document.body));
            };
        }

        // this "close()" is essential for IE to have the window.print() work e.g. if we don't do this, window.print()
        // will never run.
        newWindow.document.close();
        /* jshint +W015 */
    }

    /**
     * Looks through the HEAD tag of the current page, concatenates all its child tags into a string, all invalid tags
     * are eliminated (JIRA puts some invalid tags into the HEAD).
     *
     * @returns {string} content of the HEAD tag as string.
     */
    function buildHtmlHeadContentOfCurrentPage() {
        var headArr = [];
        $('head').children().each(function(idx, elem) {
            // Why we have to spend effort on this? Well, because our JIRA generates the following tag in the html HEAD:
            //   <div style="display: none;"></div>
            // !!!
            // if we inject that tag into html head, chrome will also injects the whole head content into BODY! and our
            // <script> block will not get called! Firefox will complain with the message "stray end tag head"
            if ($(elem).is('title, style, base, link, meta, script, noscript')) { // these are valid tags in html HEAD
                headArr.push($(elem)[0].outerHTML);
            }
        });
        return headArr.join('');
    }

    /**
     * Construct the doctype of the current page as string.
     *
     * @returns {string}
     */
    function getDoctypeOfCurrentPage() {
        // why can we just return <!DOCTYPE html>?
        // who knows one day, JIRA wants to change doctype? It that happens, this screen will be safe
        var doctype = document.doctype;
        /* jshint -W015*/ // (indentation)
        return [
            '<!DOCTYPE ',
            doctype.name,
                (doctype.publicId ? ' PUBLIC "' +  doctype.publicId + '"' : ''),
                (!doctype.publicId && doctype.systemId ? ' SYSTEM' : ''),
                (doctype.systemId ? ' "' + doctype.systemId + '"' : ''),
            '>'
        ].join('');
        /* jshint +W015 */
    }

    /**
     * @param {int} count
     * @returns {boolean} true if the count greater or equals to the limit.
     */
    function shouldShowIssueCountWarning(count) {
        return count >= ISSUE_COUNT_WARNING_THRESHOLD;
    }

    /**
     * Shows the card-size dropdown on the page and handle its events.
     */
    function showCardSizeDropdown() {
        // draw card size dropdown
        var sizes = [
            {id: 'small', label: AJS.I18n.getText('gh.rapid.common.size.small')},
            {id: 'medium', label: AJS.I18n.getText('gh.rapid.common.size.medium')},
            {id: 'large', label: AJS.I18n.getText('gh.rapid.common.size.large')}
        ];

        var cardSizeDropdown = new JIRA.Projects.Subnavigator({
            id: 'ghx-print-card-size-dropdown',
            triggerPlaceholder: '#ghx-print-toolbar .ghx-print-card-size-trigger',
            contentPlaceholder: '#ghx-print-toolbar .ghx-print-card-size-options',
            itemGroups: sizes,
            // use the last used size, if none, use the size "small"
            selectedItem: RapidBoardState.getLastCardPrintingSize(),
            hideSelectedItem: true
        });
        cardSizeDropdown.show();

        // handle card size change.
        cardSizeDropdown.on('itemSelected', function (e) {
            e.preventDefault();

            var size = e.item.id;
            $('.ghx-print-content-sizing').removeClass().addClass('ghx-print-content-sizing ghx-print-' + size);
            RapidBoardState.setLastCardPrintingSize(size);
        });
    }

    function registerEventHandlers(printContext) {
        $('#ghx-print-toolbar')
            .on('click', '#ghx-print-card-print-btn', function() {
                window.print();
            });
        $('#ghx-print-toolbar-issue-count').tooltip({
            title: function () {
                if ($('#ghx-print-toolbar-issue-count').hasClass('warning')) {
                    return AJS.I18n.getText('gh.rapid.operations.print.cards.warning.too.much.issues');
                }
                return '';
            }
        });

        var analytic = new CardPrintingAnalytic({
            actionContext: printContext.actionContext,
            AJS: printContext.AJS
        });
        analytic.registerForPrintEvent();
    }

    /**
     * Builds and shows the filter box on the screen if the filterItems array is provided.
     *
     * @param {Object[]} filterItems items to build filter dropdown. If not provided (or empty), there is no filter built.
     * @param {string} selectedItem ID of the default value to be displayed on the filter dropdown.
     */
    function showFilterBox(filterItems, selectedItem) {
        if (!_.isEmpty(filterItems)) {
            // OK, we can use pure AUI dropdown2 here. But for being consistent with the current screen
            // we use the same JIRA.Projects.Subnavigator
            var filterDropdown = new JIRA.Projects.Subnavigator({
                id: 'ghx-print-filter-cbx',
                triggerPlaceholder: '#ghx-print-toolbar .ghx-print-filter-trigger',
                contentPlaceholder: '#ghx-print-toolbar .ghx-print-filter-options',
                itemGroups: filterItems,
                selectedItem: selectedItem,
                hideSelectedItem: true
            });
            filterDropdown.show();

            CardPrintingAnalytic.takeNoteSelectedFilterId(selectedItem);

            filterDropdown.on('itemSelected', function(e) {
                e.preventDefault();

                $('.ghx-card').addClass('ghx-card-filtered-out');

                var filterId = e.item.id;
                var visibleSelector = '.sprint-' + filterId;
                if (filterId === 'backlog') {
                    visibleSelector = '.backlog';
                } else if (filterId === 'all') {
                    visibleSelector = '.ghx-card';
                }
                var visibleIssues = $(visibleSelector);
                visibleIssues.removeClass('ghx-card-filtered-out');
                updateOnIssueCountChange(visibleIssues.length);

                CardPrintingAnalytic.takeNoteSelectedFilterId(filterId);
            });
        } else {
            $('#ghx-print-toolbar').find('.ghx-print-filter-label').css('display', 'none');
        }
    }

    /**
     * Shows/hide the issue count label, showing the warning icon and a tooltip if there is too many issues.
     *
     * @param {int} count
     */
    function updateOnIssueCountChange(count) {
        $('#ghx-print-toolbar').find('.ghx-print-toolbar-item').css('display', !count ? 'none' : '');

        // update issue count
        if (count) {
            var countLabel = $('#ghx-print-toolbar-issue-count');
            countLabel.empty().append(AJS.I18n.getText('gh.rapid.common.issue.count', count));
            if (shouldShowIssueCountWarning(count)) {
                countLabel.addClass('warning');
            } else {
                countLabel.removeClass('warning');
            }
        }
    }

    return CardPrintingView;
});
