/**
 * This module is responsible for receiving print request from board views (plan view, work view of scrumm/kanban boards),
 * looking up necessary data, telling the CardPrintingView to show the screen and listening/reacting for user events
 * from the screen.
 *
 * @module jira-agile/rapid/ui/print/card-printing-controller
 * @see module:underscore
 * @see module:jira-agile/rapid/ui/print/card-printing-view
 */
define('jira-agile/rapid/ui/print/card-printing-controller', ['underscore', 'jquery', 'jira-agile/rapid/ui/print/card-printing-view'], function (_, $, CardPrintingView) {
    'use strict';

    var ALL_FILTER_ID = 'all';
    var BACKLOG_FILTER_ID = 'backlog';

    var CardPrintingController = {
        /**
         * Allows users to print selected issues from the current view (e.g to be called from the right-click context menu).
         *
         * @param {{issues: Object[], epics: map, viewMode: string, boardType: string}} data.
         * @param {Object[]} data.issues array of selected issues (RapidIssueEntry)
         * @param {map} data.epics epic data object which *maps* epic key to epic issue.
         * @param {string} data.viewMode 'plan', 'work'
         * @param {string} data.boardType 'scrum', 'kanban'
         */
        printSelectedIssues: function printSelectedIssues(data) {
            CardPrintingView.showPrintScreen({
                issues: data.issues,
                epics: data.epics,
                actionContext: {
                    triggeredFrom: 'contextMenu',
                    viewMode: data.viewMode,
                    boardType: data.boardType
                }
            });
        },

        /**
         * Print the given issues from the plan view of the current board. (e.g. to be called from the board's
         * dropdown menu when being in plan mode). Issues will be printed in the order given in the input data. Sprint's
         * issues will be printed before backlog's issues.
         *
         * @param {Object} data
         * @param {{sprintId: int, sprintName: string, issues: Array}[]} data.sprintIssues an array of sprint data with
         *      the described structure. The "issues" attribute contains an array of visible issues (RapidIssueEntry)
         *      of the corresponding sprint.
         * @param {Array} data.backlogIssues array of visible issues (RapidIssueEntry) from the backlog
         * @param {Object} data.epics epic data object which *maps* epic key to epic issue.
         * @param {string} data.viewMode 'plan', 'work'
         * @param {string} data.boardType 'scrum', 'kanban'
         */
        printBoardPlanView: function printBoardPlanView(data) {
            CardPrintingView.showPrintScreen({
                issues: preparePlanIssuesToPrint(data),
                filterItems: buildPlanFilterElements(data),
                selectedItem: ALL_FILTER_ID,
                epics: data.epics,
                cardColorStrategy: data.cardColorStrategy,

                actionContext: {
                    triggeredFrom: 'boardMenu',
                    viewMode: data.viewMode,
                    boardType: data.boardType
                }
            });
        },

        /**
         * Print the given issues from the work view of the current board. (e.g. to be called from the board's
         * dropdown menu when being in work mode).
         *
         * @param {Object} data
         * @param {Array} data.issues array of visible issues (RapidIssueEntry) to print
         * @param {string} data.viewMode 'plan', 'work'
         * @param {string} data.boardType 'scrum', 'kanban'
         */
        printBoardWorkView: function printBoardWorkView(data) {
            CardPrintingView.showPrintScreen({
                issues: data.issues,
                cardColorStrategy: data.cardColorStrategy,

                actionContext: {
                    triggeredFrom: 'boardMenu',
                    viewMode: data.viewMode,
                    boardType: data.boardType
                }
            });
        }
    };

    function preparePlanIssuesToPrint(data) {
        var issuesToPrint = [];

        // sprint
        _.each(data.sprintIssues, function (sprintData) {
            issuesToPrint.push.apply(issuesToPrint, addIssueCategory('sprint-' + sprintData.sprintId, sprintData.issues));
        });

        // backlog
        issuesToPrint.push.apply(issuesToPrint, addIssueCategory('backlog', data.backlogIssues));

        return issuesToPrint;
    }

    function addIssueCategory(category, issues) {
        return _.map(issues, function (originalIssue) {
            return _.extend({
                cardPrintingCategory: category
            }, originalIssue);
        });
    }

    function buildPlanFilterElements(data) {
        var filterItems = [];

        // we don't want to show sprints that we know there is no issue in them
        var nonemptySprintData = _.filter(data.sprintIssues, function (sprintData) {
            return _.size(sprintData.issues) > 0;
        });

        // build data objects that are compatible with JIRA.Projects.Subnavigator.
        var sprintFilters = _.map(nonemptySprintData, function (sprintData) {
            // yay! id.toString(): JIRA.Projects.Subnavigator requires this, other wise the ID will not be
            // correctly passed to the itemSelected event
            return { id: sprintData.sprintId.toString(), label: sprintData.sprintName };
        });

        // filter is only meaningful if there are more than one sprint or one sprint and a nonempty backlog
        if (_.size(sprintFilters) > 1 || _.size(sprintFilters) === 1 && !_.isEmpty(data.backlogIssues)) {
            filterItems.push([{ id: BACKLOG_FILTER_ID, label: AJS.I18n.getText('gh.rapid.plan.backlog.name') }, { id: ALL_FILTER_ID, label: AJS.I18n.getText('gh.issue.all.title') }]);
            filterItems.push(sprintFilters);
        }
        return filterItems;
    }

    return CardPrintingController;
});