/**
 * globals
 *
 * GH.RapidBoard.State
 */

/**
 * This module is repsible for sending analytic events concerning card printing.
 *
 * @module jira-agile/rapid/ui/print/card-printing-analytic
 * @see module:jquery
 */
define('jira-agile/rapid/ui/print/card-printing-analytic', ['jquery'], function ($) {
    'use strict';

    var RapidBoardState = GH.RapidBoard.State;

    /**
     * @param {Object} context
     * @param {Object} context.actionContext
     * @param {Object} context.AJS
     *
     * @constructor
     */
    function CardPrintingAnalytic(context) {
        this.context = context;
    }

    /**
     * Hooks up to browser's print event in order to send analytic data.
     */
    CardPrintingAnalytic.prototype.registerForPrintEvent = function () {
        // this is for Chrome and safari
        if (window.matchMedia) {
            var mediaQueryList = window.matchMedia('print');
            // on Chrome, when printing with browser native print command, this will fire before the print dialog shown
            mediaQueryList.addListener($.proxy(function (queryList) {
                if (!queryList.matches) {
                    triggerPrintAnalyticEvent(this.context);
                }
            }, this));
        }
        // this if for Firefox/IE
        // For IE, this event will be fired BEFORE the print dialog/preview dialog is shown (whatever printing from
        // browser native window or from window.print())!
        window.onafterprint = $.proxy(function () {
            triggerPrintAnalyticEvent(this.context);
        }, this);
    };

    /**
     * Put the selected filter-ID on a DOM attribute in order to get it back for building analytic data.
     * @param {string} filterId
     */
    CardPrintingAnalytic.takeNoteSelectedFilterId = function (filterId) {
        // "static fuction"
        var filterType;
        if (isNaN(filterId)) {
            filterType = filterId;
        } else if (parseInt(filterId) === -1) {
            // Kanban selected for dev column now has sprint Id == -1
            filterType = 'selected4dev';
        } else {
            filterType = 'sprint';
        }

        $('#subnav-trigger-ghx-print-filter-cbx').data('selected-filter-type', filterType);
    };

    /**
     * Builds analytic data and call the parent window to send it.
     *
     */
    function triggerPrintAnalyticEvent(printContext) {
        var actionContext = printContext.actionContext;
        printContext.AJS.trigger('analytics', {
            name: buildAnalyticEventKey({
                triggeredFrom: actionContext.triggeredFrom,
                boardType: actionContext.boardType,
                viewMode: actionContext.viewMode,
                cardSize: RapidBoardState.getLastCardPrintingSize()
            }),

            data: {
                issueCount: $('.ghx-card').length - $('.ghx-card-filtered-out').length
            }
        });
    }

    /**
     *
     * @param {object} data
     * @param {string} data.triggeredFrom mandatory, limited values (enumeratable): 'contextMenu', 'boardMenu'. This tells
     *          us where the user right clicks and prints (context menu) or prints the current view (board menu)
     * @param {string} data.boardType mandatory, limited values (enumeratable): 'scrum', 'kanban'
     * @param {string} data.viewMode mandatory, limited values (enumeratable): 'plan', 'work'
     * @param {string} data.cardSize mandatory, limited values (enumeratable): 'small', 'medium', 'large'
     *
     * @returns {string} example: 'gh.cardprinting.contextMenu.scrum.plan.small.all'
     */
    function buildAnalyticEventKey(data) {
        // some use cases:
        //
        // to know how many times users print from context menu (similar to board menu):
        //
        //  select sum(count)
        //  from ondemand_event_summary_current
        //  where event like 'gh.cardprinting.contextmenu.%'
        //
        // to know how many times users print from scrum:
        //  select sum(total)
        //  from
        //      (select sum(count) as total
        //      from ondemand_event_summary_current
        //      where event like 'gh.cardprinting.contextmenu.scrum%'
        //      union
        //      select sum(count) as total
        //      from ondemand_event_summary_current
        //      where event like 'gh.cardprinting.boardmenu.scrum%') as tmp

        // using 'like' operator in this case if OK because we can utilize index on the 'event' column for the first part
        // then the rest, full table scan will be perfect as number of rows (after matching the index) is small.

        return ['gh.cardprinting.', data.triggeredFrom, '.', data.boardType, '.', data.viewMode, '.', data.cardSize, '.', getCurrentFilterType()].join('');
    }

    /**
     * @returns {string} 'all', 'sprint', 'backlog', 'selected4dev' or 'notApplicable'
     */
    function getCurrentFilterType() {
        var type = $('#subnav-trigger-ghx-print-filter-cbx').data('selected-filter-type');
        return type ? type : 'notApplicable';
    }

    return CardPrintingAnalytic;
});