define('jira-agile/rapid/ui/plan/contextmenu/split-issue-dialog-view', ['require'], function (require) {
    "use strict";

    var Backbone = require('backbone');
    var SingleSelect = require('jira/ajs/select/single-select');

    AJS.formValidation.register(['nonempty'], function (field) {
        var fieldValue = field.el.value;
        if (!fieldValue || !fieldValue.trim()) {
            field.invalidate(AJS.I18n.getText('aui.validation.message.required'));
        } else {
            field.validate();
        }
    });

    return Backbone.View.extend({
        el: 'body',

        initialize: function initialize(viewModel) {
            this.model = viewModel;
            this.issue = viewModel.issue;
        },

        events: {
            'click #split-issue-dialog-close-button': 'hide',
            'click #split-issue-dialog-submit-button': 'onSubmit',
            'click .ghx-split-issues-add-row': 'addRow',
            'click .ghx-split-issues-delete-row': 'deleteRow',
            'change .ghx-split-original-issue .ghx-split-estimate': 'evaluateEstimateChange',
            'change .destination-single-select': 'evaluateEstimateChange',
            'keyup .ghx-split-summary': 'onKeyPress',
            'keyup .ghx-split-estimate': 'onKeyPress'
        },

        render: function render() {
            var _this = this;

            this.$el.append(GH.tpl.plan.contextmenu.splitissue.dialogView({
                model: this.model
            }));

            AJS.dialog2('.ghx-split-issue-dialog').show();
            AJS.dialog2('.ghx-split-issue-dialog').on('hide', this.undelegateEvents.bind(this));

            this.$('.ghx-split-issue-dialog .select').each(function () {
                new SingleSelect({
                    element: this,
                    itemAttrDisplayed: "label"
                });
            });

            this.$("#split-issue-form").on('submit', function (e) {
                e.preventDefault();
                _this.onSubmit();
                return false;
            });
        },

        hide: function hide() {
            AJS.dialog2('.ghx-split-issue-dialog').hide();
        },

        processValidationErrors: function processValidationErrors(errors) {
            var fields = this.getFields();

            for (var index in errors) {
                if ({}.hasOwnProperty.call(errors, index)) {
                    var error = errors[index];

                    if (error.errorMessageForSummary) {
                        fields[error.index].summary[0].dataAuiNotificationError = error.errorMessageForSummary;
                    }

                    if (error.errorMessageForEstimate) {
                        fields[error.index].estimate[0].dataAuiNotificationError = error.errorMessageForEstimate;
                    }
                }
            }

            this.setWaitModeEnabled(false);
        },

        onKeyPress: function onKeyPress(event) {
            if (event.keyCode == 13) {
                this.onSubmit();
            }
        },

        isInputValid: function isInputValid() {
            var isFieldEmpty = function isFieldEmpty(fieldGetter, model) {
                var isEmpty = function isEmpty(s) {
                    return !s || !s.trim();
                };

                if (isEmpty(fieldGetter(model.issue))) {
                    return true;
                }

                var submitModel = model.getSubmitModel();
                for (var issueIndex in submitModel.newIssues) {
                    if (submitModel.newIssues.hasOwnProperty(issueIndex)) {
                        var newIssue = submitModel.newIssues[issueIndex];
                        if (isEmpty(fieldGetter(newIssue))) {
                            return true;
                        }
                    }
                }
                return false;
            };

            return !(isFieldEmpty(function (issue) {
                return issue.summary;
            }, this) || this.model.estimateStatisticRequired && isFieldEmpty(function (issue) {
                return issue.estimate;
            }, this));
        },

        setWaitModeEnabled: function setWaitModeEnabled(enabled) {
            if (enabled) {
                this.$('.spinner').spin();
                this.$('#split-issue-dialog-submit-button').prop('disabled', 'disabled');
            } else {
                this.$('.spinner').spinStop();
                this.$('#split-issue-dialog-submit-button').prop('disabled', false);
            }
        },

        onSubmit: function onSubmit() {
            if (this.isInputValid()) {
                this.setWaitModeEnabled(true);
                this.trigger('submit', this.getSubmitModel());
            } else {
                this.getFields().forEach(function (field) {
                    field.summary.trigger('validate_input');

                    if (field.estimate) {
                        field.estimate.trigger('validate_input');
                    }
                });
            }
        },

        addRow: function addRow() {
            var tableRows = this.$('#ghx-split-issues-rows');
            tableRows.append(GH.tpl.plan.contextmenu.splitissue.issueRow({
                model: this.model,
                isNewIssueRow: true
            }));

            new SingleSelect({
                element: tableRows.find('.ghx-split-new-issue .select').last(),
                itemAttrDisplayed: "label"
            });
            this.$('.ghx-split-summary').last().focus();
        },

        deleteRow: function deleteRow(e) {
            this.$(e.target).closest('.ghx-split-new-issue').remove();
        },

        checkIfEstimateChanged: function checkIfEstimateChanged($issueRow, $destSelect) {
            var $estimateInput = $issueRow.find('.ghx-split-estimate');
            var newEstimate = $estimateInput.val().trim();
            var isNewIssueRow = $issueRow.hasClass('ghx-split-new-issue');
            var selectedDestId = parseInt($destSelect.val()[0]);

            if (isNewIssueRow) {
                return this.model.issueDestinations.some(function (dest) {
                    return dest.id === selectedDestId && dest.state === 'ACTIVE';
                });
            } else {
                return newEstimate !== this.issue.estimate || selectedDestId !== this.issue.containingSprint.id;
            }
        },

        toggleWarning: function toggleWarning($issueRow, warningEnabled) {
            var warningHiddenClass = 'destination-warning-hidden';
            var $warningElement = $issueRow.find('.destination-warning span');
            warningEnabled ? $warningElement.removeClass(warningHiddenClass) : $warningElement.addClass(warningHiddenClass);
        },

        evaluateEstimateChange: function evaluateEstimateChange(event) {
            var $issueRow = this.$(event.target).closest('tr');
            var $destSelect = $issueRow.find('.destination-single-select');

            //It only makes sense to check for warnings if the newly split issue might be put in an active sprint,
            //If there's no destination selector then there is nowhere to put the issue but in its current location
            // and the only case that is true is when there is only the backlog present
            if ($destSelect.length) {
                this.toggleWarning($issueRow, this.checkIfEstimateChanged($issueRow, $destSelect));
            }
        },

        getSubmitModel: function getSubmitModel() {
            var allFields = this.getFields();
            var newIssues = this.extractNewIssues(this.issue.estimateFieldId);

            return {
                originalIssue: {
                    summary: allFields[0].summary.val(),
                    estimate: allFields[0].estimate.val(),
                    estimateFieldId: this.issue.estimateFieldId,
                    destinationId: allFields[0].destination.val() ? allFields[0].destination.val()[0] : undefined
                },
                newIssues: newIssues
            };
        },

        extractNewIssues: function extractNewIssues(estimateFieldId) {
            return this.getFields().slice(1).map(function (field) {
                return {
                    summary: field.summary.val(),
                    estimate: field.estimate.val(),
                    estimateFieldId: estimateFieldId,
                    destinationId: field.destination.val() ? field.destination.val()[0] : undefined
                };
            });
        },

        getFields: function getFields() {
            var extractFields = function extractFields(container) {
                return {
                    summary: container.find(".ghx-split-summary"),
                    estimate: container.find(".ghx-split-estimate"),
                    destination: container.find(".select")
                };
            };

            // original issue
            var result = [extractFields(this.$('.ghx-split-original-issue'))];

            // new issues
            var newIssuesElements = this.$('.ghx-split-new-issue');
            for (var i = 0; i < newIssuesElements.length; i++) {
                result.push(extractFields(this.$(newIssuesElements[i])));
            }

            return result;
        }
    });
});