define('jira-agile/rapid/ui/plan/contextmenu/split-issue-action', ['require'], function(require) {
    'use strict';

    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var DialogView = require('jira-agile/rapid/ui/plan/contextmenu/split-issue-dialog-view');
    var DialogController = require('jira-agile/rapid/ui/plan/contextmenu/split-issue-dialog-controller');

    var BacklogController;
    var BacklogModel;
    GlobalEvents.on('pre-initialization', function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
        BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    });

    function shouldAppendAction(ctx) {
        return GH.Features.SPLIT_ISSUE.isEnabled() && ctx.selectedIssues.length === 1 && !ctx.hasSubtasks && GH.RapidBoard.State.isPlanMode();
    }

    //if there's at least one sprint, that means there's two places we might want to put the new issue,
    // so show the issue selector
    function shouldDisplayIssueDestinationSelector(issue) {
        return BacklogModel.sprintModels.length > 0;
    }

    function getDialogData(issue, displaysIssueDestinationSelector) {
        var dialogDataDeferred = $.Deferred();

        GH.Ajax.get({
            url: '/issue/' + issue.id + '/split',
            data: {
                rapidViewId: GH.RapidBoard.State.getRapidViewId(),
                getIssueDestinations: displaysIssueDestinationSelector
            }
        }).done(function (dialogData) {
            if (dialogData.userHasPermissionToSplit) {
                dialogDataDeferred.resolve(dialogData);
            } else {
                GH.Notification.showError(AJS.I18n.getText('gh.rapid.operations.split.issue.error.no.permission.title'), AJS.I18n.getText('gh.rapid.operations.split.issue.error.no.permission.body'), true);
                dialogDataDeferred.reject();
            }
        });

        return dialogDataDeferred;
    }

    function constructViewModel(issue) {
        var estimateFieldId;
        var estimate = '';
        if (issue.estimateStatistic && issue.estimateStatistic.statFieldId !== 'issueCount') {
            estimateFieldId = issue.estimateStatistic.statFieldId;
            estimate = issue.estimateStatistic.statFieldValue.text || '';
        }

        return {
            issue: {
                key: issue.key,
                id: issue.id,
                summary: issue.summary,
                estimateFieldId: estimateFieldId,
                estimate: estimate,
                estimateStatisticRequired: issue.estimateStatisticRequired,
                typeUrl: issue.typeUrl
            }
        };
    }

    function enhanceSprintNamesWithState(dialogData) {
        var issueDestinations = [dialogData.containingSprint].concat(dialogData.issueDestinations);
        issueDestinations.forEach(function (dest) {
            if (dest.state) {
                var stateInfo = dest.state === 'ACTIVE' ? AJS.I18n.getText('gh.sprint.customfield.dropdown.active') : AJS.I18n.getText('gh.sprint.customfield.dropdown.future');
                dest.name += ' (' + stateInfo + ')';
            }
        });
    }

    function getViewModel(issue) {
        var displaysIssueDestinationSelector = shouldDisplayIssueDestinationSelector(issue);
        var dialogDataDeferred = getDialogData(issue, displaysIssueDestinationSelector);

        var viewModel = constructViewModel(issue);
        viewModel.displaysIssueDestinationSelector = displaysIssueDestinationSelector;

        var viewModelDeferred = $.Deferred();
        dialogDataDeferred.done(function (dialogData) {
            if (displaysIssueDestinationSelector) {
                enhanceSprintNamesWithState(dialogData);
            }
            viewModel.issueDestinations = dialogData.issueDestinations; //viewModel is correctly assigned to viewmodel
            viewModel.defaultDestinationForNewIssue = dialogData.issueDestinations.find(function (dest) {
                return dest.id === -1;
            });
            viewModel.issue.containingSprint = dialogData.containingSprint;
            viewModelDeferred.resolve(viewModel);
        });

        dialogDataDeferred.fail(function () {
            console.log("failed to obtain view model");

            return viewModelDeferred.reject();
        });
        return viewModelDeferred;
    }

    function renderSplitIssueDialog(issue) {
        var viewModelDeferred = getViewModel(issue);
        viewModelDeferred.done(function (viewModel) {
            var view = new DialogView(viewModel);
            var controller = new DialogController(issue, BacklogController);

            controller.bind(view);
            view.render();
            AJS.trigger('analytics', { name: 'gh.issue.ctx.menu.action.split' });
        });
    }

    function splitActionMenuItem(ctx) {
        return {
            id: 'ghx-issue-ctx-action-split-issue',
            name: 'split-issue',
            label: AJS.I18n.getText('gh.rapid.operations.split.issue'),
            actionFn: function actionFn() {
                var selectedIssue = ctx.selectedIssuesData[0];
                renderSplitIssueDialog(selectedIssue);
            }
        };
    }

    var SplitIssueAction = {};
    SplitIssueAction.appendMenuItem = function (ctx, actionGroups) {
        if (shouldAppendAction(ctx)) {
            actionGroups.push({
                actions: [splitActionMenuItem(ctx)]
            });
        }
    };

    return SplitIssueAction;
});
