AJS.test.require(['com.pyxis.greenhopper.jira:gh-rapid'], function () {
    var $ = require('jquery');
    var _ = require('underscore');

    var CSS_EXPANDED_CLASS = "ghx-subtasks-expanded";
    var COLLAPSED_PARENTS_KEY = 'collapsed-parents';

    module('SubtasksExpandingController', {
        setup: function setup() {
            this.sandbox = sinon.sandbox.create();

            this.$parentContainer = $('<div></div>');
            this.$issueElement = $('<div></div>');
            this.mockedIssueKey = "ISSUE-KEY";
            this.assertionParentHidden = {};
            this.assertionParentHidden[this.mockedIssueKey] = true;

            this.context = AJS.test.mockableModuleContext();

            this.mockPlanIssueListView();
            this.mockBacklogModel();
            this.mockState();

            this.subtasksExpandingController = this.context.require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
            this.subtasksExpandingController._preInitialization();
        },

        teardown: function teardown() {
            this.sandbox.restore();
        },

        /**
         * Mocking
         */
        mockState: function mockState() {
            this.State = {
                getBoardSetting: {}
            };
            this.sandbox.stub(GH.RapidBoard.State, 'setBoardSetting');
            this.sandbox.stub(GH.RapidBoard.State, 'getBoardSetting').returns(this.State.getBoardSetting);

            this.State.setBoardSetting = GH.RapidBoard.State.setBoardSetting;
        },

        mockBacklogModel: function mockBacklogModel() {
            this.BacklogModel = {
                getIssueData: function getIssueData(issueKey) {
                    return {
                        key: issueKey
                    };
                },
                findModelWithIssue: sinon.stub()
            };
            this.context.mock('jira-agile/rapid/ui/plan/backlog-model', this.BacklogModel);
        },

        mockPlanIssueListView: function mockPlanIssueListView() {
            this.PlanIssueListView = {
                getIssueByKey: sinon.stub(),
                getClosestParent: sinon.stub(),
                getIssueKey: sinon.stub()
            };

            this.PlanIssueListView.getClosestParent.withArgs(this.$issueElement).returns(this.$parentContainer);
            this.PlanIssueListView.getIssueKey.withArgs(this.$parentContainer).returns(this.mockedIssueKey);
            this.PlanIssueListView.getIssueByKey.withArgs(this.mockedIssueKey).returns(this.$issueElement);

            this.context.mock('jira-agile/rapid/ui/plan/plan-issue-list-view', this.PlanIssueListView);
        },

        /**
         * Assertions
         */

        assertParentExpanded: function assertParentExpanded(result) {
            ok(result, "Parent should expand");
            ok(this.$parentContainer.hasClass(CSS_EXPANDED_CLASS), "Expanded parent should have a class");
        },

        assertParentCollapsed: function assertParentCollapsed(result) {
            ok(!result, "Parent should collapse");
            ok(!this.$parentContainer.hasClass(CSS_EXPANDED_CLASS, "Collapsed parent shouldn't have a class"));
        },

        assertHiddenParents: function assertHiddenParents(hiddenParents) {
            ok(this.State.setBoardSetting.calledOnce, 'Storing board settings should be called only once');
            deepEqual(this.subtasksExpandingController._collapsedParents, hiddenParents);
            ok(this.State.setBoardSetting.calledWith(COLLAPSED_PARENTS_KEY, hiddenParents));
        }
    });

    test('#toggleExpandState should make parent expanded if child was collapsed', function () {
        var result = this.subtasksExpandingController.toggleExpandState(this.$issueElement);

        this.assertParentExpanded(result);
        this.assertHiddenParents({});
    });

    test('#toggleExpandState should make parent collapsed if child was expanded', function () {
        this.$parentContainer.addClass(CSS_EXPANDED_CLASS);

        var result = this.subtasksExpandingController.toggleExpandState(this.$issueElement);

        this.assertParentCollapsed(result);
        this.assertHiddenParents(this.assertionParentHidden);
    });

    test("#toggleExpandState should set state to expanded if forced", function () {
        this.$parentContainer.addClass(CSS_EXPANDED_CLASS);

        var result = this.subtasksExpandingController.toggleExpandState(this.$issueElement, true);

        this.assertParentExpanded(result);
        this.assertHiddenParents({});
    });

    test("#toggleExpandState should set state to collapsed if forced", function () {
        var result = this.subtasksExpandingController.toggleExpandState(this.$issueElement, false);

        this.assertParentCollapsed(result);
        this.assertHiddenParents(this.assertionParentHidden);
    });

    test('#toggleExpandStateForKey should make parent expanded if child was collapsed', function () {
        var result = this.subtasksExpandingController.toggleExpandStateForKey(this.mockedIssueKey);

        this.assertParentExpanded(result);
        this.assertHiddenParents({});
    });

    test('#toggleExpandStateForKey should make parent collapsed if child was expanded', function () {
        this.$parentContainer.addClass(CSS_EXPANDED_CLASS);

        var result = this.subtasksExpandingController.toggleExpandStateForKey(this.mockedIssueKey);

        this.assertParentCollapsed(result);
        this.assertHiddenParents(this.assertionParentHidden);
    });

    test("#toggleExpandStateForKey should set state to expanded if forced", function () {
        this.$parentContainer.addClass(CSS_EXPANDED_CLASS);

        var result = this.subtasksExpandingController.toggleExpandStateForKey(this.mockedIssueKey, true);

        this.assertParentExpanded(result);
        this.assertHiddenParents({});
    });

    test("#toggleExpandStateForKey should set state to collapsed if forced", function () {
        var result = this.subtasksExpandingController.toggleExpandStateForKey(this.mockedIssueKey, false);

        this.assertParentCollapsed(result);
        this.assertHiddenParents(this.assertionParentHidden);
    });

    test('#getCollapsedSubTasksToSelection should include only not hidden subtasks in the same model', function () {
        var mockIssueListModel = function (parentKey, subtaskKeysList) {
            var model = {
                issueList: {
                    getSubtasks: sinon.stub()
                }
            };

            var subtasksMocked = subtaskKeysList.map(function (key) {
                return {
                    key: key
                };
            });

            this.BacklogModel.findModelWithIssue.withArgs(parentKey).returns(model);
            model.issueList.getSubtasks.withArgs(parentKey).returns(subtasksMocked);
        }.bind(this);

        mockIssueListModel('not-collapsed-issue', ['not-collapsed-issue-subtask']);
        mockIssueListModel('collapsed-issue-without-subtasks', []);
        mockIssueListModel('first-collapsed-issue-with-subtasks', ['first-collapsed-issue-with-subtasks-subtask']);
        mockIssueListModel('second-collapsed-issue-with-subtasks', ['second-collapsed-issue-with-subtasks-subtask', 'second-collapsed-issue-with-subtasks-subtask-2']);

        this.subtasksExpandingController._collapsedParents['collapsed-issue-without-subtasks'] = true;
        this.subtasksExpandingController._collapsedParents['first-collapsed-issue-with-subtasks'] = true;
        this.subtasksExpandingController._collapsedParents['second-collapsed-issue-with-subtasks'] = true;

        var result = this.subtasksExpandingController.getCollapsedSubTasksToSelection(['not-collapsed-issue', 'collapsed-issue-without-subtasks', 'first-collapsed-issue-with-subtasks', 'second-collapsed-issue-with-subtasks']);

        deepEqual(result, ['first-collapsed-issue-with-subtasks-subtask', 'second-collapsed-issue-with-subtasks-subtask', 'second-collapsed-issue-with-subtasks-subtask-2']);
    });
});