/* globals
 * GH.Components, GH.Util, GH.Dialogs, GH.Ajax, GH.BoardState, GH.ConfirmDialog
 */

/**
 * This module representing SprintController
 * @module jira-agile/rapid/ui/plan/sprint-controller
 * @requires module:jquery
 * @requires module:jira-agile/rapid/global-events
 * @requires module:jira-agile/rapid/analytics-tracker
 * @requires module:jira/analytics
 * @requires module:jira-agile/rapid/ui/plan/sprint-view
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/BacklogView
 */
define('jira-agile/rapid/ui/plan/sprint-controller', ['require'], function (require) {
    'use strict';

    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var AnalyticsTracker = require('jira-agile/rapid/analytics-tracker');
    var JiraAnalytics = require('jira/analytics');
    var SprintView = require('jira-agile/rapid/ui/plan/sprint-view');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogController;
    var BacklogView;

    // Resolve circular dependency
    GlobalEvents.on("pre-initialization", function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
        BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
    });

    var SprintController = {};

    SprintController.analytics = new AnalyticsTracker('gh.rapidboard.sprint.controller');

    /**
     * Inline edit component for active sprint info
     */
    SprintController.activeSprintInlineEdits = undefined;

    SprintController.init = function () {
        SprintView.init();

        // edit sprint
        $(GH).bind(GH.Components.EVENT_SPRINT_UPDATED, SprintController.handleSprintUpdated);

        // init inline edits
        SprintController.initInlineEdits();
    };

    /**
     * Registers sprint name, start and end date inline edits
     */
    SprintController.initInlineEdits = function () {

        /*
         * Function used by ActiveSprintInlineEdit
         * @param sprintId
         * @return the sprint data corresponding to the given sprint id
         */
        var getSprintDataById = function (sprintId) {
            var sprintModel = BacklogModel.getSprintModel(sprintId);
            if (sprintModel) {
                return sprintModel.getSprintData();
            } else {
                return null;
            }
        };

        // create and initialize the sprint edit control
        SprintController.activeSprintInlineEdits = new GH.Components.ActiveSprintInlineEdit(getSprintDataById);
        SprintController.activeSprintInlineEdits.initializeInlineEdits();
    };

    /**
     * This function is called after the Edit Sprint dialog has been successfully submitted.
     * @param sprintData the response from the server containing the updated sprint
     */
    SprintController.handleSprintUpdated = function (event, sprintData) {
        var sprintModel = BacklogModel.updateDataForSprint(sprintData.id, sprintData);
        if (sprintModel) {
            BacklogView.redrawChangedModel(sprintModel);
        }
    };


    SprintController.addSprint = function () {
        var addSprintButton = $(this);
        if (GH.Util.UI.isDisabled(addSprintButton)) {
            return;
        }

        GH.Util.UI.ariaDisable(addSprintButton);
        GH.Ajax.post({
            url: '/sprint/' + BacklogController.rapidViewData.id
        }).done(function (result) {
            // fire analytics event
            SprintController.analytics.trigger('addsprint');

            var sprint = result.success;
            // add sprint to list of sprints
            BacklogModel.addNewSprint(sprint);

            // redraw sprints
            BacklogView.updateSprintsAfterAddRemove();
        }).always(function () {
            GH.Util.UI.ariaEnable(addSprintButton);
        });
    };

    /**
     * Opens the dialog to start a future sprint
     * @param sprintId the id of the sprint to start
     */
    SprintController.openStartSprintDialog = function (sprintId) {
        // ensure that we have some issues in sprint
        var rapidViewId = GH.RapidBoard.State.getRapidViewId();
        var sprintModel = BacklogModel.getSprintModel(sprintId);
        var issueList = sprintModel.getIssueList();
        if (issueList.getAllIssues().length < 1) {
            return;
        }

        var numVisibleIssuesInSprint = issueList.getNumVisibleIssues();
        GH.Dialogs.StartSprintDialog.showDialog(rapidViewId, sprintId, numVisibleIssuesInSprint);
    };

    SprintController.deleteSprint = function (sprintId) {
        var rapidViewId = GH.RapidBoard.State.getRapidViewId();
        var sprintModel = BacklogModel.getSprintModel(sprintId);

        GH.ConfirmDialog.create('ghx-remove-planned-sprint-confirm-dialog', {
            contentEscaping: false,
            content: AJS.I18n.getText('gh.sprint.delete.confirm.message', '<b>' + AJS.escapeHTML(String(sprintModel.getName())) + '</b>'),
            header: AJS.I18n.getText('gh.sprint.delete.confirm.title'),
            onConfirmFn: function () {
                SprintController.doDeleteSprint(rapidViewId, sprintId);
            }
        }).show();
    };

    /**
     * Deletes a sprint, adding all issues to the next sprint/backlog
     */
    SprintController.doDeleteSprint = function (rapidViewId, sprintId) {
        // update model
        var backlogAffected = BacklogModel.removeSprintAndMoveIssues(sprintId);

        // update view
        if (backlogAffected) {
            // rerender everything
            BacklogView.draw();
        } else {
            // rerender sprints
            BacklogView.updateSprintsAfterAddRemove();
        }

        // fire analytics event
        SprintController.analytics.trigger('deletesprint');

        GH.Ajax.del({
            url: '/sprint/' + sprintId + '?rapidViewId=' + rapidViewId
        }).done(function () {
            JIRA.trace("gh.sprint.deleted");
        });
    };


    SprintController.addGoal = function (sprintId) {
        var $sprintGoal = $('.js-sprint-container[data-sprint-id="' + sprintId +'"] .ghx-sprint-goal');
        $sprintGoal.removeClass('ghx-hidden');
        $('.js-edit-sprintGoal-trigger', $sprintGoal).click();
    };

    /**
     * Is the twixie open for a given sprint?
     */
    SprintController.isTwixieOpen = function (sprintId) {
        return SprintController._getTwixieState(sprintId) === 'open';
    };

    /**
     * Toggles the twixie open state
     * @param sprintId
     */
    SprintController.toggleTwixie = function (sprintId) {
        var open = SprintController.isTwixieOpen(sprintId);
        SprintController.setTwixieOpen(sprintId, !open);
    };

    /**
     * Sets the twixie open state
     */
    SprintController.setTwixieOpen = function (sprintId, open) {
        // change state
        var value = open ? 'open' : 'closed';
        SprintController._setTwixieState(sprintId, value);

        // update view
        SprintView.updateTwixieState(sprintId);
    };

    /**
     * Set the twixie state for a sprint
     */
    SprintController._setTwixieState = function (sprintId, value) {
        var key = 'sprintTwix-' + sprintId;
        GH.BoardState.setPerViewValue(key, value);
    };

    /**
     * Get the twixie state for a sprint
     */
    SprintController._getTwixieState = function (sprintId) {
        var key = 'sprintTwix-' + sprintId;
        return GH.BoardState.getPerViewValue(key, 'open');
    };

    SprintController.moveSprintUp = function (sprintId) {
        SprintController._moveSprint(sprintId, BacklogModel.moveSprintUp);
    };

    SprintController.moveSprintDown = function (sprintId) {
        SprintController._moveSprint(sprintId, BacklogModel.moveSprintDown);
    };

    SprintController._moveSprint = function (sprintId, moveSprintFn) {
        var otherSprintId = moveSprintFn(sprintId);
        if (otherSprintId != null) {
            BacklogView.updateSprintsAfterAddRemove();

            GH.Ajax.put({
                url: '/sprint/' + sprintId + '/swap',
                data: {
                    'otherSprintId': otherSprintId
                }
            }).done(function () {
                // fire analytics event
                SprintController.analytics.trigger('movesprint');
            }).fail(function () {
                // undo
                moveSprintFn(otherSprintId);
                BacklogView.updateSprintsAfterAddRemove();
            });
        }
    };

    SprintController.triggerAnalytics = function (removeFromSprintModel) {
        if (removeFromSprintModel.changes.length === 1) {
            var changes = removeFromSprintModel.changes[0];
            var currentSprint = changes.currentSprint;
            var targetSprint = changes.targetSprint;
            var event;

            if (currentSprint && targetSprint &&
                currentSprint.state === 'FUTURE' &&
                targetSprint.state === 'FUTURE') {
                event = 'movebetweenfuturesprints';
            }

            if (currentSprint &&
                targetSprint &&
                currentSprint.state === 'ACTIVE' &&
                targetSprint.state === 'ACTIVE') {
                event = 'movebetweenactivesprints';
            }

            if (currentSprint &&
                targetSprint &&
                currentSprint.state === 'FUTURE' &&
                targetSprint.state === 'ACTIVE') {
                event = 'movefromfuturetoactivesprint';
            }

            if (currentSprint &&
                targetSprint &&
                currentSprint.state === 'ACTIVE' &&
                targetSprint.state === 'FUTURE') {
                event = 'movefromactivetofuturesprint';
            }

            if (!currentSprint &&
                targetSprint &&
                targetSprint.state === 'ACTIVE') {
                event = 'addtoactivesprint';
            }

            if (!currentSprint &&
                targetSprint &&
                targetSprint.state === 'FUTURE') {
                event = 'addtofuturesprint';
            }

            if (currentSprint && !targetSprint &&
                currentSprint.state === 'FUTURE') {
                event = 'removefromfuturesprint';
            }

            if (currentSprint && !targetSprint &&
                currentSprint.state === 'ACTIVE') {
                event = 'removefromactivesprint';
            }

            if (event) {
                SprintController.analytics.trigger(event);
            }
        }
    };

    SprintController.triggerAnalyticsIssuesDnD = function (issueMoveModel) {
        // issueMoveModel should have only one change
        // check implementation of IssueMoveController#calculateIssueMoveModel
        issueMoveModel.changes.forEach(function (change) {
            var currentSprintId = !!change.currentSprint ? change.currentSprint.id: -1;
            var targetSprintId = !!change.targetSprint ? change.targetSprint.id : -1;

            JiraAnalytics.send({
                name: 'jira-software.scrum.plan.issuecard.dnd',
                properties: {
                    issues: change.issueKeys.length + change.doneIssueKeys.length,
                    boardId: BacklogController.rapidViewData.id,
                    currentSprintId: currentSprintId,
                    targetSprintId: targetSprintId
                }
            });
        });
    };

    return SprintController;
});
