/* globals
 * GH.Tooltip, GH.EpicController, GH.RapidBoard, GH.tpl
 */

/**
 * This module representing SprintBacklogView
 * @module jira-agile/rapid/ui/plan/sprint-backlog-view
 * @requires module:jquery
 * @requires module:jira-agile/rapid/ui/plan/sprint-controller
 * @requires module:jira-agile/rapid/ui/plan/backlog-model
 * @requires module:jira-agile/rapid/ui/plan/backlog-statistics
 * @requires module:jira-agile/rapid/ui/work/issue-list-util
 * @requires module:jira-agile/rapid/ui/plan/backlog-controller
 * @requires module:jira-agile/rapid/ui/plan/plan-view
 * @requires module:jira-agile/rapid/ui/plan/sprint-backlog-util
 * @requires module:jira/util/logger
 */
define('jira-agile/rapid/ui/plan/sprint-backlog-view', ['require', 'underscore'], function (require, _) {
    'use strict';

    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var SprintController = require('jira-agile/rapid/ui/plan/sprint-controller');
    var PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
    var BacklogModel = require('jira-agile/rapid/ui/plan/backlog-model');
    var BacklogStatistics = require('jira-agile/rapid/ui/plan/backlog-statistics');
    var IssueListUtil = require('jira-agile/rapid/ui/plan/issue-list-util');
    var SubtasksExpandingController = require('jira-agile/rapid/ui/plan/subtasks-expanding-controller');
    var FeatureFlagManager = require('jira/featureflags/feature-manager');
    var PlanView = require('jira-agile/rapid/ui/plan/plan-view');
    var PlanDragAndDrop = require('jira-agile/rapid/ui/plan/plan-drag-and-drop');
    var SprintBacklogUtil = require('jira-agile/rapid/ui/plan/sprint-backlog-util');
    var JiraLogger = require('jira/util/logger');
    var DynamicColors = require('jira-agile/rapid/ui/component/dynamic-colors');
    var BacklogController;

    var SHOWMORE_FEATURE_FLAG = 'com.atlassian.jira.agile.darkfeature.backlog.showmore';
    var SHOWMORE_ALL_ISSUES = -1;
    /* Amount of issues shown after "Show More" section*/
    var BACKLOG_TAIL_LENGTH = 10;
    var SHOW_ALL_CLICKED = false;

    // Resolve circular dependency
    GlobalEvents.on("pre-initialization", function () {
        BacklogController = require('jira-agile/rapid/ui/plan/backlog-controller');
    });

    /**
     * Backlog view component.
     */
    var SprintBacklogView = {};

    SprintBacklogView.containerSelector = '.ghx-backlog-group';

    /**
     * Initializes this view
     */
    SprintBacklogView.init = function () {
        // Add sprint button handler
        $(document).on('click', '.js-add-sprint', SprintController.addSprint);
        // show more config
        $(document).on('click', '#gh-show-more-config-dropdown .aui-dropdown2-radio', function (event) {
            event.preventDefault();
            var limit = $(event.target).data("issues");
            SprintBacklogView.setIssueLimit(limit);
            // Reset this since configuration has been changed.
            SprintBacklogView.setShowAllClicked(false);

            SprintBacklogView.showIssues(limit);
        });

        // Add show all action
        $(document).on('click', '.js-show-all-link', function (event) {
            event.preventDefault();
            SprintBacklogView.showIssues(SHOWMORE_ALL_ISSUES, true);
        });
    };

    /**
     * Draws the backlog
     */
    SprintBacklogView.renderBacklog = function (issueRenderData, limit) {
        var html = _getBacklogHtml(issueRenderData, limit);
        var $container = $('.ghx-backlog-group');
        $container.html(html);
        enableTooltipForCreateSprintButton();
    };

    SprintBacklogView.setShowAllClicked = function (clicked) {
        SHOW_ALL_CLICKED = clicked;
    };

    /**
     * updates backlog issue limit
     */
    SprintBacklogView.setIssueLimit = function (number) {
        PlanController.setBacklogIssuesLimitConfig(number);
        GH.RapidBoard.State.pushState();
    };

    /**
     * Re-renders  backlog with selected number of issues shown
     */
    SprintBacklogView.showIssues = function (limit, showAllClick) {
        PlanView.showLoadingBacklog();

        // this is needed to render the loading indicator
        setTimeout(function () {
            if (showAllClick) {
                AJS.trigger('analyticsEvent', { name: 'jira-software.backlog.click.show.all' });
            }

            if (FeatureFlagManager.isFeatureEnabled(SHOWMORE_FEATURE_FLAG)) {
                SprintBacklogView.updateBacklog(limit);
                PlanDragAndDrop.enableDragAndDrop();
                DynamicColors.update();
            }

            if (showAllClick) {
                SprintBacklogView.setShowAllClicked(true);
            }
            PlanView.hideLoadingBacklog();
        });
    };

    /**
     * Updates the rendered backlog
     * @param limit
     */
    SprintBacklogView.updateBacklog = function (limit) {
        // load the issue render data required for drawing
        var issueRenderData = BacklogController.calculateIssueRenderData();
        SprintBacklogView.renderBacklog(issueRenderData, limit);
        JiraLogger.trace("gh.boards.updateBacklog");
    };

    /**
     * Updates the backlog header & statistics
     */
    SprintBacklogView.updateBacklogHeader = function () {
        var $newHeaderContent = $(_renderBacklogHeader()).html();
        $('.js-marker-backlog-header').html($newHeaderContent);
        enableTooltipForCreateSprintButton();
    };

    /**
     * Calculates the issue list stats for a backlog
     */
    SprintBacklogView.calculateBacklogIssueListStats = function (backlogIssueList) {
        var issueListStats = BacklogStatistics.getIssueListStats(backlogIssueList, false);

        if (GH.RapidBoard.State.isKanbanBoard()) {
            issueListStats = _.extend({}, issueListStats, BacklogStatistics.calculateStatisticsFieldValue(backlogIssueList, BacklogController.rapidViewConfig.statisticConfig));
        }
        return issueListStats;
    };

    /**
     * Get the html for the complete backlog
     */
    var _getBacklogHtml = function _getBacklogHtml(issueRenderData, limit) {
        // put together all required data
        var backlogModel2 = BacklogModel.getBacklogModel2();
        var backlogIssueList = backlogModel2.getIssueList();
        var issueListStats = SprintBacklogView.calculateBacklogIssueListStats(backlogIssueList);

        var visibleRanks = backlogIssueList.getVisibleRankables();
        var issuesMap = backlogIssueList.getAllIssues();
        var issues = IssueListUtil.issueListToTree(visibleRanks.map(function (key) {
            return issuesMap[key];
        }), BacklogModel.getAllIssuesWithMissingParents());

        IssueListUtil.applySubtaskCount(issues, backlogIssueList, issueRenderData.hiddenIssues);

        SubtasksExpandingController.applyExpandedState(issues);

        // render
        var params = {
            issueKeys: visibleRanks,
            issues: createIssueListWithShowMore(issueListStats, issues, visibleRanks, issueRenderData.hiddenIssues, limit),
            issueListStats: issueListStats,
            epics: BacklogController.getEpicMap(), // epics needed for correct labels and colours
            issueRenderData: issueRenderData,
            statLabel: BacklogModel.estimationStatistic.name,
            isRankable: BacklogModel.isRankable(),
            canManageSprints: BacklogModel.canManageSprints(),
            showEpic: GH.RapidBoard.State.getEpicShownOnRapidBoard(),
            shouldDisplayCreateSprintButton: GH.RapidBoard.State.isScrumBoard(),
            shouldDisplayShowMoreConfigButton: FeatureFlagManager.isFeatureEnabled(SHOWMORE_FEATURE_FLAG),
            extraClasses: GH.RapidBoard.State.isKanbanBoard() ? 'ghx-kanban-backlog' : undefined,
            isKanbanBoard: GH.RapidBoard.State.isKanbanBoard(),
            showMoreOptions: _getShowMoreOptions()
        };

        return GH.tpl.sprintbacklogview.renderBacklogIssues(params);
    };

    var shouldShowMore = function shouldShowMore(visibleCount, limit) {
        return limit !== SHOWMORE_ALL_ISSUES && limit < visibleCount;
    };

    /**
     * Prepare an array of issues should be shown with vertical split
     * @param issueListStats
     * @param issues
     * @param visibleRanks
     * @param hiddenIssues
     * @param limit
     * @returns {Array} Array of issues with pseudo element for creating a Show All area
     */
    function createIssueListWithShowMore(issueListStats, issues, visibleRanks, hiddenIssues, limit) {
        var excludeSubtasks = issueListStats.excludeSubtasks,
            statisticsFieldVisible = issueListStats.statisticsFieldVisible,
            visibleCount = issueListStats.visibleCount;

        var visibleIssues = excludeSubtasks ? statisticsFieldVisible : visibleCount;

        limit = limit || PlanController.getBacklogIssuesLimitConfig();

        if (!FeatureFlagManager.isFeatureEnabled(SHOWMORE_FEATURE_FLAG)) {
            return issues;
        }

        var onlyVisibleIssues = getVisibleIssues(issues, hiddenIssues);

        if (SHOW_ALL_CLICKED) {
            return onlyVisibleIssues;
        }

        if (!shouldShowMore(visibleIssues, limit)) {
            return onlyVisibleIssues;
        }

        // For KanPlan work we need to show a total amount of issues und sub-tasks,
        // but we can't expect correct count on going one after another,
        // so here I am going to work with issues array from both sides
        var filteredWithShowMoreStart = SprintBacklogUtil.getIssuesListHead(onlyVisibleIssues, limit - BACKLOG_TAIL_LENGTH);
        var filteredWithShowMoreEnd = SprintBacklogUtil.getIssuesListTail(onlyVisibleIssues, BACKLOG_TAIL_LENGTH);

        return filteredWithShowMoreStart.concat({ showMore: true, issuesNotShown: visibleIssues - limit }).concat(filteredWithShowMoreEnd);
    }

    function getVisibleIssues(issues, hiddenIssues) {
        return issues.map(function (issue) {
            if (issue.subTasks) {
                issue.subTasks = issue.subTasks.filter(function (subTask) {
                    return !hiddenIssues[subTask.key];
                });
            }

            if (hiddenIssues[issue.key] && issue.subTasks && issue.subTasks.length) {
                issue.fakeParent = true;
                return issue;
            } else if (hiddenIssues[issue.key]) {
                return false;
            } else if (issue.fakeParent && (!issue.subTasks || !issue.subTasks.length)) {
                return false;
            } else {
                return issue;
            }
        }).filter(function (issue) {
            return !!issue;
        });
    }

    /**
     * Get the html for the backlog header
     */
    var _renderBacklogHeader = function _renderBacklogHeader() {
        // put together all required data
        var backlogModel2 = BacklogModel.getBacklogModel2();
        var backlogIssueList = backlogModel2.getIssueList();
        var issueListStats = SprintBacklogView.calculateBacklogIssueListStats(backlogIssueList);

        // render
        var params = {
            issueListStats: issueListStats,
            canManageSprints: BacklogModel.canManageSprints(),
            shouldDisplayCreateSprintButton: GH.RapidBoard.State.isScrumBoard(),
            shouldDisplayShowMoreConfigButton: FeatureFlagManager.isFeatureEnabled(SHOWMORE_FEATURE_FLAG),
            showMoreOptions: _getShowMoreOptions()
        };
        return GH.tpl.sprintbacklogview.renderBacklogHeader(params);
    };

    var _getShowMoreOptions = function _getShowMoreOptions() {
        return [{
            value: 100,
            label: AJS.I18n.getText('gh.rapid.plan.config.issues.number', 100)
        }, {
            value: 500,
            label: AJS.I18n.getText('gh.rapid.plan.config.issues.number', 500)
        }, {
            value: SHOWMORE_ALL_ISSUES,
            label: AJS.I18n.getText('gh.rapid.plan.config.issues.all')
        }].map(function (option) {
            return _.extend(option, { checked: option.value === PlanController.getBacklogIssuesLimitConfig() });
        });
    };

    /**
     * Enable tipsy with default behaviour for the create sprint button,
     * since we want to take the advantage of default delayOut
     * interval so that user can click on the link within the tipsy.
     */
    function enableTooltipForCreateSprintButton() {
        $('.js-add-sprint').tooltip({
            html: true,
            title: 'data-tooltip'
        });
    }

    return SprintBacklogView;
});