/* globals
 * GH.tpl, GH.Tooltip, GH.DetailsView, GH.ViewActions, GH.GridDataController
 */

/**
 * @module jira-agile/rapid/ui/plan/plan-view
 * @requires module:underscore
 * @requires module:jquery
 * @requires module:jira-agile/rapid/ui/plan/plan-controls
 * @requires module:jira-agile/rapid/ui/plan/plan-controller
 * @requires module:jira-agile/rapid/ui/kanplan/kan-plan-feature-service
 */
define('jira-agile/rapid/ui/plan/plan-view', ['require'], function (require) {
    'use strict';

    var _ = require('underscore');
    var $ = require('jquery');
    var GlobalEvents = require('jira-agile/rapid/global-events');
    var PlanControls = require('jira-agile/rapid/ui/plan/plan-controls');
    var KanPlanAcknowledge = require('jira-agile/rapid/ui/kanplan/kanplan-acknowledge');
    var KanPlanFeatureService = require('jira-agile/rapid/ui/kanplan/kan-plan-feature-service');
    var BacklogView;
    var PlanController;

    // Resolve circular dependency
    GlobalEvents.on("pre-initialization", function () {
        PlanController = require('jira-agile/rapid/ui/plan/plan-controller');
        BacklogView = require('jira-agile/rapid/ui/plan/BacklogView');
    });

    var PlanView = {};

    /**
     * Initializes the view controller.
     */
    PlanView.init = function () {
        var $document = $(document);
        // version panel toggle
        $document.delegate('.js-view-action-toggle-releases-column, .js-release-toggle', 'click', PlanController.toggleVersionsColumn);
        // epic panel toggle
        $document.delegate('.js-view-action-toggle-epics-column, .js-epic-toggle', 'click', PlanController.toggleEpicsColumn);
        $document.delegate('.js-view-actions-plan .js-view-action-toggle-epics-lozenge', 'click', PlanController.toggleEpicsShowOnBacklog);
    };

    /**
     * Called when the work tab is shown
     */
    PlanView.show = function () {
        PlanView.detailViewPositioned = false;

        // Flag the Mode
        $('body').addClass('ghx-mode-planning');

        // create a skeleton DOM for widgets
        var skeleton = GH.tpl.plancontroller.renderSkeleton({
            // Whether to render the entire container of epics and versions
            shouldShowEpicAndVersionPanels: GH.RapidBoard.State.isScrumBoard() || KanPlanFeatureService.isEpicsAndVersionsEnabled(),
            // Whether to render epics panel inside the epics and versions container
            shouldShowEpicPanel: PlanController.isEpicsPanelEnabled()
        });

        $('#ghx-plan').empty().append(skeleton);

        var versionToggleAffordance = $('.js-release-toggle');
        GH.Tooltip.tipsify({
            selector: versionToggleAffordance,
            gravity: 'w',
            html: true,
            className: 'ghx-tipsy-alt'
        });

        var epicToggleAffordance = $('.js-epic-toggle');
        GH.Tooltip.tipsify({
            selector: epicToggleAffordance,
            gravity: 'w',
            html: true,
            className: 'ghx-tipsy-alt'
        });

        // Configure the detail view
        GH.DetailsView.setContainerSelector('#ghx-detail-view');
        GH.DetailsView.setOptions({
            canClose: true,
            showActionsCog: true,
            showSubtaskTab: true,
            showLogWorkTrigger: false,
            closeHandler: PlanController.closeDetailsView
        });

        if (GH.RapidBoard.State.isKanbanBoard()) {
            KanPlanAcknowledge.show();
        }

        // ensure we reset the container the next time we get it since it's been redrawn
        delete BacklogView.container;
    };

    PlanView.updateDetailViewState = function () {
        var opened = PlanController.isDetailsViewOpened();
        if (opened) {
            $('#ghx-plan-group').addClass('ghx-has-detail');
            $('#ghx-detail-view').css({ 'display': '' });
        } else {
            $('#ghx-plan-group').removeClass('ghx-has-detail');
            $('#ghx-detail-view').css({ 'display': 'none' });
        }
        GH.ViewActions.setDetailViewVisible(opened);
    };

    PlanView.updateVersionColumnState = function () {
        var $column = $('#ghx-version-column');
        var flagpole = $('#ghx-plan-group');
        var versionToggleAffordance = $('.js-release-toggle');
        var versionColumnVisible = PlanController.isVersionsColumnVisible() && PlanController.isVersionsPanelEnabled();
        $column.toggleClass('ghx-hidden', !versionColumnVisible);
        flagpole.toggleClass('ghx-version-expanded', versionColumnVisible);
        versionToggleAffordance.toggleClass('ghx-active', versionColumnVisible);

        if (versionColumnVisible) {
            GH.Tooltip.disable(versionToggleAffordance);

            if ($column.is(':empty') || $column.find('.ghx-classification-scrollview').is(':empty')) {
                GH.VersionView.draw();
                PlanView.showLoadingVersions();
                GH.BacklogController.updateVersionData().always(PlanView.hideLoadingVersions);
            }
        } else {
            versionToggleAffordance.attr('data-tooltip', GH.tpl.plancontroller.renderVersionPanelTipsyExpand());
            GH.Tooltip.enable(versionToggleAffordance);
        }
        GH.ViewActions.setVersionsPanelVisible(versionColumnVisible);
    };

    PlanView.updateEpicsColumnState = function () {
        var $column = $('#ghx-epic-column');
        var flagpole = $('#ghx-plan-group');
        var epicsToggleAffordance = $('.js-epic-toggle');
        var epicsColumnVisible = PlanController.isEpicsColumnVisible() && PlanController.isEpicsPanelEnabled();
        $column.toggleClass('ghx-hidden', !epicsColumnVisible);
        flagpole.toggleClass('ghx-epic-expanded', epicsColumnVisible);
        epicsToggleAffordance.toggleClass('ghx-active', epicsColumnVisible);

        if (epicsColumnVisible) {
            if ($column.is(':empty') || $column.find('.ghx-classification-scrollview').is(':empty')) {
                GH.EpicView.draw();
                PlanView.showLoadingEpics();
                GH.BacklogController.updateEpicData().always(PlanView.hideLoadingEpics);
            }
        } else {
            epicsToggleAffordance.attr('data-tooltip', GH.tpl.plancontroller.renderEpicPanelTipsyExpand());
        }
        GH.ViewActions.setEpicsPanelVisible(epicsColumnVisible);
    };

    PlanView.updateViewActions = function () {
        GH.ViewActions.setDetailViewVisible(PlanController.isDetailsViewOpened());
        GH.ViewActions.setEpicsPanelVisible(PlanController.isEpicsColumnVisible());
        GH.ViewActions.setVersionsPanelVisible(PlanController.isVersionsColumnVisible());
        if (GH.GridDataController.getModel()) {
            GH.ViewActions.setSwimlanes(GH.GridDataController.getModel().hasSwimlanes());
        }
    };

    PlanView.updateHorizontalPositioning = function () {
        GH.DetailsView.updateSize();
    };

    PlanView.doUpdateDetailsView = function () {
        // ensure the detail view is properly laid out (this is only necessary the first time after we show the tab,
        // after that resize handlers will take care of re-layouting)
        if (!PlanView.detailViewPositioned) {
            PlanView.detailViewPositioned = true;
            PlanView.updateFixedElementPositioning();
        }
    };

    /**
     * Called when the work tab is hidden
     */
    PlanView.hide = function () {

        // UnFlag the Mode
        $('body').removeClass('ghx-mode-planning');

        // empty our DOM
        $('#ghx-plan').empty();
    };

    // transitions

    /**
     * CSS Class Removal for Loading Transition on QF
     */
    PlanView.hideLoadingFilter = function () {
        $('body').removeClass('ghx-loading-quickfilter');
    };

    /**
     * CSS Class for Pool Loading Transition
     */
    PlanView.showLoadingBacklog = function () {
        $('body').addClass('ghx-loading-backlog');
        $('.ghx-throbber').spin('large');
    };

    /**
     * CSS Class Removal for Pool Loading Transition
     */
    PlanView.hideLoadingBacklog = function () {
        $('body').removeClass('ghx-loading-backlog');
        $('.ghx-throbber').spinStop();
    };

    /**
     * CSS Class for Pool Loading Transition
     */
    PlanView.showLoadingEpics = function () {
        $('body').addClass('ghx-loading-epics');
        $('#ghx-epic-column .ghx-progress-indicator.ghx-initial').spin('large');
    };

    /**
     * CSS Class Removal for Pool Loading Transition
     */
    PlanView.hideLoadingEpics = function () {
        $('body').removeClass('ghx-loading-epics');
        $('#ghx-epic-column .ghx-progress-indicator.ghx-initial').spinStop();
    };

    /**
     * CSS Class for Versions Rendering state
     */
    PlanView.showLoadingVersions = function () {
        $('body').addClass('ghx-loading-versions');
        $('#ghx-version-column .ghx-progress-indicator.ghx-initial').spin('large');
    };

    /**
     * CSS Class Removal for Versions Rendering State
     */
    PlanView.hideLoadingVersions = function () {
        $('body').removeClass('ghx-loading-versions');
        $('#ghx-version-column .ghx-progress-indicator.ghx-initial').spinStop();
    };

    /**
     * Updates all columns marked as fixed to ensure they fit within the content view area
     */
    PlanView.updateFixedElementPositioning = function () {
        // if not visible don't do anything
        if (!PlanController.isVisible) {
            return;
        }

        // update Quick Filter toggle spacing
        PlanControls.resizeQuickFilters({ mode: 'plan' });

        // update all horizontal positioning
        PlanView.updateHorizontalPositioning();
    };

    return PlanView;
});